/***********************************************************************
Autor: Tijana Sukilovic
e-mail: tijana@matf.bg.ac.rs
Datum: 07.09.2011.
Verzija: radna

Ovo je header fajl koji se koristi u kursu Geometrija na 
Matematickom Fakultetu u Beogradu. 

-Definisane su klase Tacka, Duz, Prava i Krug.
-Funkcija Nacrtaj( Objekat obj, COLOR color) cuva objekat u odgovarajucu 
listu. Boju objekta zadaje korisnik iz skupa boja:
{red, green, blue, yellow, orange, magenta, cyan, violet, white, black}.
-Funkcija Prikazi() iscrtava sliku
************************************************************************/


#include <iostream>
#include<algorithm>
#include <list>
using namespace std;

#include <math.h>
#include "glut.h"


//ukoliko koristite stariju verziju programa od Visual Studio 2008, sledeca dva reda ne treba da budu pod komentarom
#define min( a, b ) a  <  b ? a : b
#define max( a, b ) a  >  b ? a : b

#ifndef PI
#define PI 3.14159265358979323846
#endif

//podesavanje prozora
#define WINDOW_WIDTH    1024
#define WINDOW_HEIGHT   700
#define WINDOW_X        100
#define WINDOW_Y        50
#define WINDOW_TITLE    "Geometrija - prikaz zadatka"

//podesavanje perspektive  
#define FOV_ANGLE       60    //u verzijama Visual Studio 2008 i novijim, podesiti ugao na 30
#define CLIPPLANE_NEAR  1.0
#define CLIPPLANE_FAR   1000.0

float VIEWER_X, VIEWER_Y, VIEWER_Z; 

enum COLOR { red, green, blue, yellow, orange, magenta, cyan, violet, white, black };

//deklaracija klasa
class Tacka;
class Duz;
class Prava;
class Krug;

//deklaracija funkcija za preseke pravih i duzi
int PresekPravih( Prava p, Prava q, Tacka* X );
int PresekDuzi( Duz a, Duz b, Tacka* X );

//deklaracija funkcija za prikaz crteza
void Nacrtaj( Tacka t, COLOR c );
void Nacrtaj( Duz d, COLOR c );
void Nacrtaj( Prava p, COLOR c );
void Nacrtaj( Krug k, COLOR c );
void PodesiBoju( COLOR color );
void PromeniDimenziju( int w, int h );
void SacuvajPoziciju( int button, int state, int x, int y );
void TransformisiScenu( int x, int y );
void Prikazi( int argc, char **argv );
void NacrtajObjekte();

class Tacka {
public:
	float x, y, z;
	COLOR color;

	Tacka() : x( 0.0 ), y( 0.0 ), z( 0.0 ), color( black ) {}
	Tacka( float X, float Y ) : x( X ), y( Y ), z( 0.0 ), color( black ) {}
	Tacka( float X, float Y, float Z ) : x( X ), y( Y ), z( Z ), color( black ) {}
	Tacka( const Tacka& t ) : x( t.x ), y( t.y ), z( t.z ), color( t.color ) {}
	~Tacka() {}

	Tacka& operator=( const Tacka& t ) {
		if( &t != this ) { x = t.x; y = t.y; z = t.z; color = t.color; }
        return *this;
    } 

	float Norm2() { return x*x + y*y + z*z; }

	void Nacrtaj() {
		PodesiBoju( color );
		glPointSize( 4.0 );
		glBegin( GL_POINTS );
		glVertex3f( x, y, z ); 
		glEnd();
	}
};

class Duz {
public:
	Tacka start, end;
	COLOR color;

	Duz() : start(), end(), color( black ) {}
	Duz( Tacka s, Tacka e ) : start( s ), end( e ), color( black ) {}
	Duz( const Duz& d ) : start( d.start ), end( d.end ), color( d.color ) {}
	~Duz() {}

	Duz& operator=( const Duz& d ) {
		if( &d != this ) { start = d.start; end = d.end; color = d.color; }
        return *this;
    } 

	void Nacrtaj() {
		glPointSize( 4.0 );
		PodesiBoju( color );

		glBegin(GL_LINES);
		glVertex3f( start.x, start.y, start.z );
		glVertex3f( end.x, end.y, end.z );
		glEnd();

		glBegin( GL_POINTS );
		glVertex3f( start.x, start.y, start.z ); 
		glVertex3f( end.x, end.y, end.z );
		glEnd();
	}
};

class Prava {
public:
	Tacka start, pravac;
	COLOR color;

	Prava() : start(), pravac(), color( black ) {}
	Prava( Tacka s, Tacka p ) : start( s ), pravac( p ), color( black ) {}
	Prava( const Prava& p ) : start( p.start ), pravac( p.pravac ), color( p.color ) {}
	~Prava() {}

	Prava& operator=( const Prava& p ) {
		if( &p != this ) { start = p.start; pravac = p.pravac; color = p.color; }
        return *this;
    } 

	void Nacrtaj() {
		PodesiBoju( color );

		glBegin( GL_LINES );
		glVertex3f( start.x + 100*pravac.x, start.y + 100*pravac.y, start.z + 100*pravac.z );
		glVertex3f( start.x - 100*pravac.x, start.y - 100*pravac.y, start.z - 100*pravac.z );
		glEnd();
	}
};

class Krug {
public:
	Tacka centar;
	float r;
	COLOR color;

	Krug() : centar(), r( 0.0 ), color( black ) {}
	Krug( Tacka c, float R ) : centar( c ), r( R ), color( black ) {}
	Krug( const Krug& k ) : centar( k.centar ), r( k.r ), color( k.color ) {}
	~Krug() {}

	Krug& operator=( const Krug& k ) {
		if( &k != this ) { centar = k.centar; r = k.r; color = k.color; }
        return *this;
    } 

	void Nacrtaj() {
		float angle;
		PodesiBoju( color );
		glBegin( GL_LINE_LOOP );
		for( int i = 0; i < 100; i++ ) {
			angle = 2*PI*i/100;
			glVertex3f( centar.x + r*cos(angle), centar.y + r*sin(angle), centar.z );
		}
		glEnd();
	}
};

//ispituje medjusobni polozaj pravih p i q - vraca 1 ako se seku, 0 ako su paralelne i 2 ako se poklapaju (ako su prave mimoilazne, vraca 3)
int PresekPravih( Prava p, Prava q, Tacka* X ) {
	float epsilon = 0.00001;
	Tacka PQ( q.start.x - p.start.x, q.start.y - p.start.y, q.start.z - p.start.z );

	//ispituje da li su prave mimoilazne
	float Det = -p.pravac.z*q.pravac.y*PQ.x + p.pravac.y*q.pravac.z*PQ.x + p.pravac.z*q.pravac.x*PQ.y - p.pravac.x*q.pravac.z*PQ.y - p.pravac.y*q.pravac.x*PQ.z + p.pravac.x*q.pravac.y*PQ.z;
    if( Det * Det > epsilon * p.pravac.Norm2() * q.pravac.Norm2() )
		return 3;

	float DPQq = PQ.x * q.pravac.y - PQ.y * q.pravac.x; // D(PQ,q)
	float Dpq = p.pravac.x * q.pravac.y - p.pravac.y * q.pravac.x; // D(p,q)

	if( Dpq * Dpq > epsilon * p.pravac.Norm2() * q.pravac.Norm2() ) {
		//prave se seku
		float t = (DPQq / Dpq);
		X->x = p.start.x + t * p.pravac.x;
		X->y = p.start.y + t * p.pravac.y;
		return 1;
	}
	else {
		if( DPQq * DPQq > epsilon * p.pravac.Norm2() * PQ.Norm2() ) 
			return 0; //prave su paralelne
		else
			return 2; //prave se poklapaju
	}
}

//ispituje medjusobni polozaj duzi a i b - vraca 1 ako se duzi seku, 0 ako nemaju zajednickih tacaka, a 2 ako se poklapaju ili delimicno poklapaju (ako su prave mimoilazne, vraca 3)
int PresekDuzi( Duz a, Duz b, Tacka* X ) {
	float epsilon = 0.00001;
	Tacka PQ( b.start.x - a.start.x, b.start.y - a.start.y, b.start.z - a.start.z );
	Tacka p( a.end.x - a.start.x, a.end.y - a.start.y, a.end.z - a.start.z ); 
	Tacka q( b.end.x - b.start.x, b.end.y - b.start.y, b.end.z - b.start.z );

	//ispituje da li su prave mimoilazne
	float Det = -p.z*q.y*PQ.x + p.y*q.z*PQ.x + p.z*q.x*PQ.y - p.x*q.z*PQ.y - p.y*q.x*PQ.z + p.x*q.y*PQ.z;
    if( Det * Det > epsilon * p.Norm2() * q.Norm2() )
		return 3;

	float DPQp = PQ.x * p.y - PQ.y * p.x; // D(PQ,p)
	float DPQq = PQ.x * q.y - PQ.y * q.x; // D(PQ,q)
	float Dpq = p.x * q.y - p.y * q.x; // D(p,q)

	if( Dpq * Dpq > epsilon * p.Norm2() * q.Norm2() ) {
		//prave se seku
		float s = (DPQp / Dpq);
		float t = (DPQq / Dpq);

		if( s >= 0 && s <= 1 && t >= 0 && t <= 1 ) {
			X->x = a.start.x + t * p.x;
			X->y = a.start.y + t * p.y;
			
			return 1;
		}
		else
			return 0;
	}
	else {
		if( DPQp * DPQp > epsilon * p.Norm2() * PQ.Norm2() ) 
			return 0; //duzi su paralelne
		else { //krajnje tacke duzi su kolinearne
			//ako su tacke na vertikali, zamenimo koordinate da bismo sortirali po x koordinati
			if( a.start.x == a.end.x ) {
				swap( a.start.x, a.start.y );
				swap( a.end.x, a.end.y );
				swap( b.start.x, b.start.y );
				swap( b.end.x, b.end.y );
			}
			//sortiramo po x koordinati
			if( a.start.x > a.end.x ) swap( a.start, a.end );
			if( b.start.x > b.end.x ) swap( b.start, b.end );

			if( ( a.start.x <= b.start.x && a.end.x <= b.start.x ) || a.start.x > b.end.x )
				return 0; //nema preseka
			return 2; //postoji preklapanje
		}
	}
	return 0;
}

void PodesiBoju( COLOR color ) {
	switch( color ) {
	   case red : glColor3f(1.0, 0.0, 0.0); break; 
	   case green : glColor3f(0.0, 1.0, 0.0); break; 
	   case blue : glColor3f(0.0, 0.0, 1.0); break; 
	   case yellow : glColor3f(1.0, 1.0, 0.0); break; 
	   case orange : glColor3f(1.0, 0.5, 0.0); break; 
	   case magenta : glColor3f(1.0, 0.0, 1.0); break; 
	   case cyan : glColor3f(0.0, 1.0, 1.0); break; 
	   case violet : glColor3f(0.5, 0.0, 0.5); break; 
	   case white : glColor3f(1.0, 1.0, 1.0); break;
	   case black : glColor3f(0.0, 0.0, 0.0); break; 
	   default : break;
	}
}

static Tacka UP_LEFT( WINDOW_WIDTH , WINDOW_HEIGHT );
static Tacka DOWN_RIGHT( 0.0, 0.0 );

list<Tacka> tacke;
list<Duz> duzi;
list<Prava> prave;
list<Krug> krugovi;

void Nacrtaj( Tacka t, COLOR c ) {
	t.color = c;
	tacke.push_back( t );

	UP_LEFT.x = min(t.x, UP_LEFT.x); 
	UP_LEFT.y = min(t.y, UP_LEFT.y); 
	DOWN_RIGHT.x = max(t.x, DOWN_RIGHT.x); 
	DOWN_RIGHT.y = max(t.y, DOWN_RIGHT.y);
}

void Nacrtaj( Duz d, COLOR c ) {
	d.color = c;
	duzi.push_back( d );

	UP_LEFT.x = min(d.start.x, UP_LEFT.x); 
	UP_LEFT.y = min(d.start.y, UP_LEFT.y); 
	DOWN_RIGHT.x = max(d.start.x, DOWN_RIGHT.x); 
	DOWN_RIGHT.y = max(d.start.y, DOWN_RIGHT.y);  
	
	UP_LEFT.x = min(d.end.x, UP_LEFT.x); 
	UP_LEFT.y = min(d.end.y, UP_LEFT.y); 
	DOWN_RIGHT.x = max(d.end.x, DOWN_RIGHT.x); 
	DOWN_RIGHT.y = max(d.end.y, DOWN_RIGHT.y);
}

void Nacrtaj( Prava p, COLOR c ) {
	p.color = c;
	prave.push_back( p );
}

void Nacrtaj( Krug k, COLOR c ) {
	k.color = c;
	krugovi.push_back( k );

	UP_LEFT.x = min(k.centar.x - k.r, UP_LEFT.x); 
	UP_LEFT.y = min(k.centar.y - k.r, UP_LEFT.y); 
	DOWN_RIGHT.x = max(k.centar.x + k.r, DOWN_RIGHT.x); 
	DOWN_RIGHT.y = max(k.centar.y + k.r, DOWN_RIGHT.y);
}

//inicijalizuje okruzenje i poziva funkciju NacrtajObjekte() koja zaista crta geometrijske objekte
void Prikazi( int argc, char **argv ) {
	glutInit(&argc, argv);

    // kreira prozor 
    glutInitDisplayMode(GLUT_RGB | GLUT_DEPTH | GLUT_DOUBLE);
    glutInitWindowSize(WINDOW_WIDTH, WINDOW_HEIGHT);
    glutInitWindowPosition(WINDOW_X, WINDOW_Y);
    glutCreateWindow(WINDOW_TITLE);
    glClearColor(0.9, 0.9, 0.8, 0.0);

    // podesava vidljivost 
	VIEWER_X = (UP_LEFT.x + DOWN_RIGHT.x)/2.0;
	VIEWER_Y = (UP_LEFT.y + DOWN_RIGHT.y)/2.0;
	VIEWER_Z = 5*max((fabs(UP_LEFT.x) + DOWN_RIGHT.x), (fabs(UP_LEFT.y) + DOWN_RIGHT.y));

    glMatrixMode(GL_PROJECTION);
    glLoadIdentity();
    gluPerspective(FOV_ANGLE, WINDOW_WIDTH/WINDOW_HEIGHT, CLIPPLANE_NEAR, CLIPPLANE_FAR);
    gluLookAt( VIEWER_X, VIEWER_Y, VIEWER_Z, VIEWER_X, VIEWER_Y, 0.0, 0.0, 1.0, 0.0);

    glMatrixMode(GL_MODELVIEW);
    glLoadIdentity();
    glTranslatef(-VIEWER_X, -VIEWER_Y, -VIEWER_Z);

    glutDisplayFunc( NacrtajObjekte );
	glutIdleFunc( NacrtajObjekte );
	glutMouseFunc( SacuvajPoziciju );
	glutMotionFunc( TransformisiScenu );
	glutReshapeFunc( PromeniDimenziju );
	glutMainLoop();
}

//funkcije misa
int sacuvajX, sacuvajY;
bool klikLevo = false, klikDesno = false, klikSrednji = false;
Tacka centar;

void SacuvajPoziciju( int button, int state, int x, int y ) {
	if( state == GLUT_DOWN ) {
		sacuvajX = x;
		sacuvajY = y;
		
		GLint viewport[4];
		glGetIntegerv( GL_VIEWPORT, viewport );

		GLdouble mvMatrica[16], projMatrica[16];
		GLint stvarnoY; 
		GLdouble wx, wy, wz; 
		
		glGetIntegerv (GL_VIEWPORT, (GLint*)viewport);
		glGetDoublev (GL_MODELVIEW_MATRIX, mvMatrica);
		glGetDoublev (GL_PROJECTION_MATRIX, projMatrica);
		stvarnoY = viewport[3] - (GLint) y - 1;
		gluUnProject ((GLdouble) x, (GLdouble) stvarnoY, 0.0, mvMatrica, projMatrica, viewport, &wx, &wy, &wz);
		centar = *new Tacka( wx, wy, wz );

		switch ( button ) {
            case GLUT_LEFT_BUTTON:   klikLevo = true;    break;
            case GLUT_MIDDLE_BUTTON: klikSrednji = true; break;
			case GLUT_RIGHT_BUTTON:  klikDesno = true;   break;
        }
	}
	else {
		klikLevo = false;
		klikDesno = false;
		klikSrednji = false;
	}

	glutPostRedisplay();
}

//u zavisnosti od komande, rotira, translira ili skalira sliku
void TransformisiScenu( int x, int y ) {
	bool izmenjen = false;

	const int dX = x - sacuvajX;
	const int dY = y - sacuvajY;
	if( dX == 0 && dY == 0 ) return; //nema promene scene

	GLint viewport[4];
    glGetIntegerv( GL_VIEWPORT, viewport );
	
	GLdouble mvMatrica[16], projMatrica[16];
	GLint stvarnoY; 
	GLdouble wx, wy, wz; 
		
	glGetIntegerv (GL_VIEWPORT, (GLint*)viewport);
	glGetDoublev (GL_MODELVIEW_MATRIX, mvMatrica);
	glGetDoublev (GL_PROJECTION_MATRIX, projMatrica);
	stvarnoY = viewport[3] - (GLint) y - 1;
	gluUnProject ((GLdouble) x, (GLdouble) stvarnoY, 0.0, mvMatrica, projMatrica, viewport, &wx, &wy, &wz);

	if( klikSrednji || ( klikLevo && klikDesno ) ) { //sklairanje
		float lambda = exp( (float)dY/( 3.0*WINDOW_HEIGHT ) );

		glTranslatef( VIEWER_X, VIEWER_Y, VIEWER_Z );
		glScalef( lambda, lambda, lambda );
		glTranslatef( -VIEWER_X, -VIEWER_Y, -VIEWER_Z );
		izmenjen = true;
	} 
	else if( klikDesno ) { //rotacija
		float ugao = sqrt( (float)( dX*dX + dY*dY ) )/(float)( viewport[2] + 1 )*180.0;
		if( dX > 0 ) ugao *= -1.0f;
		glTranslatef( VIEWER_X, VIEWER_Y, VIEWER_Z );
		glRotatef( ugao, 0, 0, 1 );
		glTranslatef( -VIEWER_X, -VIEWER_Y, -VIEWER_Z );

		izmenjen = true;
	}
	else if( klikLevo ) { //translacija
		glTranslatef( wx - centar.x, wy - centar.y, wz - centar.z );
		izmenjen = true;
	}	

	if( izmenjen ) {
		sacuvajX = x;
		sacuvajY = y;

		glutPostRedisplay();
	}
}

//podesava razmeru crteza pri promeni dimenzije prozora
void PromeniDimenziju( int w, int h ) {
	if( h == 0 )	h = 1;
	float ratio = 1.0* w / h;

	glMatrixMode( GL_PROJECTION );
	glLoadIdentity();
	
	glViewport( 0, 0, w, h );

	gluPerspective( FOV_ANGLE, ratio, CLIPPLANE_NEAR, CLIPPLANE_FAR );
}

//crta objekte
void NacrtajObjekte() {
	glClear( GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT );
	glPushMatrix();

	for( list<Krug>::iterator k = krugovi.begin(); k != krugovi.end(); k++ ) k->Nacrtaj();
	for( list<Prava>::iterator p = prave.begin(); p != prave.end(); p++ ) p->Nacrtaj();
	for( list<Duz>::iterator d = duzi.begin(); d != duzi.end(); d++ ) d->Nacrtaj();
	for( list<Tacka>::iterator t = tacke.begin(); t != tacke.end(); t++ ) t->Nacrtaj();

    glPopMatrix();
    glFlush();
    glutSwapBuffers();
}

