#include <stdio.h>
#include <ctype.h>
#include <limits.h>

#define MEM_SIZE 256 /* Velicina memorije u bajtovima*/
#define AC_SIZE  256 /* Raspon vrednosti akumulatora */
#define NAME_LEN 80  /* Maksimalna duzina imena datoteke */

/* Operacioni kodovi instrukcija */
#define MUA  0x00 /* Memorija u akumulator */
#define AUM  0x01 /* Akumulator u memoriju */
#define ADD  0x02 /* Sabiranje */
#define CPL  0x03 /* Potpuni komplement */
#define AND  0x04 /* Bitska konjunkcija */
#define XOR  0x05 /* Bitska ekskluzivna disjunkcija */
#define SHL  0x06 /* Pomeranje akumulatora ulevo */
#define SHR  0x07 /* Aritmeticko pomeranje akumulatora udesno */
#define HALT 0x08 /* Zaustavljanje izvrsavanja */
#define JZ   0x09 /* Skok ako je postavljen zero flag */
#define ADDC 0x0A /* Sabiranje ukljucujuci prenos */
#define FUA  0x0B /* Flag-ovi u akumulator */

/* Nacini adresiranja */
#define DIRECT    0x00 /* Direktno adresiranje */
#define IMMEDIATE 0x01 /* Neposredno adresiranje */
#define INDIRECT  0x02 /* Indirektno adresiranje */
#define RELATIVE  0x03 /* Relativno adresiranje */

/* Maske */
#define ADDR_MODE_MASK 0x03 /* Maska za izdvajanje nacina adresiranja */

/* Nacini izvrsavanja */
#define RUN      0 /* Kompletno izvrsavanje */
#define STEP     1 /* Izvrsavanje jednog koraka */

/* Makroi za manipulaciju heksadekadnim ciframa */
#define IS_HEX(c) (isdigit(c) || tolower(c) >= 'a' && tolower(c) <= 'f')
#define HEX_VALUE(c) (isdigit(c)?c-'0':tolower(c)-'a'+10)

/* Tip podataka memorije */
typedef unsigned char byte;

/* Globalne promenljive */
byte memory[MEM_SIZE]; /* Memorija */
int inst_len[]={2,2,2,1,2,2,2,2,1,2,2,1}; /* Duzine instrukcija */
int breakpts[MEM_SIZE]; /* Logicki niz koji odredjuje postojanje breakpoint-a */

byte ic; /* Brojac instrukcija (instruction counter) */
byte ac; /* Akumulator (accumulator) */

/* Flag-ovi */
struct
{
  byte v : 1; /* Overflow - prekoracenje kod oznacenog sabiranja */
  byte s : 1; /* Sign - znak vrednosti akumulatora */
  byte z : 1; /* Zero - akumulator==0 */
  byte c : 1; /* Carry - prekoracenje (prenos) kod neoznacenog sabiranja */
} flags;

/* Funkcija koja vraca operacioni kod instrukcije */
byte GetOpCode(byte data)
{
  return data >> 4;
}

/* Funkcija koja izdvaja nacin adresiranja instrukcije */
byte GetAddrMode(byte data)
{
  return (data & ADDR_MODE_MASK);
}

/* Funkcija koja uzimajuci u obzi nacin adresiranja koristi adresu *
 * addr za podataka uzimanje iz memorije                           */
byte GetData(byte addr, int addr_mode)
{
  switch(addr_mode)
    {
    case DIRECT:
      return memory[addr];
    case IMMEDIATE:
      return addr;
    case INDIRECT:
      return memory[memory[addr]];
    case RELATIVE:
      return memory[(byte)(ic+addr)];
    }
}

/* Funkcija koja simulira izvrsavanje programa. *
   addr - adresa na kojoj pocinje izvrsavanje   *
   mode - Nacin izvrsavanja: RUN ili STEP       */
void Run(byte addr, int mode)
{
  /* Promenljiva koja pamti adresu poslednjeg breakpoint-a *
   * koji je zaustavio izvrsavanje. Sluzi da se ne bismo   *
   * zaustavljali na istom breakpoint-u kad god pokusamo   *
   * da nastavimo izvrsavanje.                             */
  static int lastbp=-1;
  ic=addr; /* Postavljamo ic na adresu sa koje nastavljamo izvrsavanje */

  /* U petlji uzimamo instrukcije i podatke iz memorije */
  while(1)
    {
      byte op_code;   /* Operacioni kod instrukcije */
      byte data;      /* Podatak */
      byte addr_mode; /* Nacin adresiranja */
      byte tempc;     /* Privremeno cuvanje carry flaga */
      int sign;       /* Znak sabiraka */
      int both;       /* Signal da su oba sabirka istog znaka */
      int first_bit;  /* Promenljiva za cuvanje najtezeg bita */

      data=memory[ic]; /* Uzimamo instrukciju iz memorije */
      op_code=GetOpCode(data); /* Izdvajamo operacioni kod */


      /* Proveravamo breakpoint */
      if(breakpts[ic] && mode!=STEP && lastbp!=ic)
        {
	  lastbp=ic; /* Pamtimo adresu breakpoint-a */

	  /* Stampamo podatke o breakpoint-u i instrukciji *
	   * koja je sledeca na redu za izvrsavanje        */
	  printf("Breakpoint: 0x%02X\n",ic);
	  printf("Next 0x%02X: ",ic);
	  printf("%02X",memory[ic]);
	  if(inst_len[op_code]==2 && ic < MEM_SIZE-1)
	    printf(" %02X\n",memory[ic+1]);
	  else
	    putchar('\n');
	  return;
        }

      lastbp=-1; /* Posto je bp prosao resetujemo promenljivu */
      if((data&0x0F) > RELATIVE) /* Proveravamo korektnost nacina adresiranja*/
        {
	  printf("Nepoznat nacin adresiranja: %x\n",data&0x0F);
	  return;
        }
      addr_mode=GetAddrMode(data); /* Izdvajamo nacin adresiranja */
      if(op_code==HALT) /* Zaustavljamo izvrsavanje ako smo naisli na HALT */
	return;
      /* Ako instrukcija zahteva argumente, uzimamo ih iz memorije */
      if(inst_len[op_code]==2)
	/* Pazimo da adresa bude validna */
	if(ic==MEM_SIZE-1)
	  {
	    printf("Adresa van opsega 0x%x!\n", MEM_SIZE);
	    return;
	  }
	else
	  data=memory[ic+1];
      /* Ako je adresiranje relativno, pazimo na validnost adresa */
      if(addr_mode==RELATIVE)
	if(data < MEM_SIZE/2 && ic >= MEM_SIZE - data)
	  {
	    printf("Adresa van opsega 0x%x!\n", (int)data+ic);
	    return;
	  }
	else if(data >= MEM_SIZE/2 && ic < MEM_SIZE - data)
	  {
	    printf("Adresa van opsega -0x%x!\n", data-ic);
	    return;
	  }

      /* Izvrsavanje instrukcije */
      switch(op_code)
        {
	case MUA:
	  ac = GetData(data, addr_mode); /* Uzimamo podatke iz memorije */
	  flags.s = ac >= AC_SIZE/2; /* Postavljamo flag-ove */
	  flags.z = ac==0;
	  break;
	case AUM:
	  /* AUM je specificna u pogledu nacina adresiranja, pa *
	   * je izvrsavamo malo drugacije od drugih instrukcija */
	  if(addr_mode==IMMEDIATE)
	    {
	      printf("Neposredno adresiranje nije definisano za instrukciju AUM!\n");
	      return;
	    }
	  else if(addr_mode==DIRECT)
	    memory[data]=ac;
	  else if(addr_mode==INDIRECT)
	    memory[memory[data]]=ac;
	  else
	    memory[(byte)(ic+addr)]=ac;
	  break;
	case ADD:
	  data = GetData(data, addr_mode);
	  /* Izracunavamo prenos (prekoracenje) pri pretpostavci *
	   * da su sabirci neoznaceni brojevi                    */
	  flags.c = data >= AC_SIZE - ac;

	  /* Signaliziramo da li su oba sabirka istog znaka kao *
	   * oznaceni brojevi i pamtimo o kom znaku se radi     */
	  if(data >= AC_SIZE/2 && ac >= AC_SIZE/2 || data < AC_SIZE/2 && ac < AC_SIZE/2)
	    {
	      both=1;
	      sign = ac >= AC_SIZE/2;
	    }
	  else
	    both=0;
	  ac += data; /* Vrsimo sabiranje */

	  /* Ako su sabirci bili istog znaka moglo je doci do *
	   * prekoracenja, pa to proveravamo                  */
	  flags.v = both?((ac&0x80)==0x80)!=sign:0;
	  flags.s = ac >= AC_SIZE/2;
	  flags.z = ac==0;
	  break;
	case CPL:

	  /* U slucaju komplementiranja najmanjeg negativnog broja *
	   * dolazi do prekoracenja, pa na to moramo paziti        */
	  flags.v = ac==AC_SIZE/2;
	  ac = AC_SIZE - ac; /* Komplementiranje */
	  flags.s = ac >= AC_SIZE/2;
	  flags.z = ac==0;
	  break;
	case AND:
	  ac &= GetData(data, addr_mode); /* Vrsimo bitsku konjunkciju */
	  flags.s = ac >= AC_SIZE/2;
	  flags.z = ac==0;
	  break;
	case XOR:
	  ac ^= GetData(data, addr_mode); /* Bitsko ekskluzivno ili */
	  flags.s = ac >= AC_SIZE/2;
	  flags.z = ac==0;
	  break;
	case SHL:
	  data = GetData(data, addr_mode);
	  first_bit=ac&0x80;

	  /* Ako je izbijena neka jedinica u slucaju pozitivnog ili    *
	   * nula u slucaju negativnog broja, doslo je do prekoracenja */
	  if(!first_bit)
	    flags.v = (ac&((signed char)0x80>>(data-1)))!=0;
	  else
	    flags.v = ~(ac|(0xFFu>>data))!=0;

	  /* Poslednji istisnuti bit ide u carry flag */
	  flags.c=((ac&(0x80u>>(data-1)))==(0x80u>>(data-1)));
	  ac <<= data;  /* Pomeranje */

	  /* Ako je znak promenjen doslo je do prekoracenja */
	  flags.v = flags.v || (first_bit!=(ac&0x80));
	  flags.s = ac >= AC_SIZE/2;
	  flags.z = ac==0;
	  break;
	case SHR:
	  data = GetData(data, addr_mode);

	  /* Poslednji istisnuti bit ide u carry flag */
	  flags.c=(ac&(0x01<<(data-1)))==(0x01<<(data-1));

	  /* Pomeranje vrsimo kao aritmeticko */
	  ac = (signed char)ac >> data;
	  flags.s = ac >= AC_SIZE/2;
	  flags.z = ac==0;
	  break;
	case JZ:

	  /* Ako je postavljen zerro flag menjamo brojac instrukcija *
	   * kako bismo nastavili izvrsavanje na drugom mestu        */
	  if(flags.z)
	    {
	      ic = GetData(data, addr_mode);
	    }
	  break;
	case ADDC:

	  /* Prvo sabiramo sabirke, kao i ranije */
	  data = GetData(data, addr_mode);
	  tempc=flags.c; /* U medjuvremenu se cuva carry flag */
	  flags.c = data >= AC_SIZE - ac; /* Kao kod obicnog sabiranja */
	  if(data >= AC_SIZE/2 && ac >= AC_SIZE/2 || data < AC_SIZE/2 && ac < AC_SIZE/2)
	    {
	      both=1;
	      sign = ac >= AC_SIZE/2;
	    }
	  else
	    both=0;
	  ac += data;
	  flags.v = both?((ac&0x80)==0x80)!=sign:0;

	  /* Sada na medjurezultat dodajemo carry flag koji smo sacuvali */
	  data=tempc;

	  /* Prenosa ima ako ga je bilo ranije ili ako je nastao sada */
	  flags.c = flags.c || data >= AC_SIZE - ac;
	  if(ac < AC_SIZE/2) /* Testovi su sad jednostavniji */
	    both=1;
	  else
	    both=0;
	  ac += data;

	  /* Za overflow vazi slicno kao za carry, s tim sto vise   *
	   * nema potrebe za cuvanjem znaka jer je carry pozitivan, *
	   * pa ako su sabirci istog znaka, onda su pozitivni       */
	  flags.v = flags.v || both?((ac&0x80)==0x80)!=0:0;
	  flags.s = ac >= AC_SIZE/2;
	  flags.z = ac==0;
	  break;
	case FUA:

	  /* Postavljamo flag-ove u akumulator redosledom VSZC */
	  ac = (flags.v << 3) + (flags.s << 2) + (flags.z << 1) + flags.c;
	  flags.z = ac==0;
	  flags.s = 0;
	  break;
	default:
	  printf("Nepoznat kod instrukcije: 0x%02X\n", op_code);
	  return;
        }

      /* Ako nije bilo skoka ... */
      if(!(op_code==JZ && flags.z==1))
	/* ... pazimo na validnost nove adrese ... */
	if(ic >= MEM_SIZE - inst_len[op_code])
	  {
	    printf("Adresa van opsega 0x%x!\n", (int)ic+inst_len[op_code]);
	    return;
	  }
	else
	  /* ... i dobijamo je dodavanjem duzine instrukcije na brojac */
	       		
	  {
	    ic+=inst_len[op_code];
	  }
      /* U slucaju izvrsavanja korak po korak, stampamo podatke *
       * o sledecoj instrukciji i njegnom argumentu ako ga ima  *
       * i ako bi njegova adresa bila validna                   */
      if(mode==STEP)
        {
	  printf("Next 0x%02X: ",ic);
	  printf("%02X",memory[ic]);
	  if(inst_len[GetOpCode(memory[ic])]==2 && ic < MEM_SIZE-1)
	    printf(" %02X\n",memory[ic+1]);
	  else
	    putchar('\n');
	  return;
        }
    }
}

/* Funkcija koja puni memoriju sadrzajem fajla */
int LoadMem(FILE *f)
{
  int c;         /* Pomocna promenljiva u koju smestamo rezultat getchar-a */
  int addr_ld=0; /* Broj polubajtova adrese koji su ucitani */
  byte addr=0;   /* Adresa */
  byte data;     /* Podatak */
  int cnt=0;     /* Broj procitanih polubajtova od poslednje procitane adrese */

  /* Ucitavanje sadrzaja fajla u memoriju */
  while((c=fgetc(f))!=EOF)
    {
      if(c==' ' || c=='\t' || c=='\r') /* Preskacemo razmake i tabulatore */
	continue;
      else if(c=='\n') /* U slucaju novog reda, resetujemo promenljive */
        {
	  cnt=0;
	  addr_ld=0;
	  continue;
        }
      else if(c=='@')
        {
	  while((c=fgetc(f))!=EOF && c!='\n');
	  cnt=0;
	  addr_ld=0;
	  continue;
        }

      /* Svi karakteri moraju biti heksadekadne cifre */
      if(!IS_HEX(c))
	{
	  return 0;
	}
      /* U memoriji mora biti mesta za podatke */
      if(addr_ld==2 && cnt%2==0 && MEM_SIZE-cnt/2==addr)
	return 0;

      /* Prvi polubajt adrese smestamo u visi polubajt promenljive */
      if(addr_ld==0)
        {
	  addr=HEX_VALUE(c) << 4;
	  addr_ld++; /* Povecavamo broj procitanih polubajtova adrese */
        }

      /* Drugi polubajt adrese smestamo u nizi polubajt promenljive */
      else if(addr_ld==1)
        {
	  addr+=HEX_VALUE(c);
	  addr_ld++;
        }

      /* Ako je ucitana adresa, ali nije podatak, onda njegov prvi *
       * polubajt smestamo u visi polubajt promenljive             */
      else if(cnt%2==0)
        {
	  data=HEX_VALUE(c) << 4;
	  cnt++; /* Povecavamo broj procitanih polubajtova podataka */
        }

      /* Drugi polubajt podataka smestamo u nizi polubajt promenljive */
      else
        {
	  data+=HEX_VALUE(c);
	  /* Kad je procitan drugi polubajt, dobijeni bajt *
	   * smestamo u memoriju                           */
	  memory[addr+cnt++/2]=data;
        }
    }
  return 1; /* Signaliziramo da punjenje memorije uspesno izvrseno */
}

int GetByte(void)
{
  int c;
  int value;

  /* Preskacemo vodece beline */
  while((c=getchar())!=EOF && isspace(c) && c!='\n');

  /* Posto postoji potreba da se '\n' obradi na poseban *
   * nacin, eksplicitno ga prijavljujemo                */
  if(c=='\n')
    return -c;

  /* Proveravamo ispravnost cifara i formiramo bajt */
  if(!IS_HEX(c))
    return -1;
  value=HEX_VALUE(c) << 4;
  c=getchar();
  if(c=='\n')
    return -c;
  if(!IS_HEX(c))
    return -1;
  value+=HEX_VALUE(c);
  return value; /* Vracamo ucitanu vrednost */
}

int main()
{
  int c=0; /* Promenljiva u kojoj cuvamo povratnu vrednost getchar-a */
  flags.z=1;
  while((c=getchar())!=EOF)
    {
      if(isspace(c)) /* Preskacemo beline */
	continue;
      if(c=='q') /* Prekidamo kad nadjemo na q (quit) */
	break;
      else if(c=='r') /* r (run) sluzi za pokretanje programa */
        {
	  int addr; /* Promenljiva u kojoj pamtimo adresu */

	  /* Uzimamo bajt, a ukoliko dodje do greske, prijavljujemo je */
	  if((addr=GetByte())<0)
            {
	      printf("Losa adresa!\n");
	      if(addr!=-'\n')
		while((c=getchar())!='\n' && c!=EOF);
	      continue;
            }

	  /* Uklanjamo ostatak linije sa ulaza */
	  while((c=getchar())!=EOF && c!='\n');
	  Run((byte)addr, RUN);  /* Pokrecemo program sa adrese addr */
        }
      else if(c=='l') /* Komanda l (load) puni memoriju sadrzajem fajla */
        {
	  char s[NAME_LEN+1]; /* Pomocna promenljiva za ime fajla */
	  int i=0;            /* Indeks niza s */
	  FILE *f;            /* Pokazivac na fajl */
	  while((c=getchar())!=EOF && isspace(c) && c!='\n');
	  if(c=='\n' || c==EOF)
            {
	      printf("Mora biti zadato ime datoteke!\n");
	      continue;
            }

	  /* Ucitavamo ime fajla */
	  s[i++]=c;
	  while((c=getchar())!=EOF && c!='\n' && i<NAME_LEN)
	    s[i++]=c;
	  s[i]='\0';

	  /* Sklanjamo ostatak linije sa ulaza */
	  if(c!='\n')
	    while((c=getchar())!='\n' && c!=EOF);
	  if((f=fopen(s,"r"))==NULL) /* Otvaramo fajl */
            {
	      printf("Datoteka %s ne moze biti otvorena!\n", s);
	      continue;
            }
	  if(!LoadMem(f)) /* Punimo memoriju sadrzajem datoteke */
            {
	      printf("Greska pri unosu u memoriju!\n");
	      fclose(f);
	      continue;
            }
	  fclose(f);  /* Zatvaramo datoteku */
	  flags.v=0;
	  flags.c=0;
	  flags.z=0;
	  flags.s=0;
        }

      /* Komanda p (print) stampa sadrzaj neke adrese ili opsega adresa */
      else if(c=='p')
        {
	  int addr1; /* Adresa od koje pocinjemo ispis */
	  int addr2; /* Adresa na kojoj zavrsavamo ispis */
	  int i;     /* Brojac */
	  if((addr1=GetByte())<0)
            {
	      printf("Losa adresa!\n");
	      if(addr1!=-'\n')
		while((c=getchar())!='\n' && c!=EOF);
	      continue;
            }
	  while((c=getchar())!=EOF && c!='\n' && isspace(c));

	  /* Ukoliko postoji druga adresa deo uzimamo je, a ako ne *
	   * onda drugu adresu postavljamo na prvu. Drugu mozemo   *
	   * pomocu scanf-a jer nema potrebe da proveravam         */
	  if(c=='-')
            {
	      if((addr2=GetByte())<0)
                {
		  printf("Losa adresa!\n");
		  if(addr2!=-'\n')
		    while((c=getchar())!='\n' && c!=EOF);
		  continue;
                }
            }
	  else
	    addr2=addr1;
	  if(c!='\n')
	    while((c=getchar())!=EOF && c!='\n');

	  /* Ispisujemo adrese i sadrzaje memorijskih lokacija *
	   * u trazenom opsegu.                                */
	  for(i=addr1; i<=addr2; i++)
	    printf("%02X: 0x%02X\n",i,memory[i]);
        }
      else if(c=='f') /* Komanda f (flags) stampa flag-ove */
        {
	  while((c=getchar())!=EOF && c!='\n');
	  printf("Flags (VSZC): %c%c%c%c\n",flags.v+'0',flags.s+'0',flags.z+'0',flags.c+'0');
        }
      else if(c=='b') /* Komanda b manipulise breakpoint-ima */
        {
	  int addr;
	  int i;

	  /* Ako nema adrese, stampaju se pozicije svh breakpoint-a *
	   * a akoje ima, ona se uzima sa ulaza                     */
	  if((addr=GetByte())<0)
            {
	      printf("Breakpoints:\n");
	      for(i=0; i<MEM_SIZE; i++)
		if(breakpts[i])
		  printf("0x%02X\n",i);
	      if(c!='\n')
		while((c=getchar())!=EOF && c!='\n');
	      continue;
            }
	  while((c=getchar())!=EOF && c!='\n');

	  /* Ukoliko je data adresa i breakpoint na njoj ne postoji, *
	   * postavlja se, a ukoliko postoji, ponistava se           */
	  breakpts[addr]=!breakpts[addr];
        }

      /* Komanda c (continue) nastavlja prekinuto izvrsavanje programa */
      else if(c=='c')
        {
	  while((c=getchar())!=EOF && c!='\n');
	  Run(ic, RUN);
        }
      else if(c=='a') /* Ova komanda stampa sadrzaj akumulatora */
        {
	  while((c=getchar())!=EOF && c!='\n');
	  printf("ACC: 0x%02X\n", ac);
        }
      else if(c=='i') /* Ova komanda stampa sadrzaj instruction countera */
        {
	  while((c=getchar())!=EOF && c!='\n');
	  printf("IC: 0x%02X\n", ic);
        }
      else if(c=='n') /* n (next) izvrsava jednu - sledecu instrukciju */
        {
	  while((c=getchar())!=EOF && c!='\n');
	  Run(ic, STEP);
        }
      else if(c!='\n') /* Svaku neprepoznatu komandu prijavljujemo */
        {
	  printf("Losa komanda!\n");
	  while((c=getchar())!='\n' && c!=EOF);
	  continue;
        }
    }
  return 0; /* Signaliziramo da je izvrsavanje bilo uspesno */
}
