#include <iostream>
#include <set>
#include <map>
#include <vector>
#include <memory>
#include <algorithm>
#include <iterator>

using namespace std;

typedef set<unsigned> AtomSet;

class BaseFormula;

typedef shared_ptr<BaseFormula> Formula;

class BaseFormula : public enable_shared_from_this<BaseFormula> {

public:
  enum Type { T_TRUE, T_FALSE, T_ATOM, T_NOT, T_AND, T_OR, T_IMP, T_IFF };
  
  virtual void printFormula(ostream & ostr) const = 0;
  virtual Type getType() const = 0;
  virtual void getAtoms(AtomSet & set) const = 0;
  virtual Formula substitute(unsigned p, bool t) = 0;
  virtual ~BaseFormula() {}
};

ostream & operator << (ostream & ostr, const Formula & f);

class AtomicFormula : public BaseFormula {
};

class LogicConstant : public AtomicFormula {

public:
  virtual Formula substitute(unsigned p, bool t);
  virtual void getAtoms(AtomSet & set) const; 
};

class True : public LogicConstant {

public:
  virtual void printFormula(ostream & ostr) const;
  virtual Type getType() const;
};

class False : public LogicConstant {

public:
  virtual void printFormula(ostream & ostr) const;
  virtual Type getType() const;
};

class Atom : public AtomicFormula {
private:
  unsigned _var_num;
public:
  Atom(unsigned num);
  unsigned getVariableNumber() const; 
  virtual void printFormula(ostream & ostr) const;
  virtual Type getType() const;
  virtual void getAtoms(AtomSet & set) const;
  virtual Formula substitute(unsigned p, bool t);
};

class UnaryConnective : public BaseFormula {
protected:
  Formula _op;
public:
  UnaryConnective(const Formula & op);
  const Formula & getOperand() const;
  virtual void getAtoms(AtomSet & set) const;
 
};

class Not : public UnaryConnective {
public:

  using UnaryConnective::UnaryConnective;
  
  virtual void printFormula(ostream & ostr) const;
  virtual Type getType() const;
  virtual Formula substitute(unsigned p, bool t);
};

class BinaryConnective : public BaseFormula {
protected:
  Formula _op1; 
  Formula _op2;
public:
  BinaryConnective(const Formula & op1, const Formula & op2);
  const Formula & getOperand1() const;
  const Formula & getOperand2() const; 
  virtual void getAtoms(AtomSet & set) const;
 
};

class And : public BinaryConnective {
public:
  using BinaryConnective::BinaryConnective;
  
  virtual void printFormula(ostream & ostr) const;
  virtual Type getType() const;
  virtual Formula substitute(unsigned p, bool t);
};

class Or : public BinaryConnective {
public:
  using BinaryConnective::BinaryConnective;

  virtual void printFormula(ostream & ostr) const; 
  virtual Type getType() const;
  virtual Formula substitute(unsigned p, bool t);
};

class Imp : public BinaryConnective {
public:
  using BinaryConnective::BinaryConnective;

  virtual void printFormula(ostream & ostr) const;
  virtual Type getType() const; 
  virtual Formula substitute(unsigned p, bool t);
};

class Iff : public BinaryConnective {
public:
  using BinaryConnective::BinaryConnective;

  virtual void printFormula(ostream & ostr) const;
  virtual Type getType() const;
  virtual Formula substitute(unsigned p, bool t);
};


// DEFINICIJE FUNKCIJA

// FUNKCIJE ZA ODREDJIVANJE TIPA FORMULE ---------------------------------

BaseFormula::Type True::getType() const
{
  return T_TRUE;
}

BaseFormula::Type False::getType() const 
{
  return T_FALSE;
}

BaseFormula::Type Atom::getType() const 
{
  return T_ATOM;
}

BaseFormula::Type Not::getType() const
{
  return T_NOT;
}

BaseFormula::Type And::getType() const
{
  return T_AND;
}

BaseFormula::Type Or::getType() const
{
  return T_OR;
}

BaseFormula::Type Imp::getType() const
{
  return T_IMP;
}

BaseFormula::Type Iff::getType() const
{
  return T_IFF;
}

// --------------------------------------------------------------------

// FUNKCIJE ZA STAMPANJE FORMULA -----------------------------------------

void True::printFormula(ostream & ostr) const
{
  ostr << "TRUE";
}

void False::printFormula(ostream & ostr) const
{
  ostr << "FALSE";
}

void Atom::printFormula(ostream & ostr) const
{
  ostr << "p_" << _var_num;
}

void Not::printFormula(ostream & ostr) const
{
  ostr << "(~" << _op << ")";
}

void And::printFormula(ostream & ostr) const
{
  ostr << "(" << _op1 <<  " /\\ " << _op2 << ")";
}

void Or::printFormula(ostream & ostr) const
{
  ostr << "(" << _op1 << " \\/ " << _op2 << ")";
}

void Imp::printFormula(ostream & ostr) const
{
  ostr << "(" << _op1 <<  " => " << _op2 << ")";  
}

void Iff::printFormula(ostream & ostr) const
{
    ostr << "(" << _op1 << " <=> " << _op2 << ")";
}
// -----------------------------------------------------------------------


// FUNKCIJE ZA IZDVAJANJE SKUPA ATOMA FORMULE -------------------------

void LogicConstant::getAtoms(AtomSet & set) const
{}

void Atom::getAtoms(AtomSet & set) const
{
  set.insert(_var_num);
}

void UnaryConnective::getAtoms(AtomSet & set) const 
{
  _op->getAtoms(set);
}

void BinaryConnective::getAtoms(AtomSet & set) const
{
  _op1->getAtoms(set);
  _op2->getAtoms(set);
}

// -----------------------------------------------------------------------

// FUNKCIJE SUBSTITUCIJE ---------------------------------------------

/* Supstitucija je ovde uproscena u odnosu na onu koja je radjena na casu,
   jer se uvek menja iskazno slovo i to uvek nekom logickom konstantom.
   Ako je parametar t == true, tada menjamo sa True, u suprotnom menjamo
   sa False. Takodje, funkcije supstitucije su ovde objedinjene sa 
   simplifikacijom, kako bismo odmah po primeni zamene uproscavali formulu,
   tokom istog prolaska kroz stablo. */

Formula LogicConstant::substitute(unsigned p, bool t)
{
  return shared_from_this();
}

Formula Atom::substitute(unsigned p, bool t)
{
  if(_var_num == p)
    {
      if(t)
	return make_shared<True>();
      else
	return make_shared<False>();
    }
  else
    return shared_from_this();
}


Formula Not::substitute(unsigned p, bool t)
{
  Formula sub_op = _op->substitute(p, t);
  if(sub_op->getType() == T_TRUE)
    return make_shared<False>();
  else if(sub_op->getType() == T_FALSE)
    return make_shared<True>();
  else
    return make_shared<Not>(sub_op);  

}

Formula And::substitute(unsigned p, bool t)
{
  Formula sub_op1 = _op1->substitute(p, t);
  Formula sub_op2 = _op2->substitute(p, t);

  if(sub_op1->getType() == T_TRUE)
    return sub_op2;
  else if(sub_op2->getType() == T_TRUE)
    return sub_op1;
  else if(sub_op1->getType() == T_FALSE ||
	  sub_op2->getType() == T_FALSE)
    return make_shared<False>();
  else
    return make_shared<And>(sub_op1, sub_op2);

  
}

Formula Or::substitute(unsigned p, bool t)
{
  Formula sub_op1 = _op1->substitute(p, t);
  Formula sub_op2 = _op2->substitute(p, t);
  
  if(sub_op1->getType() == T_FALSE) 
    return sub_op2;
  else if(sub_op2->getType() == T_FALSE)
    return sub_op1;
  else if(sub_op1->getType() == T_TRUE ||
	  sub_op2->getType() == T_TRUE)
    return make_shared<True>();
  else
    return make_shared<Or>(sub_op1, sub_op2);  
}

Formula Imp::substitute(unsigned p, bool t)
{
  Formula sub_op1 = _op1->substitute(p, t);
  Formula sub_op2 = _op2->substitute(p, t);

  if(sub_op1->getType() == T_TRUE)
    return sub_op2;
  else if(sub_op2->getType() == T_TRUE)
    return make_shared<True>();
  else if(sub_op1->getType() == T_FALSE) 
    return make_shared<True>();
  else if(sub_op2->getType() == T_FALSE)
    return make_shared<Not>(sub_op1);
  else
    return make_shared<Imp>(sub_op1, sub_op2);  
}

Formula Iff::substitute(unsigned p, bool t)
{
  Formula sub_op1 = _op1->substitute(p, t);
  Formula sub_op2 = _op2->substitute(p, t);
  
  if(sub_op1->getType() == T_FALSE && 
     sub_op2->getType() == T_FALSE)
    return make_shared<True>();
  else if(sub_op1->getType() == T_TRUE)
    return sub_op2;
  else if(sub_op2->getType() == T_TRUE)
    return sub_op1;
  else if(sub_op1->getType() == T_FALSE) 
    return make_shared<Not>(sub_op2);
  else if(sub_op2->getType() == T_FALSE)
    return make_shared<Not>(sub_op1);
  else
    return make_shared<Iff>(sub_op1, sub_op2);
}

// ----------------------------------------------------------------------

// FUNKCIJA ZA ELIMINACIJU PROMENLJIVE (DEO POD C)
Formula eliminate(const Formula & f, unsigned p)
{
  shared_ptr<Or> or_formula = make_shared<Or>(f->substitute(p, true), f->substitute(p, false));
  Formula op1 = or_formula->getOperand1();
  Formula op2 = or_formula->getOperand2();

  /* Dodatni korak simplifikacije na nivou vodece disjunkcije */
  if(op1->getType() == BaseFormula::T_FALSE || op2->getType() == BaseFormula::T_TRUE) 
    return op2;
  else if(op2->getType() == BaseFormula::T_FALSE || op1->getType() == BaseFormula::T_TRUE)
    return op1;
  else
    return or_formula;  
}

// FUNKCIJA ZA ISPITIVANJE ZADOVOLJIVOSTI (DEO POD D)
bool satisfiable(Formula f)
{
  AtomSet atoms;
  f->getAtoms(atoms);
  for(auto p : atoms)
    {
      /* Eliminisemo jednu po jednu promenljivu */
      f = eliminate(f, p);

      cout << "DEBUG: " << f << endl;

      /* Ako se svede na True, formula je zadovoljiva, a u 
	 ako se svede na False, tada je nezadovoljiva */
      if(f->getType() == BaseFormula::T_TRUE)
	return true;
      else if(f->getType() == BaseFormula::T_FALSE)
	return false;
    }

  /* Do ove tacke se nikada ne stize, jer ce se formula sigurno u nekom trenutku 
     svesti na True ili False (makar kada se ukloni poslednja promenljiva). */
  return true;
}



// OSTALE FUNKCIJE (konstruktori, get-eri...)

Atom::Atom(unsigned num)
  :_var_num(num)
{}

unsigned Atom::getVariableNumber() const 
{
  return _var_num;
}

UnaryConnective::UnaryConnective(const Formula & op)
  :_op(op)
{}

const Formula & UnaryConnective::getOperand() const
{
  return _op;
}

BinaryConnective::BinaryConnective(const Formula & op1, const Formula & op2)
  :_op1(op1),
   _op2(op2)
{}
  
const Formula & BinaryConnective::getOperand1() const
{
  return _op1;
}

const Formula & BinaryConnective::getOperand2() const
{
  return _op2;
}


// OPERATOR ZA ISPIS ----------------------------------------------------

ostream & operator << (ostream & ostr, const Formula & f)
{
  f->printFormula(ostr);
  return ostr;
}

// -----------------------------------------------------------------------

int main()
{

  Formula p = make_shared<Atom>(1);
  Formula q = make_shared<Atom>(2);
  Formula r = make_shared<Atom>(3);
  Formula s = make_shared<Atom>(4);
  Formula t = make_shared<Atom>(5);
  Formula pq = make_shared<Imp>(p, make_shared<Not>(q));
  Formula rqt = make_shared<Or>(r, make_shared<Imp>(q, t));
  Formula lh = make_shared<And>(pq, rqt);
  Formula rh = make_shared<Imp>(make_shared<And>(q, make_shared<Or>(s, make_shared<Not>(t))), make_shared<Not>(s));
  Formula f = make_shared<Imp>(lh, rh);

  cout << f << endl;

  if(satisfiable(f))
    cout << "SAT" << endl;
  else
    cout << "UNSAT" << endl;
  
  return 0;
}
