.intel_syntax noprefix

.text
.global fir

	## Funkcija fir() primjenjuje FIR filter sa datim koeficijentima
	## na dati ulazni niz. Argumenti funkcije su:
	##   [ebp+8] - broj koeficijenata
        ##   [ebp+12] - niz koeficijenata
        ##   [ebp+16] - broj elemenata ulaznog niza
        ##   [ebp+20] - ulazni niz
        ##   [ebp+24] - izlazni niz
        ## Pretpostavlja se da je broj koeficijenata veci od 0, kao i da
	## je broj elemenata ulaznog niza veci ili jednak ovom broju.
	## Funkcija ne vraca nikakvu vrijednost.
fir:
        ## Prolog funkcije.
	enter	0, 0
        push    ebx
        push    esi
        push    edi

        ## Provjerava se da li procesor podrzava SSE instrukcije.
        mov     eax, 1
        cpuid
        test    edx, 0x2000000
        jz done

        ## Cuva se sadrzaj SSE registara (i registara koprocesora).
        mov     edx, esp
        and     esp, 0xfffffff0
        sub     esp, 512
        fxsave  [esp]

        ## U registru ecx se izracunava broj elemenata izlaznog niza;
	## ovaj registar ce biti koriscen kao brojac.
        mov     ecx, [ebp+16]
        sub     ecx, [ebp+8]
        inc     ecx

        ## Registar esi ce pokazivati na tekuci element ulaznog niza i
	## incijalizuje se na adresu iza poslednjeg elementa ulaznog
	## niza koji ce ucestvovati u izracunavanju prvog elementa
        ## izlaznog niza. Registar edi ce pokazivati na tekuci element
	## izlaznog niza i postavlja se na adresu prvog elementa toga 
	## niza.
        mov     esi, [ebp+20]
        mov     eax, [ebp+8]
        shl     eax, 2
        add     esi, eax
        mov     edi, [ebp+24]

next_element:
        ## Registar ecx ce u unutrasnjoj petlji, u kojoj se racuna jedan
	## element izlaznog niza, takodje sluziti kao brojac i
        ## inicijalizuje se na broj koeficijenata filtera.  Registar ebx
        ## ce pokazivati na tekuci koeficijent filtera i inicijalizuje
	## se na adresu prvog koeficijenta filtera.
        push    ecx
        mov     ecx, [ebp+8]
        mov     ebx, [ebp+12]

        ## Svaki element izlaznog niza jednak je sumi proizvoda
	## koeficijenata filtera i odgovarajucih elemenata ulaznog
	## niza. Ovi proizvodi ce biti racunati po 4 paralelno, a
	## parcijalne sume ce biti cuvane u registru xmm0.
        xorps   xmm0, xmm0
        
next_four:
        ## Provjerava se da li ima jos 4 proizvoda koeficijenata filtera
	## i elememata ulaznog niza za paralelno izracunavanje.  Ako to
	## nije slucaj, prelazi se na pojedinacno izracunavanje proizvoda.
        cmp     ecx, 4
        jl next_one

        ## Ucitavaju se 4 naredna koeficijenta filtera u registar xmm1 i
	## 4 naredna elementa ulaznog niza u registar xmm2.  Registar
        ## xmm2 se potom preuredjuje da bi elementi ulaznog niza dosli
	## na odgovarajuca mjesta u pogledu izracunavanja proizvoda.
        movups  xmm1, [ebx]
        add     ebx, 16
        sub     esi, 16
        movups  xmm2, [esi]
        shufps  xmm2, xmm2, 0x1b

        ## Izracunavaju se 4 proizvoda i dodaju parcijalnim sumama u
	## registru xmm0.
        mulps   xmm1, xmm2
        addps   xmm0, xmm1

        ## Prelazi se na narednu iteraciju.
        sub     ecx, 4
        jmp next_four

next_one:
        ## Provjerava se da li ima jos proizvoda koeficijenata filtera
	## i elememata ulaznog niza za izracunavanje.  Ako to nije
	## slucaj, prelazi se na sumiranje parcijalnih suma.
        jecxz calc_sum

        ## Ucitava se naredni koeficijent filtera u registar xmm1 i 
        ## mnozi sa odgovarajucim elementom ulaznog niza, a proizvod se
	## potom dodaje jednoj parcijalnoj sumi u registru xmm0.
        movss   xmm1, [ebx]
        add     ebx, 4
        sub     esi, 4
        mulss   xmm1, [esi]
        addss   xmm0, xmm1

        ## Prelazi se na narednu iteraciju.
        dec     ecx
        jmp next_one

calc_sum:
        ## Na osnovu 4 parcijalne sume iz registra xmm1, izracunava se
        ## finalna suma, koja predstavlja tekuci element izlaznog niza i
	## ta vrijednost se smjesta na odgovarajucu adresu.
        movhlps xmm1, xmm0
        addps   xmm0, xmm1
        movaps  xmm1, xmm0
        shufps  xmm1, xmm1, 0x55
        addss   xmm0, xmm1
        movss   [edi], xmm0

        ## Azurira se registar esi tako da opet pokazuje iza poslednjeg
	## elementa ulaznog niza koji ce ucestvovati u izracunavanju
	## narednog elementa izlaznog niza.
        mov     eax, [ebp+8]
        inc     eax
        shl     eax, 2
        add     esi, eax

        ## Prelazi se na izracunavanje narednog elementa izlaznog niza.
        pop     ecx
        add     edi, 4
        loop next_element

        ## Vraca se prethodni sadrzaj u registre koprocesora i SSE
	## registre.
        fxrstor [esp]
        mov     esp, edx

done:	
	## Epilog funkcije.
        pop     edi
        pop     esi
        pop     ebx
	leave
        ret
