.intel_syntax noprefix
	
.text

.global quadratic_form

#############################################################################
##
## Funkcija racuna vredost kvadratne forme  f = x' * A * x, gde je A matrica
## dimenzije n x n, a x je vektor duzine n. Matrica je predstavljena nizom
## pokazivaca na vrste. Funkcija koristi paralelne SSE instrukcije.
##
## void quadratic_form(int n, float **a, float *x, float * f);
##
## -- int n     --  [ebp+8]  -- dimenzija matrice
## -- float **a --  [ebp+12] -- adresa niza pokazivaca na vrste matrice
## -- float  *x --  [ebp+16] -- pokazivac na vektor x
## -- float  *f --  [ebp+20] -- pokazivac na odredisnu lokaciju
##
#############################################################################
quadratic_form:
	## Prolog funkcije
	enter 0,0
	push edi
	push esi
	push ebx
	
	## Proverava se da li procesor podrzava SSE instrukcije.
        mov     eax, 1
        cpuid
        test    edx, 0x2000000
        jz not_supported

        ## Cuva se sadrzaj SSE registara (i registara koprocesora).
        mov     edx, esp
        and     esp, 0xfffffff0
        sub     esp, 512
        fxsave  [esp]

	## Kvadratnu formu racunacemo kao izraz f = x[0] * y[0] + ... +
	## +... + x[n-1] * y[n-1], gde je y[i] = sum_j (a[i][j] * x[j])
	## U svakoj iteraciji spoljasnje petlje cemo racunati SSE
	## instrukcijama skalarni proizvod sum_j (a[i][j] * x[j]),a zatim
	## cemo tako izracunati proizvod pomnoziti sa x[i] i dodati na sumu.
	
	## U registar ebx smestamo adresu pocetka niza pokazivaca na vrste.
	mov ebx, [ebp+12]

	## U registar esi smestamo adresu pocetka vektora x
	mov esi, [ebp+16]
	
	## U svakoj iteraciji spoljasnje petlje racunamo y[i], gde je y = A*x
	## Registar eax ce nam cuvati indeks i.
	xor eax, eax

	## Rezultat cemo akumulirati u najnizem podatku registra xmm7
	xorps xmm7, xmm7
	
next_y:
	## Poredimo i >= n
	cmp eax, [ebp+8]
	jae store

	## U registar edi ucitavamo adresu i-te vrste matrice. U registru
	## esi nalazi se adresa pocetka vektora x. Sada treba skalarno
	## pomnoziti vektore duzine n na koje pokazuju esi i edi.
	mov edi, [ebx + 4*eax]

	## Vrednost adrese pocetka vektora x odlazemo na stek, zato sto cemo
	## registar esi menjati u unutrasnjoj petlji.
	push esi
	
	## Registar ecx se postavlja na n. Ovaj registar ce u svakom trenutku
	## sadrzati broj preostalih sabiraka sume koje treba izracunati.
	mov ecx, [ebp+8]
	
	## Registar xmm0 ce nam sadrzati parcijalne sume.
	xorps xmm0, xmm0
	
next_four:
	## Ako je preostalo bar jos 4 sabirka sume...
	cmp ecx, 4
	jb remaining

	## Racunamo paralelno cetiri sabirka sume i dodajemo ih na parcijalne
	## sume.
	movups xmm1, [esi]
	add esi, 16
	movups xmm2, [edi]
	add edi, 16
	mulps xmm1, xmm2
	addps xmm0, xmm1

	## Prelazimo na sledeca cetiri sabirka
	sub ecx, 4
	jmp next_four

remaining:
	## Ako nema vise sabiraka...
	jecxz add_to_sum

next_one:	
	## Racunamo sledeci sabirak i dodajemo ga na najnizu parcijalnu sumu.
	movss xmm1, [esi]
	add esi, 4
	mulss xmm1, [edi]
	add edi, 4
	addss xmm0, xmm1

	## Prelazimo na sledeci sabirak.
	loop next_one

add_to_sum:
	## Racunamo zbir parcijalnih suma iz registra xmm0 (dobijamo y[i])
	movhlps xmm1, xmm0
	addps xmm0, xmm1
	movaps xmm1, xmm0
	shufps xmm1, xmm1, 0x55
	addss xmm0, xmm1

	## Vracamo adresu pocetka niza x u esi.
	pop esi
	
	## Mnozimo y[i] * x[i] i dodajemo na sumu.
	mulss xmm0, [esi + 4 * eax]
	addss xmm7, xmm0
	
	## Prelazimo na sledece y[i].
	inc eax
	jmp next_y

store:
	## Smestamo rezultat u za to predvidjenu lokaciju.
	mov edi, [ebp+20]
	movss [edi], xmm7
	
finish:
        ## Vraca se prethodni sadrzaj u registre koprocesora i SSE
	## registre.
        fxrstor [esp]
        mov     esp, edx
	
done:	
	## Epilog funkcije
	pop ebx
	pop esi
	pop edi
	leave
	ret

not_supported:
	## Ako SSE instrukcije nisu podrzane, prekidamo program exit(1)
	## pozivom. 
	mov eax, 1
	mov ebx, 1
	int 0x80
	