.intel_syntax noprefix

.text

.global circles

#############################################################################
##
## Funkcija odredjuje broj parova krugova u ravni takvih da je jedan od
## krugova u potpunosti sadrzan u drugom krugu. Funkcija koristi paralelne
## SSE2 instrukcije. Deklaracija funkcije je data sa:
## 
## int circles(int n, double * cr); 
##
## -- int n       --  [ebp+8]  -- broj krugova
## -- double * cr --  [ebp+12] -- adresa niza u kome se nalaze podaci o
## krugovima (redom za svaki od krugova, pri cemu se za svaki od njih navode
## x i y koordinata centra i poluprecnik).
##
#############################################################################
circles:	
	## Prolog funkcije
	enter 0,0
	push esi
	
	## Proverava se da li procesor podrzava SSE2 instrukcije.
        mov     eax, 1
        cpuid
        test    edx, 0x4000000
        jz not_supported

        ## Cuva se sadrzaj SSE registara (i registara koprocesora).
        mov     edx, esp
        and     esp, 0xfffffff0
        sub     esp, 512
        fxsave  [esp]

	## U registar ecx smestamo broj n, a u registar esi adresu pocetka
	## niza.
	mov ecx, [ebp + 8]
	mov esi, [ebp + 12]

	## Registar eax ce sluziti kao brojac pronadjenih parova.
	xor eax, eax

	## Umanjujemo brojac, zato sto ce spoljasnja petlja imati n-1
	##iteraciju.
	dec ecx

	## Spoljasnja petlja prolazi kroz niz krugova od prvog do
	## pretposlednjeg, i za svakog od njih proverava za sve krugove koji
	## su iza njega u nizu da li sa njim zadovoljavaju trazeno svojstvo
	## (bilo da se prvi krug sadrzi u drugom, ili obratno).
next_circle1:

	## Ucitavamo u xmm0 koordinate centra prvog kruga, a u nizi deo
	## registra xmm1 ucitavamo poluprecnik tog kruga.
	movupd xmm0, [esi]
	add esi, 16
	movsd xmm1, [esi]
	add esi, 8

	## Cuvamo na steku adresu esi (koja trenutno pokazuje na koordinate
	## centra narednog kruga, odakle upravo treba zapoceti sledecu
	## iteraciju spoljasje petlje) i registar ecx (cija je trenutna
	## vrednost upravo jednaka potrebnom broju iteracija nastupajuce
	## unutrasnje petlje).
	push esi
	push ecx

	## U unutrasnjoj petlji prolazimo kroz sve krugove iza tekuceg kruga
	## i za svaki od njih proveravamo da li sa tekucim krugom zadovoljava
	## dato svojstvo.
next_circle2:

	## Ucitavamo koordinate centra drugog kruga u xmm2, a poluprecnik
	## u nizi deo registra xmm3.
	movupd xmm2, [esi]
	add esi, 16
	movsd xmm3, [esi]
	add esi, 8

	## Racunamo rastojanje izmedju centara dva kruga.
	movapd xmm4, xmm0
	subpd xmm4, xmm2
	mulpd xmm4, xmm4
	movapd xmm5, xmm4
	shufpd xmm5, xmm5, 0b11
	addsd xmm4, xmm5
	sqrtsd xmm4, xmm4

	## Ako je prvi krug sadrzan u drugom, tada je zbir rastojanja izmedju
	## centara krugova i poluprecnika prvog kruga manji ili jednak od
	## poluprecnika drugog kruga. 
	movsd xmm5, xmm1
	addsd xmm5, xmm4
	comisd xmm5, xmm3
	jbe found

	## Ako je drugi krug sadrzan u prvom, tada je zbir rastojanja izmedju
	## centara krugova i poluprecnika drugog kruga manji ili jednak od
	## poluprecnika prvog kruga.
	movsd xmm5, xmm3
	addsd xmm5, xmm4
	comisd xmm5, xmm1
	jbe found
	jmp continue
	
found:
	## Ako smo pronasli par krugova sa datim svojstvom, uvecavamo brojac.
	inc eax
	
continue:
	loop next_circle2

	## Vracamo sa steka sacuvane vrednosti esi i ecx.
	pop ecx
	pop esi
	loop next_circle1

	## Povratna vrednost se sada nalazi u eax registru.
	
finish:
        ## Vraca se prethodni sadrzaj u registre koprocesora i SSE
	## registre.
        fxrstor [esp]
        mov     esp, edx
	
done:	
	## Epilog funkcije
	pop esi
	leave
	ret

not_supported:
	## Ako SSE2 instrukcije nisu podrzane, prekidamo program exit(1)
	## pozivom. 
	mov eax, 1
	mov ebx, 1
	int 0x80
	