#include <stdio.h>
#include <stdlib.h>


/* NAPOMENA: Dvostruko povezana lista je struktura podataka
   koja se sastoji od sekvence cvorova. Svaki cvor sadrzi
   podatak (odredjenog tipa) i pokazivace na prethodni i sledeci 
   cvor u sekvenci. Prvi cvor u sekvenci naziva se glava liste. 
   Ostatak liste (bez glave) je takodje lista,i naziva se rep liste.
   Lista koja ne sadrzi cvorove naziva se prazna lista. Prilikom
   baratanja listom mi cuvamo samo pokazivac na glavu liste.
   Kada pristupimo glavi liste, u njoj imamo zapisanu adresu 
   sledeceg elementa, pa mu samim tim mozemo pristupiti. Kada mu
   pristupimo, u njemu je sadrzana adresa sledeceg elementa, pa
   preko tog pokazivaca mozemo da mu pristupimo, itd. U svakom cvoru
   imamo zapisanu i adresu prethodnog elementa, pa se kroz listu
   mozemo kretati i unazad. Poslednji  element u listi nema sledeci 
   element: u tom slucaju se njegov pokazivac na sledeci postavlja 
   na NULL. Slicno, glava liste nema prethodni element, pa je njen
   pokazivac na prethodni postavljen na NULL. Takodje, prazna lista 
   se predstavlja NULL pokazivacem.

   Prednost koriscenja povezanih lista  u odnosu na dinamicki
   niz je u tome sto se elementi mogu efikasno umetati i brisati
   sa bilo koje pozicije u nizu, bez potrebe za realokacijom ili
   premestanjem elemenata. Nedostatak ovakvog pristupa je to sto
   ne mozemo nasumicno pristupiti proizvoljnom elementu, vec se 
   elementi moraju obradjivati redom (iteracijom kroz listu). 

   Prednost dvostruko povezane liste u odnosu na jednostruko povezanu
   je u tome sto se mozemo kretati u oba smera kroz listu. Posledica
   toga je da se mnoge operacije mogu jednostavnije obaviti (tipicno,
   brisanje tekuceg elementa). Nedostatak je to sto se velicina memorije
   potrebne za cuvanje cvora povecava za velicinu jednog pokazivaca. */

/* Struktura koja predstavlja cvor liste */
typedef struct cvor {
int vrednost;          /* podatak koji cvor sadrzi */
struct cvor *sledeci;  /* pokazivac na sledeci cvor */
struct cvor *prethodni; /* pokazivac na prethodni */
} Cvor;


/* Pomocna funkcija koja kreira cvor. Funkcija vrednost
   novog cvora inicijalizuje na broj, dok pokazivace na
   prethodni i sledeci cvor u novom cvoru postavlja na NULL. 
   Funkcija  ispisuje poruku o gresci i prekida program u 
   slucaju da malloc() funkcija ne uspe da alocira prostor. 
   Funkcija vraca pokazivac na novokreirani cvor */
Cvor * napravi_cvor(int broj)
{
     Cvor *novi = NULL;
     if((novi = (Cvor *) malloc(sizeof(Cvor))) == NULL)
       {
           fprintf(stderr,"malloc() greska!\n");
           exit(1);
       }

     novi->vrednost = broj;
     novi->prethodni = NULL;
     novi->sledeci = NULL;
     return novi;
}

/* Funkcija dodaje novi cvor na pocetak liste. Funkcija
   kreira novi cvor koriscenjem funkcije napravi_cvor().
   Funkcija vraca pokazivac na novu glavu liste */
Cvor * dodaj_na_pocetak_liste(Cvor *glava, int broj)
{
    Cvor * novi = napravi_cvor(broj);
    novi->sledeci = glava;
    if(glava != NULL) glava->prethodni = novi;

    return novi; 
}

/* Funkcija dodaje novi cvor na kraj liste. Funkcija
   kreira novi cvor koriscenjem funkcije napravi_cvor().
   Funkcija vraca pokazivac na glavu liste (koji moze
   biti promenjen u slucaju da je lista inicijalno bila 
   prazna). */
Cvor * dodaj_na_kraj_liste(Cvor *glava, int broj)
{
   Cvor * novi = napravi_cvor(broj);
   Cvor *tekuci = glava;

  /* slucaj prazne liste. U tom slucaju je glava nove liste
     upravo novi cvor. */
  if(glava == NULL) return novi;

  /* Ako lista nije prazna, tada se krecemo duz liste sve dok
     ne dodjemo do poslednjeg cvora (tj. do cvora ciji pokazivac
     na sledeci pokazuje na NULL) */
   while(tekuci->sledeci != NULL)
    tekuci = tekuci->sledeci;
   
   /* Dodajemo novi element na kraj preusmeravanjem pokazivaca */
   tekuci->sledeci = novi;
   novi->prethodni = tekuci;

   /* Vracamo novu - staru glavu liste */
   return glava;
}

/* Funkcija dodaje novi element u sortiranu listu tako da i nova
   lista ostane sortirana. Funkcija kreira novi cvor koriscenjem
   funkcije napravi_cvor(). Funkcija vraca pokazivac na glavu liste
   (koji moze biti promenjen u slucaju da je novi element dodat na
    pocetak liste) */
Cvor * dodaj_sortirano(Cvor *glava, int broj)
{
   Cvor * novi = napravi_cvor(broj);
   Cvor * tekuci = glava;

   /* u slucaju prazne liste glava nove liste je
      upravo novi element */
   if(glava == NULL) return novi;

   /* ako je novi element manji ili jednak od glave,
      tada novi element mora da bude nova glava */
   if(glava->vrednost >= novi->vrednost) 
     {
         novi->sledeci = glava;
         glava->prethodni = novi;
         return novi;
     }

   /* u slucaju da je glava manja od novog elementa, tada se krecemo kroz
      listu sve dok se ne dodje do elementa ciji je sledeci element veci ili
      jednak od novog elementa, ili dok se ne dodje do poslednjeg elementa. */
   while(tekuci->sledeci != NULL && tekuci->sledeci->vrednost < novi->vrednost)
     tekuci = tekuci->sledeci;

   /* U svakom slucaju novi element dodajemo IZA tekuceg elementa */
   novi->sledeci = tekuci->sledeci;
   novi->prethodni = tekuci;
   if(tekuci->sledeci != NULL)
     tekuci->sledeci->prethodni = novi;
   tekuci->sledeci = novi; 

   /* vracamo staru - novu glavu */
   return glava;
}

/* Funkcija trazi u listi element cija je vrednost jednaka 
   datom broju. Funkcija vraca pokazivac na cvor liste u 
   kome je sadrzan trazeni broj ili NULL u slucaju da takav 
   element ne postoji u listi */
Cvor * pretrazi_listu(Cvor *glava, int broj)
{
     for(; glava != NULL ; glava = glava->sledeci)
      if(glava->vrednost == broj)
        return glava;

     return NULL;
}

/* Funkcija brise tekuci element liste, tj. element liste
   na koji pokazuje pokazivac tekuci. Funkcija vraca pokazivac
   na glavu liste, koja moze biti promenjena ako je upravo
   glava obrisani cvor */
Cvor * obrisi_tekuci(Cvor * glava, Cvor * tekuci)
{
    if(tekuci == NULL)
       return glava;

    /* Preusmeravamo pokazivace prethodnog i sledeceg
       ako oni postoje */
     if(tekuci->prethodni != NULL)
       tekuci->prethodni->sledeci = tekuci->sledeci;
     if(tekuci->sledeci != NULL)
       tekuci->sledeci->prethodni = tekuci->prethodni;

    /* Ako je cvor koji brisemo glava, tada moramo da
       promenimo glavu (sledeci element postaje glava) */
     if(tekuci == glava)
       glava = tekuci->sledeci;

    /* Brisemo tekuci cvor */
       free(tekuci);
       return glava;
}

/* Funkcija brise iz liste sve cvorove koji sadrze dati broj.
   Funkcija vraca pokazivac na glavu liste (koji moze biti 
   promenjen u slucaju da se obrise stara glava) */
Cvor *obrisi_element(Cvor *glava, int broj)
{
    Cvor *tekuci = glava;
    Cvor *pomocni;

    /* Pretrazujemo listu pocev od tekuceg elementa trazeci
       datu vrednost. Ako je pronadjemo, brisemo nadjeni cvor
       i nastavljamo pretragu od elementa koji sledi nakon
       upravo obrisanog */
    while((tekuci = pretrazi_listu(tekuci, broj)) != NULL)
      {
         pomocni = tekuci->sledeci;
         glava = obrisi_tekuci(glava, tekuci);
         tekuci = pomocni;
      }
     /* vracamo novu glavu */
     return glava;
}

/* Funkcija prikazuje elemente liste pocev 
   od glave ka kraju liste */
void prikazi_listu(Cvor *glava)
{
   putchar('[');
   for(;glava != NULL; glava = glava->sledeci)
     printf("%d ", glava->vrednost);
   putchar(']');

   putchar('\n');
}

/* Funkcija prikazuje elemente liste u obrnutom poretku */
void prikazi_listu_obrnuto(Cvor * glava)
{
   /* Ako je lista prazna... */
   if(glava == NULL)
   {
    printf("[]\n");
    return;
   }

    /* Prolazimo kroz listu dok ne stignemo do poslednjeg 
       elementa. */
    while(glava->sledeci != NULL)
       glava = glava->sledeci;

    /* Ispisujemo elemente liste iteracijom u suprotnom smeru */
    putchar('[');
    for(;glava != NULL; glava = glava->prethodni)
      printf("%d ", glava->vrednost);
     putchar(']');

    putchar('\n');

}

/* Funkcija oslobadja dinamicku memoriju zauzetu 
   od strane liste. Funkcija vraca NULL, tj. 
   vrednost koju treba dodeliti pokazivackoj 
   promenljivoj, s obzirom da je sada lista prazna. */
Cvor * oslobodi_listu(Cvor *glava)
{
  Cvor *pomocni;

  while(glava != NULL)
  {
    /* moramo najpre zapamtiti adresu sledeceg
       elementa, a tek onda osloboditi glavu */
    pomocni = glava->sledeci;
    free(glava);
    glava = pomocni;
  }

 return NULL;

}



/* test program */
int main()
{

Cvor *glava = NULL;
Cvor *pomocni = NULL;
int broj;


/* Testiranje dodavanja na pocetak */
printf("-------------------------------------------------------\n");
printf("---------- Testiranje dodavanja na pocetak ------------\n");
do {
printf("-------------------------------------------------------\n");
printf("Prikaz trenutnog sadrzaja liste:\n");
prikazi_listu(glava);
printf("-------------------------------------------------------\n");
printf("Dodati element na pocetak liste (ctrl-D za kraj unosa):\n");
} while(scanf("%d", &broj) > 0 &&  
                    (glava = dodaj_na_pocetak_liste(glava, broj)) );

printf("-------------------------------------------------------\n");
putchar('\n');

/* Testiranje prikaza obrnute liste */
printf("-------------------------------------------------------\n");
printf("------------ Testiranje prikaza obrnute liste  --------\n");
printf("Prikaz liste u obrnutom poretku:\n");
prikazi_listu_obrnuto(glava);


/* Testiranje pretrage elementa */
printf("-------------------------------------------------------\n");
printf("---------------- Testiranje pretrage ------------------\n");
printf("-------------------------------------------------------\n");
printf("Prikaz trenutnog sadrzaja liste:\n");
prikazi_listu(glava);
printf("-------------------------------------------------------\n");
printf("Uneti broj koji se trazi: ");
scanf("%d",&broj);

if((pomocni = pretrazi_listu(glava, broj)) == NULL)
    printf("Trazeni broj nije u listi\n");
else
   printf("Trazeni element %d je u listi\n", pomocni->vrednost); 
printf("-------------------------------------------------------\n");
putchar('\n');

glava = oslobodi_listu(glava);

/* Testiranje dodavanja na kraj */
printf("-------------------------------------------------------\n");
printf("------------ Testiranje dodavanja na kraj -------------\n");
do{
printf("-------------------------------------------------------\n");
printf("Prikaz liste:\n");
prikazi_listu(glava);
printf("-------------------------------------------------------\n");
printf("Dodati element na kraj liste (ctrl-D za kraj unosa):\n");
} while(scanf("%d",&broj) > 0 &&  
                   (glava = dodaj_na_kraj_liste(glava, broj)));
printf("-------------------------------------------------------\n");
putchar('\n');

/* Testiranje prikaza obrnute liste */
printf("-------------------------------------------------------\n");
printf("------------ Testiranje prikaza obrnute liste  --------\n");
printf("Prikaz liste u obrnutom poretku:\n");
prikazi_listu_obrnuto(glava);

/* Testiranje brisanja elemenata */
printf("-------------------------------------------------------\n");
printf("--------------- Testiranje brisanja -------------------\n");
printf("-------------------------------------------------------\n");
printf("Prikaz trenutnog sadrzaja liste:\n");
prikazi_listu(glava);
printf("-------------------------------------------------------\n");
printf("Uneti broj koji se brise: ");
scanf("%d", &broj);

glava = obrisi_element(glava, broj);
printf("-------------------------------------------------------\n");
printf("Lista nakon izbacivanja:\n");
prikazi_listu(glava);
printf("-------------------------------------------------------\n");
putchar('\n');

glava = oslobodi_listu(glava);

/* Testiranje sortiranog dodavanja */
printf("-------------------------------------------------------\n");
printf("----------- Testiranje sortiranog dodavanja -----------\n");
do{
printf("-------------------------------------------------------\n");
printf("Prikaz liste:\n");
prikazi_listu(glava);
printf("-------------------------------------------------------\n");
printf("Dodati element sortirano (ctrl-D za kraj unosa):\n");
} while(scanf("%d",&broj) > 0 &&  
              (glava = dodaj_sortirano(glava, broj)));
printf("-------------------------------------------------------\n");
putchar('\n');

/* Testiranje prikaza obrnute liste */
printf("-------------------------------------------------------\n");
printf("------------ Testiranje prikaza obrnute liste  --------\n");
printf("Prikaz liste u obrnutom poretku:\n");
prikazi_listu_obrnuto(glava);

glava = oslobodi_listu(glava);

return 0;

}
