﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;

namespace LinqExpressions
{
    class Person
    {
        public string Name
        {
            set;
            get;
        }

        public string Gender
        {
            set;
            get;
        }


        public int YearOfBirth
        {
            set;
            get;
        }

	public override bool Equals(object obj)
	{
	   Person p = obj as Person;
	   if(p == null)
	     return false;
	   return p.Name == Name && p.Gender == Gender && p.YearOfBirth == YearOfBirth;
	}

    public static bool operator == (Person p, object o)
    {
        return Object.ReferenceEquals(p, o) || p.Equals(o);
    }
    
    public static bool operator != (Person p, object o)
    {
        return !(p == o);
    }

    public override int GetHashCode()
    {
        return Name.GetHashCode();
    }

    public override string ToString()
    {
        return Name + ", " + Gender + ", " + YearOfBirth;
    }
}

    class Program
    {
        static void Main(string[] args)
        {
            HashSet<Person> persons = new HashSet<Person>();

            persons.Add(new Person() { Name = "Pera", Gender = "Male", YearOfBirth = 1990 });
            persons.Add(new Person() { Name = "Joca", Gender = "Male", YearOfBirth = 1987 });
            persons.Add(new Person() { Name = "Sava", Gender = "Male", YearOfBirth = 1975 });
            persons.Add(new Person() { Name = "Raka", Gender = "Male", YearOfBirth = 1993 });
            persons.Add(new Person() { Name = "Ana", Gender = "Female", YearOfBirth = 1990 });
            persons.Add(new Person() { Name = "Marija", Gender = "Female", YearOfBirth = 1984 });
            persons.Add(new Person() { Name = "Ivana", Gender = "Female", YearOfBirth = 1974 });
            persons.Add(new Person() { Name = "Sanja", Gender = "Female", YearOfBirth = 1992 });

            
            // LINQ izrazi (opis linija u onom redu u kome se pojavljuju):
            //
            // from p in c -- ova linija znaci da se uzimaju elementi iz kolekcije c (tekuci element oznacen sa p). 
            //                Moze biti vise uzastopnih from linija, sto igra ulogu Dekartovog proizvoda.
            // where uslov  -- opciona linija, moze sadrzati proizvoljan C# logicki izraz. U njemu tipicno 
            //                 ucestvuju promenljive uvedene u from liniji.
            //
            // orderby vrednost [ascending|descending] -- sortira kolekciju po vrednosti, opadajuce ili rastuce.
            //                                            ukoliko zelimo dodatno sortiranje po drugoj vrednosti,
            //                                            mozemo vise vrednosti razdvojiti zarezima. Podrazumeva 
            //                                            se rastuce sortiranje.
            //
            // select vrednost -- uvek dolazi na kraju, bira sta zelimo selektovati.
            // 
            // group p by vrednost -- pri cemu se vrednost izracunava na osnovu p, gde je p varijabla uvedena
            //                        u okviru from linije. Vrednost je najcesce neko svojstvo (atribut) u p,
            //                        po kome vrsimo grupisanje. Svaki LINQ izraz se zavrsava TACNO JEDNOM od
            //                        select ili group by linija (dakle uvek stoji jedna ili druga na kraju, ali
            //                        nikako obe ne mogu biti u izrazu). group by vraca kolekciju kolekcija, pri
            //                        cemu svaka od tih kolekcija ima svojstvo Key (vrednost po kome je grupisana)
            //                        a sastoji se od svih elemenata originalne strukture koji imaju tu vrednost
            //                        datog svojstva.
            //
            // Spajanje:
            //
            // from p1 in c1
            // join p2 in c2
            // on p1.S1 equals p2.S2 -- ovim se dobija spajanje dve tabele po svojstvima S1 iz prve i S2 iz druge.
            //                          Ovo se moze postici i sa dve from linije, pa restrikcijom, ali je ovako
            //                          efikasnije.
            //
            // Pored ovih, mogu postojati jos i linije:
            //
            // let p = izraz         -- ovim se uvodi ime za neki izraz, koje se u daljem upitu moze koristiti
            //                          umesto tog izraza. Ova linija tipicno ide posle from linije, a pre uslova, 
            //                          ili izmedju visestrukih from linija (npr. za definisanje pomocnih tabela).
            //
            // into var              -- ova linija ide iza select ili group by i u nju se smesta rezultat upita,
            //                          kako bi se na ovaj upit mogao nadovezati drugi upit koji koristi rezultat
            //                          prethodnog kao tabelu koju pretrazuje. Ovo je ekvivalent SQL-ovoj WITH-AS
            //                          konstrukciji.

            // Izdvaja sve muskarce
            var males = from person in persons
                        where person.Gender == "Male"
                        select person;

            foreach (Person x in males)
            {
                Console.WriteLine(x);
            }

            // Izdvaja sve devojke, selektuje samo imena
            var females = from person in persons
                          where person.Gender == "Female"
                          select person.Name;

            foreach (string x in females)
            {
                Console.WriteLine(x);
            }

            // Izdvaja sve osobe starije od 30 godina, selektuje ime i godinu rodjenja. Sortira po godini rodjenja, a zatim po imenima, rastuce.
            var olders = from person in persons
                         where DateTime.Now.Year > person.YearOfBirth + 30
                         orderby person.YearOfBirth ascending, person.Name descending
                         select new { person.Name, person.YearOfBirth };
                         
            foreach (var x in olders)
            {
                Console.WriteLine(x);
            }

            // Izdvaja kolekciju grupa, pri cemu se grupisanje vrsi po polu.
            var groups = from person in persons
                         group person by person.Gender;

            // Za svaku grupu, prikazujemo kljuc grupisanja, a zatim i clanove grupe.
            foreach (var group in groups)
            {
                Console.WriteLine("Gender: " + group.Key);
                foreach (var elem in group)
                    Console.WriteLine(elem);
            }

            // Izdvajamo statistike o grupi: broj elemenata grupe, prosecnu vrednost godina clanova grupe.
            var groups_stat = from g in groups
                             select new { Gender = g.Key, Count = g.Count(), Average = g.Average(y => DateTime.Now.Year - y.YearOfBirth) }; 

            foreach (var s in groups_stat)
            {
                Console.WriteLine(s.Gender + " " + s.Count + " " + s.Average);
            }

//            var pairs = persons.Where(x => x.Gender == "Male").Join(persons.Where(x => x.Gender == "Female"), x => x.YearOfBirth, x => x.YearOfBirth, (x, y) => new { Male = x.Name, Female = y.Name, x.YearOfBirth });

            var pairs = from p1 in (from p in persons where p.Gender == "Male" select p)
                        join p2 in (from p in persons where p.Gender == "Female" select p)
                        on p1.YearOfBirth equals p2.YearOfBirth
                        select new { Male = p1.Name, Female = p2.Name, p1.YearOfBirth };


            foreach (var pair in pairs)
                Console.WriteLine(pair);
        }
    }
}
