﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;

namespace Lambda
{
    class Program
    {
        // Delegati predstavljaju specijalne tipove koji su 
        // ekvivalentni C-ovim pokazivacima na funkcije. 
        // Delegatski podatak moze prihvatiti referencu na 
        // bilo koju staticku funkciju odgovarajuceg tipa
        // (u ovom slucaju funkciju koja prihvata dva int-a
        // i vraca int). Delegati se tipicno koriste kao 
        // parametri funkcija ciji rad treba da bude 
        // parametrizovan nekom drugom funkcijom.
        public delegate int IntComparator(int x, int y);

        // Primer: funkcija koja sortira niz celih brojeva rastuce
        // u skladu sa definisanim poretkom (komparator bi trebalo
        // da vrati < 0 ako je prvi manji, > 0 ako je prvi veci, 
        // a nulu ukoliko su jednaki.
        static void sortArray(int[] array, IntComparator comp)
        {
            // Sortiranje izborom minimalnog elementa
            for (int i = 0; i < array.Length; i++)
            {
                int min = i;
                for (int j = i + 1; j < array.Length; j++)
                {
                    // poziv funkcije poredjenja putem delegata
                    if (comp(array[j], array[min]) < 0)
                    {
                        min = j;
                    }
                }

                if (i != min)
                {
                    int temp = array[i];
                    array[i] = array[min];
                    array[min] = temp;
                }
            }
        }

        // Delegat koji prihvata int a vraca bool
        public delegate bool IntFilter(int x);

        // Funkcija vraca novi niz nastao od datog niza izbacivanjem
        // elemenata koji ne prolaze kroz dati filter, tj. za koje
        // filter vraca false.
        static int[] getElements(int[] array, IntFilter filter)
        {
            // Najpre brojimo koliko elemenata zadovoljava uslov
            // datog filtra.
            int count = 0;
            foreach (int x in array)
                if (filter(x))
                    count++;

            // Kreiramo niz potrebne duzine.
            int[] retArray = new int[count];

            // Upisujemo u rezultujuci niz elemente koji
            // zadovoljavaju uslov datog filtra.
            for (int i = 0, j = 0; i < array.Length; i++)
            {
                if (filter(array[i]))
                    retArray[j++] = array[i];
            }

            return retArray;
        }

        // Primer staticke funkcije kojom se moze 
        // inicijalizovati delegat IntComparator
        static int ascending(int x, int y)
        {
            return x - y;
        }

        // Opadajuci poredak
        static int descending(int x, int y)
        {
            return y - x;
        }

        static void Main(string[] args)
        {
            int[] array = new int[] { 4, 1, 7, 8, 3, 9 };

            foreach(int x in array)
                Console.Write(x + " ");
            Console.WriteLine();

            // Sortiranje u opadajucem poretku. Funkciji
            // sortArray se predaje staticka funkcija descending
            // kojom se inicijalizuje delegatski parametar.
            sortArray(array, descending);

            foreach (int x in array)
                Console.Write(x + " ");
            Console.WriteLine();

            // Delegatske funkcije se mogu definisati i "inline". Ovo je
            // mnogo cesca praksa nego definisanje statickih 
            // funkcija za tu svrhu.
            IntComparator comp = delegate (int x, int y)
            {
                return x - y;
            };

            // Lambda izrazi: predstavljaju "kompaktniju" sintaksu
            // za definiciju delegatskih funkcija.
            IntComparator comp2 = (int x, int y) =>
            {
                return x - y;
            };

            // Tipovi parametara u lambda izrazima se mogu
            // i izostaviti.
            IntComparator comp3 = (x, y) =>
            {
                return x - y;
            };

            // Najzad, ukoliko se telo delegatske funkcije sastoji samo
            // iz jedne return naredbe, onda se sintaksa moze
            // jos dodatno pojednostaviti.
            IntComparator comp4 = (x, y) => x - y;

            // Poziv funkcije za sortiranje.
            sortArray(array, comp);

            foreach (int x in array)
                Console.Write(x + " ");
            Console.WriteLine();

            // Moze i ovako: u slucaju da je sintaksa definicije
            // delegatske funkcije maksimalno svedena, moze se 
            // ubaciti "na lice mesta", bez da se deklarise 
            // delegatska promenljiva. U vecini slucajeva cemo 
            // upravo tako i raditi.
            sortArray(array, (x, y) => x - y);
            foreach (int x in array)
                Console.Write(x + " ");
            Console.WriteLine();

            // Ako je u pitanju delegat sa samo jednim parametrom, tada
            // se definicija delegatske funkcije moze jos pojednostaviti,
            // jer zagrade oko jednoclane liste parametara nisu neophodne.
            int[] retArray = getElements(array, x => x % 2 == 0);

            foreach (int x in retArray)
                Console.Write(x + " ");
            Console.WriteLine();
        }
    }
}
