#include <iostream>
#include <set>
#include <map>
#include <string>
#include <sstream>
#include <vector>
#include <algorithm>
#include <memory>
#include <functional>
#include <list>

using namespace std;

/* Funkcijski i predikatski simboli */
typedef string FunctionSymbol;
typedef string PredicateSymbol;

/* Signatura se sastoji iz funkcijskih i predikatskih simbola kojima
   su pridruzene arnosti (nenegativni celi brojevi) */
class Signature {
private:
  map<FunctionSymbol,  unsigned> _functions;
  map<PredicateSymbol, unsigned> _predicates;
public:
  
  /* Dodavanje funkcijskog simbola date arnosti */
  void addFunctionSymbol(const FunctionSymbol & f, unsigned arity);

  /* Dodavanje predikatskog simbola date arnosti */
  void addPredicateSymbol(const PredicateSymbol & p, unsigned arity);

  /* Provera da li postoji dati funkcijski simbol, i koja mu je arnost */
  bool checkFunctionSymbol(const FunctionSymbol & f, unsigned & arity) const;

  /* Provera da li postoji dati predikatski simbol, i koja mu je arnost */
  bool checkPredicateSymbol(const PredicateSymbol & f, unsigned & arity) const;
  
};


/* Tip podatka za predstavljanje varijable */
typedef string Variable;

/* Skup varijabli */
typedef set<Variable> VariableSet;

class BaseTerm;
typedef std::shared_ptr<BaseTerm> Term;

/* Uopstena supstitucija */
typedef vector< pair<Variable, Term> > Substitution;

/* Funkcija prikazuje uopstenu supstituciju u preglednom obliku */
ostream & operator << (ostream & ostr, const Substitution & sub);

/* Apstraktna klasa BaseTerm koja predstavlja termove */
class BaseTerm : public enable_shared_from_this<BaseTerm> {

public:
  /* Termovi mogu biti ili varijable ili funkcijski simboli primenjeni
     na (0 ili vise) termova */
  enum Type { TT_VARIABLE, TT_FUNCTION };

  /* Vraca tip terma */
  virtual Type getType() const = 0;

  /* Prikazuje term */
  virtual void printTerm(ostream & ostr) const = 0;

  /* Ispituje sintaksnu jednakost termova */
  virtual bool equalTo(const Term & t) const = 0;
  
  /* Vraca skup svih varijabli koje se pojavljuju u termu */
  virtual void getVars(VariableSet & vars) const = 0;
 
  /* Odredjuje da li se data varijabla nalazi u termu */
  bool containsVariable(const Variable & v) const;

  /* Uopstena zamena */
  virtual Term substitute(const Substitution & sub) = 0;

  /* Zamena varijable v termom t */
  Term substitute(const Variable & v, const Term & t);

  virtual ~BaseTerm() {}
};

ostream & operator << (ostream & ostr, const Term & t);

/* Term koji predstavlja jednu varijablu */
class VariableTerm : public BaseTerm {
private:
  Variable _v;
public:
  VariableTerm(const Variable & v);
  virtual Type getType() const;
  const Variable & getVariable() const;
  virtual void printTerm(ostream & ostr) const;
  virtual bool equalTo(const Term & t) const;
  virtual void getVars(VariableSet & vars) const; 
  virtual Term substitute(const Substitution & sub);
};

/* Term koji predstavlja funkcijski simbol primenjen na odgovarajuci
   broj podtermova */
class FunctionTerm : public BaseTerm {
private:
  const Signature & _sig;
  FunctionSymbol _f;
  vector<Term> _ops;

public:
  FunctionTerm(const Signature & s, const FunctionSymbol & f, 
	       const vector<Term> & ops);
  FunctionTerm(const Signature & s, const FunctionSymbol & f, 
	       vector<Term> && ops = vector<Term> ());

  virtual Type getType() const;
  const Signature & getSignature() const;
  const FunctionSymbol & getSymbol() const;
  const vector<Term> & getOperands() const;
  virtual void printTerm(ostream & ostr) const;
  virtual bool equalTo(const Term & t) const;
  virtual void getVars(VariableSet & vars) const;
  virtual Term substitute(const Substitution & sub); 
};


class BaseFormula;
typedef std::shared_ptr<BaseFormula> Formula;

/* Apstraktna klasa kojom se predstavljaju formule */
class BaseFormula : public enable_shared_from_this<BaseFormula> {
  
public:

  /* Tipovi formula (dodatak u odnosu na iskaznu logiku su formule 
     kod kojih je vodeci simbol univerzalni ili egzistencijalni
     kvantifikator */
  enum Type { T_TRUE, T_FALSE, T_ATOM, T_NOT, 
	      T_AND, T_OR, T_IMP, T_IFF, T_FORALL, T_EXISTS };
  
  /* Prikaz formule */ 
  virtual void printFormula(ostream & ostr) const = 0;
  
  /* Tip formule */
  virtual Type getType() const = 0;
  
  /* Slozenost formule */
  virtual unsigned complexity() const = 0;

  /* Sintaksna jednakost dve formule */
  virtual bool equalTo(const Formula & f) const = 0;

  /* Ocitava sve varijable koje se pojavljuju u formuli. Ako
     je zadat drugi parametar sa vrednoscu true, tada se izdvajaju samo
     slobodne varijable u formuli */
  virtual void getVars(VariableSet & vars, bool free = false) const = 0;

  /* Ispituje da li se varijabla pojavljuje u formuli (kao slobodna ili
     vezana) */
  bool containsVariable(const Variable & v, bool free = false) const;

  /* Uopstena zamena */
  virtual Formula substitute(const Substitution & sub) = 0;

  Formula substitute(const Variable & v, const Term & t);

  virtual ~BaseFormula() {}
};

ostream & operator << (ostream & ostr, const Formula & f);

/* Klasa predstavlja sve atomicke formule (True, False i Atom) */
class AtomicFormula : public BaseFormula {

public:
  virtual unsigned complexity() const;
};

/* Klasa predstavlja logicke konstante (True i False) */
class LogicConstant : public AtomicFormula {

public:
  virtual bool equalTo(const Formula & f) const;
  virtual void getVars(VariableSet & vars, bool free) const;
  virtual Formula substitute(const Substitution & sub);
};

/* Klasa predstavlja True logicku konstantu */
class True : public LogicConstant {

public:
  virtual void printFormula(ostream & ostr) const;
  virtual Type getType() const;
};

/* Klasa predstavlja logicku konstantu False */
class False : public LogicConstant {

public:
  virtual void printFormula(ostream & ostr) const;
  virtual Type getType() const; 
};

/* Klasa predstavlja atom, koji za razliku od iskazne logike ovde ima
   znatno slozeniju strukturu. Svaki atom je predikatski simbol primenjen
   na odgovarajuci broj podtermova */
class Atom : public AtomicFormula {
private:
  const Signature & _sig;
  PredicateSymbol _p;
  vector<Term> _ops;

public:
  Atom(const Signature & s, 
       const PredicateSymbol & p, 
       const vector<Term> & ops);
  Atom(const Signature & s, 
       const PredicateSymbol & p, 
       vector<Term> && ops = vector<Term>());

  const PredicateSymbol & getSymbol() const;
  const Signature & getSignature() const;
  const vector<Term> & getOperands() const;
  virtual void printFormula(ostream & ostr) const;
  virtual Type getType() const; 
  virtual bool equalTo(const Formula & f) const;
 
  virtual void getVars(VariableSet & vars, bool free) const;
  virtual Formula substitute(const Substitution & sub);
};


/* Klasa unarni veznik (obuhvata negaciju) */
class UnaryConjective : public BaseFormula {
protected:
  Formula _op;
public:
  UnaryConjective(const Formula & op);
  const Formula & getOperand() const;
  virtual unsigned complexity() const;
  virtual bool equalTo(const Formula & f) const;
  virtual void getVars(VariableSet & vars, bool free) const;
};

/* Klasa koja predstavlja negaciju */
class Not : public UnaryConjective {
public:

  using UnaryConjective::UnaryConjective;
  virtual void printFormula(ostream & ostr) const; 
  virtual Type getType() const;
  virtual Formula substitute(const Substitution & sub);
};

/* Klasa predstavlja sve binarne veznike */
class BinaryConjective : public BaseFormula {
protected:
   Formula _op1, _op2;
public:
  BinaryConjective(const Formula & op1, const Formula & op2);
  const Formula & getOperand1() const;
  const Formula & getOperand2() const;
  virtual unsigned complexity() const;
  virtual bool equalTo(const Formula & f) const;
  virtual void getVars(VariableSet & vars, bool free) const;
};

/* Klasa predstavlja konjunkciju */
class And : public BinaryConjective {
public:
  using BinaryConjective::BinaryConjective;
  virtual void printFormula(ostream & ostr) const;
  virtual Type getType() const; 
  virtual Formula substitute(const Substitution & sub); 
 };


/* Klasa predstavlja disjunkciju */
class Or : public BinaryConjective {
public:
  using BinaryConjective::BinaryConjective;
  virtual void printFormula(ostream & ostr) const;
  virtual Type getType() const;
  virtual Formula substitute(const Substitution & sub);
};

/* Klasa predstavlja implikaciju */
class Imp : public BinaryConjective {
public:
  using BinaryConjective::BinaryConjective;
  virtual void printFormula(ostream & ostr) const;
  virtual Type getType() const;
  virtual Formula substitute(const Substitution & sub);
};


/* Klasa predstavlja ekvivalenciju */
class Iff : public BinaryConjective {

public:
  using BinaryConjective::BinaryConjective;
  virtual void printFormula(ostream & ostr) const; 
  virtual Type getType() const;
  virtual Formula substitute(const Substitution & sub);
};

/* Klasa predstavlja kvantifikovane formule */
class Quantifier : public BaseFormula {
protected:
  Variable _v;
  Formula _op;

public:
  Quantifier(const Variable & v, const Formula & op);
  const Variable & getVariable() const;
  const Formula & getOperand() const;
  virtual unsigned complexity() const;
  virtual bool equalTo(const Formula & f) const;
  virtual void getVars(VariableSet & vars, bool free) const;
};


/* Klasa predstavlja univerzalno kvantifikovanu formulu */
class Forall : public Quantifier {
public:
  using Quantifier::Quantifier;
  virtual Type getType() const;
  virtual void printFormula(ostream & ostr) const;
  virtual Formula substitute(const Substitution & sub);
};


/* Klasa predstavlja egzistencijalnog kvantifikatora */
class Exists : public Quantifier {
public:
  using Quantifier::Quantifier;
  virtual Type getType() const;
  virtual void printFormula(ostream & ostr) const;
  virtual Formula substitute(const Substitution & sub);
};

/* Funkcija vraca varijablu koja se ne pojavljuje ni u f ni u jednom
   od termova iz ts */
Variable getUniqueVariable(const Formula & f,  const vector<Term> & ts);

/* Funkcije za unifikaciju */

typedef list< pair<Term, Term> > TermPairs;
typedef list< pair<Formula, Formula> > AtomPairs;

/* Prikazuje na izlazu skup parova termova koje zelimo 
   da unifikujemo */
ostream & operator << (ostream & ostr, const TermPairs & pairs);



/* Ispituje da li je skup parova termova unifikabilan, i vraca
   najopstiji unifikator ako jeste */
bool unify(const TermPairs & pairs, Substitution & sub);

/* Ispituje da li je skup parova atoma unifikabilan, i vraca
   najopstiji unifikator ako jeste */
bool unify(const AtomPairs & pairs, Substitution & sub);

/* Specijalni slucaj -- dva terma */
bool unify(const Term & t1, const Term & t2, Substitution & sub);

/* Specijalni slucaj -- dva atoma */
bool unify(const Formula & a1, const Formula & a2, Substitution & sub);

/* Pomocna funkcije za unifikaciju -- transformise skup parova
   termova primenjujuci pravila iz algoritma. */
bool do_unify(TermPairs & pairs);

/* Primenjuje pravilo factoring */
void applyFactoring(TermPairs & pairs);

/* Primenjuje pravilo tautology */
void applyTautology(TermPairs & pairs);

/* Primenjuje pravilo orientation i vraca true ako je tim pravilom nesto
   promenjeno */
bool applyOrientation(TermPairs & pairs);

/* Primenjuje pravilo decomposition/collision i vraca true ako je tim
   pravilom nesto promenjeno */
bool applyDecomposition(TermPairs & pairs, bool & collision);

/* Primenjuje pravilo application/cycle i vraca true ako je tim pravilom 
   nesto promenjeno. */
bool applyApplication(TermPairs & pairs, bool & cycle);


/* Funkcije za rezoluciju */

/* Klazu je predstavljena kao niz literala */
typedef vector<Formula> Clause;

/* CNF formula je niz klauza */
typedef vector<Clause> CNF;

/* Prikazuje klauzu */
ostream & operator << (ostream & ostr, const Clause & c);

/* Prikazuje CNF formulu */
ostream & operator << (ostream & ostr, const CNF & cnf);

/* Vraca skup varijabli koje se pojavljuju u klauzi */
void getClauseVars(const Clause & c, VariableSet & vars);

/* Ispituje da li klauza sadrzi varijablu */
bool clauseContainsVariable(const Clause & c, const Variable & v);

/* Funkcija vraca novu jedinstvenu varijablu koja ne postoji ni u c1 ni u c2 */
Variable getUniqueVariable(const Clause & c1, const Clause & c2);

/* Ispituje da li klauza sadrzi literal */
bool clauseContainsLiteral(const Clause & c, const Formula & f);

/* Ispituje da li je klauza c1 sadrzana u klauzi c2 */
bool clauseSubsumed(const Clause & c1, const Clause & c2);

/* Vraca suprotni literal datog literala */
Formula oppositeLiteral(const Formula & l);

/* Ispituje da li je klauza c tautologija (tj. da li sadzi neki atom
   A i njegovu negaciju ~A) */
bool clauseTautology(const Clause & c);

/* Proverava da li u skupu postoji klauza koja je podskup date
   klauza c */
bool clauseExists(const CNF & cnf, const Clause & c);

/* Vrsi uopstenu zamenu u svim literalima klauze */
void substituteClause(Clause & c, const Substitution & sub);

/* Vrsi zamenu varijable v termom t u svim literalima klauze */
void substituteClause(Clause & c, const Variable & v, const Term & t);

/* Brise k-ti literal iz klauze */
void removeLiteralFromClause(Clause & c, unsigned k);

/* Nadovezuje dve klauze */
void concatClauses(const Clause & c1, const Clause & c2, Clause & cr);

/* Primenjuje pravilo rezolucije nad klauzama c1 i c2 po literalima 
   na pozicijama i, ondosno j, primenjujuci substituciju sub */
void resolveClauses(const Clause & c1, const Clause & c2, 
		    unsigned i, unsigned j, 
		    const Substitution & sub, Clause & res);

/* Ispituje da li je moguce primeniti pravilo rezolucije nad klauzama k i l
   i ako je moguce, primenjuje ga na sve moguce nacine i dodaje dobijene klauze
   u CNF */
bool tryResolveClauses(CNF & cnf, unsigned k, unsigned l);

/* Ispituje da li je moguce naci dve klauze nad kojima se moze primeniti
   pravilo rezolucije, i primenjuje ga ako je moguce, vracajuci rezolventnu
   klauzu res. Parametri i i j su tu zbog optimizacije, da bismo znali dokle
   smo stigli u pretrazi, tj. da je ne bismo pokretali svaki put iz pocetka. */
bool resolventFound(CNF & cnf, unsigned & i, unsigned & j);

/* Primenjuje pravilo grupisanja: literal na poziciji j se uklanja, 
   a na ostale se primenjuje zamena sub */
void groupLiterals(const Clause & c, unsigned j, const Substitution & sub, Clause & res);

/* Pokusava da primeni pravilo grupisanja na k-tu klauzu u formuli. Ukoliko uspe da na 
   taj nacin dobije novu klauzu, vraca true i u CNF dodaje rezultujucu
   klauzu */
bool tryGroupLiterals(CNF & cnf, unsigned k);

/* Za sve klauze redom ispituje da li moze primeniti pravilo grupisanja
   na neki nacin. Parametar k se prenosi po referenci zbog efikasnosti
   (da ne bismo svaki put iz pocetka pokretali postupak) */
bool groupingFound(CNF & cnf, unsigned & k);


/* Metod rezolucije */
bool resolution(const CNF & cnf);


// Definicije funkcija clanica -----------------------------------------

// Funkcije substitucije -----------------------------------------------

Term BaseTerm::substitute(const Variable & v, const Term & t)
{
  return substitute({{v, t}});
}

Formula  BaseFormula::substitute(const Variable & v, const Term & t)
{
  return substitute({{v, t}});
}

Term VariableTerm::substitute(const Substitution & sub) 
{
  for(unsigned i = 0; i < sub.size(); i++)
    if(_v == sub[i].first)
      return sub[i].second;
  
    return shared_from_this();
}

Term FunctionTerm::substitute(const Substitution & sub) 
{
  vector<Term> sub_ops;
  
  for(unsigned i = 0; i < _ops.size(); i++)
    sub_ops.push_back(_ops[i]->substitute(sub));
  
  return make_shared<FunctionTerm>(_sig, _f, sub_ops);
}

Formula LogicConstant::substitute(const Substitution & sub) 
{
  return shared_from_this();
}

Formula Atom::substitute(const Substitution & sub) 
{
  vector<Term> sub_ops;
  
  for(unsigned i = 0; i < _ops.size(); i++)
    sub_ops.push_back(_ops[i]->substitute(sub));
  
  return make_shared<Atom>(_sig, _p, sub_ops); 
}

Formula Not::substitute(const Substitution & sub) 
{
  return make_shared<Not>(_op->substitute(sub));
}

Formula And::substitute(const Substitution & sub) 
{
  return make_shared<And>(_op1->substitute(sub), _op2->substitute(sub));
}

Formula Or::substitute(const Substitution & sub) 
{
  return make_shared<Or>(_op1->substitute(sub), _op2->substitute(sub));
}

Formula Imp::substitute(const Substitution & sub) 
{
  return make_shared<Imp>(_op1->substitute(sub), _op2->substitute(sub));
}

Formula Iff::substitute(const Substitution & sub) 
{
  return make_shared<Iff>(_op1->substitute(sub), _op2->substitute(sub));
}


Formula Forall::substitute(const Substitution & sub)
{
  Substitution subnv;
  
  for(unsigned i = 0; i < sub.size(); i++)
    if(sub[i].first != _v)
      subnv.push_back(sub[i]);

  if(subnv.size() == 0)
    return shared_from_this();

  /* Proveravamo da li se varijabla _v nalazi u bar nekom od termova */
  bool contained = false;
  for(unsigned i = 0; !contained || i < subnv.size(); i++)
    contained = subnv[i].second->containsVariable(_v);

  /* Ako neki od termova sadrzi kvantifikovanu varijablu, tada moramo najpre
     preimenovati kvantifikovanu varijablu (nekom varijablom koja
     nije sadzana ni u termovima ni u formuli sto nam daje funkcija
     getUniqueVariable) */
    if(contained)
      {
	vector<Term> ts;
	for(unsigned i = 0; i < subnv.size(); i++)
	  ts.push_back(subnv[i].second);
	
	Variable new_v = getUniqueVariable(shared_from_this(), ts);
	
	Formula sub_op = _op->substitute(_v, make_shared<VariableTerm>(new_v));
	return make_shared<Forall>(new_v, sub_op->substitute(subnv));
      }
    else
      return make_shared<Forall>(_v, _op->substitute(subnv)); 
}

Formula Exists::substitute(const Substitution & sub)
{
  Substitution subnv;
  
  for(unsigned i = 0; i < sub.size(); i++)
    if(sub[i].first != _v)
      subnv.push_back(sub[i]);
  
  if(subnv.size() == 0)
    return shared_from_this();
  
  /* Proveravamo da li se varijabla _v nalazi u bar nekom od termova */
  bool contained = false;
  for(unsigned i = 0; !contained || i < subnv.size(); i++)
    contained = subnv[i].second->containsVariable(_v);
  
  /* Ako neki od termova sadrzi kvantifikovanu varijablu, tada moramo najpre
     preimenovati kvantifikovanu varijablu (nekom varijablom koja
     nije sadzana ni u termovima ni u formuli sto nam daje funkcija
     getUniqueVariable) */
  if(contained)
    {
      vector<Term> ts;
      for(unsigned i = 0; i < subnv.size(); i++)
	ts.push_back(subnv[i].second);
      
      Variable new_v = getUniqueVariable(shared_from_this(), ts);
      
      Formula sub_op = _op->substitute(_v, make_shared<VariableTerm>(new_v));
      return make_shared<Exists>(new_v, sub_op->substitute(subnv));
    }
  else
    return make_shared<Exists>(_v, _op->substitute(subnv)); 
}

// ---------------------------------------------------------------------

// Funkcije za odredjivanje sintaksne identicnosti termova i formula ---

bool VariableTerm::equalTo(const Term & t) const
{
  return t->getType() == TT_VARIABLE && 
    ((VariableTerm *) t.get())->getVariable() == _v;
}

bool FunctionTerm::equalTo(const Term & t) const
{
  if(t->getType() != TT_FUNCTION)
    return false;
  
  if(_f != ((FunctionTerm *) t.get())->getSymbol())
    return false;
  
  const vector<Term> & t_ops = ((FunctionTerm *) t.get())->getOperands();
  
  if(_ops.size() != t_ops.size())
    return false;
  
  for(unsigned i = 0; i < _ops.size(); i++)
    if(!_ops[i]->equalTo(t_ops[i]))
      return false;
  
  return true;
}

bool LogicConstant::equalTo( const Formula & f) const
{
  return f->getType() == this->getType();
}


bool Atom::equalTo(const Formula & f) const
{
  if(f->getType() != T_ATOM)
    return false;
  
  if(_p != ((Atom *) f.get())->getSymbol())
    return false;
  
  const vector<Term> & f_ops = ((Atom *) f.get())->getOperands();
  
  if(_ops.size() != f_ops.size())
    return false;
  
  for(unsigned i = 0; i < _ops.size(); i++)
    if(!_ops[i]->equalTo(f_ops[i]))
      return false;
  
    return true;
}

bool UnaryConjective::equalTo(const Formula & f) const
{
  return f->getType() == this->getType() && 
    _op->equalTo(((UnaryConjective *)f.get())->getOperand());
}

bool BinaryConjective::equalTo( const Formula & f) const
{
  return f->getType() == this->getType() && 
    _op1->equalTo(((BinaryConjective *)f.get())->getOperand1()) 
    &&  
    _op2->equalTo(((BinaryConjective *)f.get())->getOperand2());
}

bool Quantifier::equalTo(const Formula & f) const
{
  return f->getType() == getType() &&
    ((Quantifier *) f.get())->getVariable() == _v && 
    ((Quantifier *) f.get())->getOperand()->equalTo(_op);
}

// ---------------------------------------------------------------------

// Funkcije za odredjivanje skupa varijabli ----------------------------

void VariableTerm::getVars(VariableSet & vars) const
{
  vars.insert(_v);
}

void FunctionTerm::getVars(VariableSet & vars) const
{
  for(unsigned i = 0; i < _ops.size(); i++)
    _ops[i]->getVars(vars);
}

void LogicConstant::getVars(VariableSet & vars, bool free) const
{
  return;
}

void Atom::getVars(VariableSet & vars, bool free) const
{
  for(unsigned i = 0; i < _ops.size(); i++)
    {
      _ops[i]->getVars(vars);
    }
}

void UnaryConjective::getVars(VariableSet & vars, bool free) const
{
  _op->getVars(vars, free);
}

void BinaryConjective::getVars(VariableSet & vars, bool free) const
{
  _op1->getVars(vars, free);
  _op2->getVars(vars, free);
}

void Quantifier::getVars(VariableSet & vars, bool free) const
{
  bool present = false;

  if(free)
    {
      /* Pamtimo da li je kvantifikovana varijabla vec u skupu slobodnih
	 varijabli */
      if(vars.find(_v) != vars.end())
	present = true;
    }
  
  _op->getVars(vars, free);
  if(!free)
    vars.insert(_v);
  
  if(free)
    {
      /* Ako varijabla ranije nije bila prisutna u skupu slobodnih varijabli,
	 tada je brisemo, zato sto to znaci da se ona pojavljuje samo u 
	 podformuli kvantifikovane formule,a u njoj je vezana kvantifikatorom */
      if(!present && vars.find(_v) != vars.end())
	vars.erase(_v);
    }
}

// ---------------------------------------------------------------------

// Funkcije za odredjivanje slozenosti formule -------------------------

unsigned AtomicFormula::complexity() const
{
  return 0;
}  

unsigned UnaryConjective::complexity() const
{
  return _op->complexity() + 1;
}

unsigned BinaryConjective::complexity() const
{
  return _op1->complexity() + _op2->complexity() + 1;
}

unsigned Quantifier::complexity() const
{
  return _op->complexity() + 1;
}

// ---------------------------------------------------------------------

// Funkcije za stampanje -----------------------------------------------


void VariableTerm::printTerm(ostream & ostr) const
{
  ostr << _v;
}

void FunctionTerm::printTerm(ostream & ostr) const
{
  ostr << _f;

  for(unsigned i = 0; i < _ops.size(); i++)
    {
      if(i == 0)
	ostr << "(";
      ostr << _ops[i];
      if(i != _ops.size() - 1)
	ostr << ",";
      else
	ostr << ")";
    }
}

void True::printFormula(ostream & ostr) const
{
  ostr << "True";
}

void False::printFormula(ostream & ostr) const
{
  ostr << "False";
}

void Atom::printFormula(ostream & ostr) const
{
  ostr << _p;
  for(unsigned i = 0; i < _ops.size(); i++)
    {
      if(i == 0)
	ostr << "(";
      ostr << _ops[i];
      if(i != _ops.size() - 1)
	ostr << ",";
      else
	ostr << ")";
    }
}

void Not::printFormula(ostream & ostr) const
{
  ostr << "(~" << _op << ")";
}

void And::printFormula(ostream & ostr) const
{
  ostr << "(" << _op1 << " /\\ " << _op2 << ")";
}

void Or::printFormula(ostream & ostr) const
{
  ostr << "(" << _op1 << " \\/ " << _op2 << ")";
}

void Imp::printFormula(ostream & ostr) const
{
  ostr << "(" << _op1 << " ==> " << _op2 << ")";
}

void Iff::printFormula(ostream & ostr) const
{
  ostr << "(" << _op1 << " <=> " << _op2 << ")";
}

void Forall::printFormula(ostream & ostr) const
{
  ostr << "(A " << _v << ").(" << _op << ")";
}

void Exists::printFormula(ostream & ostr) const
{
  ostr << "(E " << _v << ").(" << _op << ")";
}


ostream & operator << (ostream & ostr, const Term & t)
{
  t->printTerm(ostr);
  return ostr;
}

ostream & operator << (ostream & ostr, const Formula & f)
{
  f->printFormula(ostr);
  return ostr;
}

// ---------------------------------------------------------------------

ostream & operator << (ostream & ostr, const Substitution & sub)
{
  ostr << "[ ";
  for(unsigned i = 0; i < sub.size(); i++)
    {     
      ostr << sub[i].first << " ---> " << sub[i].second;
      if(i < sub.size() - 1)
	ostr << ", ";      
    }
  ostr << " ]";

  return ostr;
}

// Funkcije za unifikaciju -----------------------------------------------

ostream & operator << (ostream & ostr, const TermPairs & pairs)
{
  for(const auto & p : pairs)
    {
      ostr << "(" << p.first << ", " << p.second << ") ";      
    }
  return ostr;
}



bool unify(const AtomPairs & pairs, Substitution & sub)
{
  TermPairs term_pairs;

  for(auto i = pairs.begin(); i != pairs.end(); i++)
    {
      if(((Atom*)(*i).first.get())->getSymbol() == ((Atom*)(*i).second.get())->getSymbol())
	{
	  const vector<Term> & ops_f = ((Atom*)(*i).first.get())->getOperands();
	  const vector<Term> & ops_s = ((Atom*)(*i).second.get())->getOperands();

	  for(unsigned k = 0; k < ops_f.size(); k++)
	    term_pairs.push_back(make_pair(ops_f[k], ops_s[k]));
	}
      else
	return false;
    }
  return unify(term_pairs, sub);
}

bool unify(const TermPairs & pairs, Substitution & sub)
{
  TermPairs res_pairs = pairs;

  if(!do_unify(res_pairs))
    return false;

  for(auto i = res_pairs.cbegin(); i != res_pairs.cend(); i++)
    {
      VariableTerm * vt = (VariableTerm *) (*i).first.get();
      sub.push_back(make_pair(vt->getVariable(), (*i).second));
    }
  
  return true;
}

bool unify(const Term & t1, const Term & t2, Substitution & sub)
{
  return unify({{t1, t2}}, sub);
}

bool unify(const Formula & a1, const Formula & a2, Substitution & sub)
{
  return unify({{a1, a2}}, sub);
}


bool do_unify(TermPairs & pairs)
{
  bool repeat =  false;
  bool collision = false;
  bool cycle = false;

  do {
    
    applyFactoring(pairs);    
    applyTautology(pairs);

    repeat = 
      applyOrientation(pairs) ||
      applyDecomposition(pairs, collision) ||
      applyApplication(pairs, cycle);
    
    if(collision || cycle)
      return false;
    
  } while(repeat);
  
  return true;
}


void applyFactoring(TermPairs & pairs)
{
  for(auto i = pairs.begin(); i != pairs.end(); i++)
    {
      auto j = i;
      j++;
      while(j != pairs.end())
	{
	  if((*j).first->equalTo((*i).first) &&
	     (*j).second->equalTo((*i).second))
	    {
	      cout << "Factoring applied: (" << (*j).first << "," << (*j).second << ")" << endl;
	      auto erase_it = j;
	      j++;
	      pairs.erase(erase_it);
	      cout << pairs << endl;
	    }
	  else
	    j++;
	}
    }
}

void applyTautology(TermPairs & pairs)
{
  auto i = pairs.begin();
  
  while(i != pairs.end())
    {
      if((*i).first->equalTo((*i).second))
	{
	  cout << "Tautology applied: (" << (*i).first << "," << (*i).second << ")" << endl;
	  auto erase_it = i;
	  i++;
	  pairs.erase(erase_it);
	  cout << pairs << endl;
	}
      else
	i++;
    }
}

bool applyOrientation(TermPairs & pairs)
{
  bool ret = false;

  for(auto i = pairs.begin(); i != pairs.end(); i++)
    {
      if((*i).first->getType() != BaseTerm::TT_VARIABLE &&
	 (*i).second->getType() == BaseTerm::TT_VARIABLE)
	{
	  cout << "Orientation applied: (" << (*i).first << "," << (*i).second << ")" << endl;
	  swap((*i).first, (*i).second);
	  ret = true;
	  cout << pairs << endl;
	}
    }

  return ret;
}

bool applyDecomposition(TermPairs & pairs, bool & collision)
{
  bool ret = false;
  
  auto i = pairs.begin();
  while(i != pairs.end())
    {
      if((*i).first->getType() == BaseTerm::TT_FUNCTION &&
	 (*i).second->getType() == BaseTerm::TT_FUNCTION)
	{
	  FunctionTerm * ff = (FunctionTerm *) (*i).first.get();
	  FunctionTerm * fs = (FunctionTerm *) (*i).second.get();

	  if(ff->getSymbol() == fs->getSymbol())
	    {
	      const vector<Term> & ff_ops = ff->getOperands();
	      const vector<Term> & fs_ops = fs->getOperands();

	      for(unsigned k = 0; k < ff_ops.size(); k++)
		{
		  pairs.push_back(make_pair(ff_ops[k], fs_ops[k]));
		}

	      cout << "Decomposition applied: (" << (*i).first << "," << (*i).second << ")" << endl;
	      auto erase_it = i;
	      i++;
	      pairs.erase(erase_it);
	      cout << pairs << endl;
	      ret = true;
	    }
	  else
	    {
	      cout << "Collision detected: " << ff->getSymbol() << " != " << fs->getSymbol() << endl;
	      collision = true;
	      return true;
	    }
	  
	}
      else
	i++;
    }

  collision = false;
  return ret;

}

bool applyApplication(TermPairs & pairs, bool & cycle)
{
  bool ret = false;

  for(auto i = pairs.begin(); i != pairs.end(); i++)
    {
      if((*i).first->getType() == BaseTerm::TT_VARIABLE)
	{
	  VariableTerm * vt = (VariableTerm *) (*i).first.get();
	  if((*i).second->containsVariable(vt->getVariable()))
	    {
	      cycle = true;
	      cout << "Cycle detected: " << (*i).second <<  " contains " << vt->getVariable() << endl;
	      return true;
	    }
	  else
	    {
	      bool changed = false;
	      for(auto j = pairs.begin(); j != pairs.end(); j++)
		{
		  if(j != i)
		    {
		      if((*j).first->containsVariable(vt->getVariable()))
			{
			  (*j).first = (*j).first->
			    substitute(vt->getVariable(),
				       (*i).second);
			  ret = true;
			  changed = true;
			}
		      if((*j).second->containsVariable(vt->getVariable()))
			{
			  (*j).second = (*j).second->
			    substitute(vt->getVariable(),
				       (*i).second);
			  ret = true;
			  changed = true;
			}
		    }
		}
	      if(changed)
		{
		  cout << "Application applied: (" << (*i).first << "," << (*i).second << ")" << endl; 
		  cout << pairs << endl;
		}
	    }
	}
    }
  cycle = false;
  return ret;
}

// -----------------------------------------------------------------------

// Rezolucija ------------------------------------------------------------

ostream & operator << (ostream & ostr, const Clause & c)
{
  ostr << "{ ";
  for(unsigned i = 0; i < c.size(); i++)
    {
      ostr << c[i];
      if(i != c.size() - 1)
	ostr << ", ";
    }
  ostr << " }";
  return ostr;
}

ostream & operator << (ostream & ostr, const CNF & cnf)
{
  ostr << "{ ";
  for(unsigned i = 0; i < cnf.size(); i++)
    {
      ostr << cnf[i];
      if(i != cnf.size() - 1)
	ostr << ", ";
    }
  ostr << " }";
  return ostr;
}

void getClauseVars(const Clause & c, VariableSet & vars)
{
  for(unsigned i = 0; i < c.size(); i++)
    c[i]->getVars(vars);
}

bool clauseContainsVariable(const Clause & c, const Variable & v)
{
  VariableSet vars;
  getClauseVars(c, vars);
  return vars.find(v) != vars.end();
}

Variable getUniqueVariable(const Clause & c1, const Clause & c2)
{
  static unsigned i;
  
  Variable v;
  
  do {
    v = string("cv") + to_string(++i);
  } while(clauseContainsVariable(c1, v) || clauseContainsVariable(c2, v));
  
  return v;
}

bool clauseContainsLiteral(const Clause & c, const Formula & f)
{
  for(unsigned i = 0; i < c.size(); i++)
    if(f->equalTo(c[i]))
      return true;

  return false;
}



bool clauseSubsumed(const Clause & c1, const Clause & c2)
{
  for(unsigned i = 0; i < c1.size(); i++)
    {
      if(!clauseContainsLiteral(c2, c1[i]))
	return false;
    }
  return true;
}

 
Formula oppositeLiteral(const Formula & l)
{
  if(l->getType() == BaseFormula::T_NOT)
    return ((Not *)l.get())->getOperand();
  else
    return make_shared<Not>(l);
}

bool clauseTautology(const Clause & c)
{
  for(unsigned i = 0; i < c.size(); i++)
    if(clauseContainsLiteral(c, oppositeLiteral(c[i])))
      return true;

  return false;
}


bool clauseExists(const CNF & cnf, const Clause & c)
{
  for(unsigned i = 0; i < cnf.size(); i++)
    {
      if(clauseSubsumed(cnf[i], c))
	return true;
    }
  return false;
}

void substituteClause(Clause & c, const Substitution & sub)
{
  for(unsigned i = 0; i < c.size(); i++)
    c[i] = c[i]->substitute(sub);
}

void substituteClause(Clause & c, const Variable & v, const Term & t)
{
  Substitution sub;
  sub.push_back(make_pair(v, t));
  substituteClause(c, sub);
}

void removeLiteralFromClause(Clause & c, unsigned k)
{
  c[k] = c.back();
  c.pop_back();
}

void concatClauses(const Clause & c1, const Clause & c2, Clause & cr)
{
  cr = c1;
  for(unsigned i = 0; i < c2.size(); i++)
    cr.push_back(c2[i]);
}

void resolveClauses(const Clause & c1, const Clause & c2, 
		    unsigned i, unsigned j, 
		    const Substitution & sub, Clause & res)
{
  Clause c1p = c1;
  Clause c2p = c2;
  removeLiteralFromClause(c1p, i);
  removeLiteralFromClause(c2p, j);
  concatClauses(c1p, c2p, res);
  substituteClause(res, sub);
}

bool tryResolveClauses(CNF & cnf, unsigned k, unsigned l)	
{
  VariableSet vars;
  bool ret = false;
  Clause c1 = cnf[k], c2 = cnf[l];

  getClauseVars(c2, vars);

  for(auto it = vars.cbegin(); it != vars.cend(); it++)
    {
      if(clauseContainsVariable(c1, *it))
	{
	  Variable new_v = getUniqueVariable(c1, c2);
	  substituteClause(c2, *it, make_shared<VariableTerm>(new_v));
	}
    }

  for(unsigned i = 0; i < c1.size(); i++)
    for(unsigned j = 0; j < c2.size(); j++)
      {
	if(c1[i]->getType() == BaseFormula::T_ATOM &&
	   c2[j]->getType() == BaseFormula::T_NOT)
	  {
	    Formula c1a =  c1[i];
	    Formula c2n = c2[j];
	    Formula c2a = ((Not*)c2n.get())->getOperand();
	    
	    Substitution sub;
	    if(unify(c1a, c2a, sub))
	      {
		Clause r;
		resolveClauses(c1, c2, i, j, sub, r);
		if(!clauseExists(cnf, r) && !clauseTautology(r))
		  {
		    cnf.push_back(r);
		    cout << "Resolution applied: " << k << ", " << l << endl;
		    cout << "Parent clauses: " << c1 << ", " << c2 << endl;
		    cout << "Substitution: " << sub << endl;
		    cout << "Resolvent: " << r << endl;
		    cout << cnf << endl;
		    ret = true;
		  }
	      }
	  }
	else if(c1[i]->getType() == BaseFormula::T_NOT &&
		c2[j]->getType() == BaseFormula::T_ATOM)
	  {
	    Formula c2a =  c2[j];
	    Formula c1n =  c1[i];
	    Formula c1a =  ((Not*)c1n.get())->getOperand();
	    
	    Substitution sub;
	    if(unify(c1a, c2a, sub))
	      {
		Clause r;
		resolveClauses(c1, c2, i, j, sub, r);
		if(!clauseExists(cnf, r) && !clauseTautology(r))
		  {
		    cnf.push_back(r);
		    cout << "Resolution applied: " << k << ", " << l << endl;
		    cout << "Parent clauses: " << c1 << ", " << c2 << endl;
		    cout << "Substitution: " << sub << endl;
		    cout << "Resolvent: " << r << endl;
		    cout << cnf << endl;
		    ret = true;
		  }
	      }
	  }
      }  
  return ret;
}

bool resolventFound(CNF & cnf, unsigned & i, unsigned & j)
{
  bool ret = false;
  while(j < cnf.size())
    {
      if(tryResolveClauses(cnf, i, j))
	{	  
	  ret = true;
	}

      if(i < j - 1)
	i++;
      else
	{
	  i = 0; j++;
	}	       
    }
  return ret;
}

void groupLiterals(const Clause & c, unsigned j, const Substitution & sub, Clause & res)
{
  res = c;
  removeLiteralFromClause(res, j);
  substituteClause(res, sub);
}

bool tryGroupLiterals(CNF & cnf, unsigned k)
{
  const Clause c = cnf[k];
  bool ret = false;
  for(unsigned i = 0; i < c.size(); i++)
    for(unsigned j = i + 1; j < c.size(); j++)
      {
	Substitution sub;
	if(c[i]->getType() == BaseFormula::T_ATOM &&
	   c[j]->getType() == BaseFormula::T_ATOM)
	  {
	    if(unify(c[i], c[j], sub))
	      {
		Clause r;
		groupLiterals(c, j, sub, r);
		if(!clauseExists(cnf, r) && !clauseTautology(r))
		  {
		    cnf.push_back(r);
		    cout << "Grouping applied: " << k << endl;
		    cout << "Parent clause: " << c << endl;
		    cout << "Substitution: " << sub << endl;
		    cout << "Grouped clause: " << r << endl;
		    cout << cnf << endl;
		    ret = true;
		  }
	      }	   
	  }
	else if(c[i]->getType() == BaseFormula::T_NOT &&
		c[j]->getType() == BaseFormula::T_NOT)
	  {
	    Formula cia =  ((Not*)c[i].get())->getOperand();
	    Formula cja =  ((Not*)c[j].get())->getOperand();
	    if(unify(cia, cja, sub))
	      {
		Clause r;
		groupLiterals(c, j, sub, r);
		if(!clauseExists(cnf, r) && !clauseTautology(r))
		  {
		    cnf.push_back(r);
		    cout << "Grouping applied: " << k << endl;
		    cout << "Parent clause: " << c << endl;
		    cout << "Substitution: " << sub << endl;
		    cout << "Grouped clause: " << r << endl;
		    cout << cnf << endl;
		    ret = true;
		  }
	      }	   	    
	  }	
      }
  return ret;
}
  
bool groupingFound(CNF & cnf, unsigned & k)
{
  bool ret = false;
  while(k < cnf.size())
    {
      if(tryGroupLiterals(cnf, k))
	{	  
	  ret = true;
	}      
      k++;
    }
  return ret;
}


bool resolution(const CNF & cnf)
{
  CNF vcnf = cnf;
  unsigned i = 0, j = 1, k = 0;

  unsigned s = 0;
  while(groupingFound(vcnf, k) || resolventFound(vcnf, i, j))
    {
      for(; s < vcnf.size(); s++)	
	if(vcnf[s].size() == 0)
	  return false;
    }

  return true;
}

// Ostale funkcije clanice -----------------------------------------------

// Klasa Signature -------------------------------------------------------

void Signature::addFunctionSymbol(const FunctionSymbol & f, unsigned arity)
{
  _functions.insert(make_pair(f, arity));
}

void Signature::addPredicateSymbol(const PredicateSymbol & p, unsigned arity)
{
  _predicates.insert(make_pair(p, arity));
}

bool Signature::checkFunctionSymbol(const FunctionSymbol & f, 
				    unsigned & arity) const
{
  auto it = _functions.find(f);
  
  if(it != _functions.end())
    {
      arity = it->second;
      return true;
    }
  else
    return false;
}

bool Signature::checkPredicateSymbol(const PredicateSymbol & f, 
				     unsigned & arity) const
{
  auto it = _predicates.find(f);
  
  if(it != _predicates.end())
    {
      arity = it->second;
      return true;
    }
  else
    return false;
}

// -----------------------------------------------------------------------

// Klasa BaseTerm ------------------------------------------------------------

bool BaseTerm::containsVariable(const Variable & v) const
{
  VariableSet vars;
  getVars(vars);
  return vars.find(v) != vars.end();
}
// -----------------------------------------------------------------------

// Klasa VariableTerm ----------------------------------------------------

VariableTerm::VariableTerm(const Variable & v)
  :_v(v)
{}

BaseTerm::Type VariableTerm::getType() const
{
  return TT_VARIABLE;
}

const Variable & VariableTerm::getVariable() const
{
  return _v;
}

// -----------------------------------------------------------------------

// Klasa FunctionTerm ----------------------------------------------------

FunctionTerm::FunctionTerm(const Signature & s, const FunctionSymbol & f, 
			   const vector<Term> & ops)
  :_sig(s),
   _f(f),
   _ops(ops)
{
  unsigned arity;
  if(!_sig.checkFunctionSymbol(_f, arity) || arity != _ops.size())
    throw "Syntax error!";
}

FunctionTerm::FunctionTerm(const Signature & s, const FunctionSymbol & f, 
			   vector<Term> && ops)
  :_sig(s),
   _f(f),
   _ops(std::move(ops))
{
  unsigned arity;
  if(!_sig.checkFunctionSymbol(_f, arity) || arity != _ops.size())
    throw "Syntax error!";
}

BaseTerm::Type FunctionTerm::getType() const
{
  return TT_FUNCTION;
}


const Signature & FunctionTerm::getSignature() const
{
  return _sig;
}

const FunctionSymbol & FunctionTerm::getSymbol() const
{
  return _f;
}

const vector<Term> & FunctionTerm::getOperands() const
{
  return _ops;
}

// ----------------------------------------------------------------------

// Klasa BaseFormula --------------------------------------------------------

bool BaseFormula::containsVariable(const Variable & v, bool free) const
{
  VariableSet vars;
  getVars(vars, free);
  return vars.find(v) != vars.end();
}

// ----------------------------------------------------------------------

Variable getUniqueVariable(const Formula & f,  const vector<Term> & ts)
{
  static unsigned i = 0;
  
  Variable v;
  
  while(true) {
    v = string("uv") + to_string(++i);
    
    if(f->containsVariable(v))
      continue;
    
    unsigned j;
    for(j = 0; j < ts.size(); j++)
      {
	if(ts[j]->containsVariable(v))
	  break;
      }

    if(j == ts.size())
      break;
  }
  
  return v;
}

// Klasa AtomicFormula --------------------------------------------------

// Klasa LogicConstant --------------------------------------------------

BaseFormula::Type True::getType() const
{
  return T_TRUE;
}
// ----------------------------------------------------------------------

// Klasa False ----------------------------------------------------------

BaseFormula::Type False::getType() const
{
  return T_FALSE;
}

// ----------------------------------------------------------------------

// Klasa Atom -----------------------------------------------------------

Atom::Atom(const Signature & s, 
	   const PredicateSymbol & p, 
	   const vector<Term> & ops)
  :_sig(s),
   _p(p),
   _ops(ops)
{
  unsigned arity;
  if(!_sig.checkPredicateSymbol(_p, arity) || arity != _ops.size())
    throw "Syntax error!";
}

Atom::Atom(const Signature & s, 
	   const PredicateSymbol & p, 
	   vector<Term> && ops)
  :_sig(s),
   _p(p),
   _ops(std::move(ops))
{
  unsigned arity;
  if(!_sig.checkPredicateSymbol(_p, arity) || arity != _ops.size())
    throw "Syntax error!";
}

const PredicateSymbol & Atom::getSymbol() const
{
  return _p;
}

const Signature & Atom::getSignature() const
{
  return _sig;
}

const vector<Term> & Atom::getOperands() const
{
  return _ops;
}


BaseFormula::Type Atom::getType() const
{
  return T_ATOM;
}

// -----------------------------------------------------------------------

// Klasa UnaryConjective -------------------------------------------------

UnaryConjective::UnaryConjective(const Formula & op)
  :_op(op)
{}

const Formula & UnaryConjective::getOperand() const
{
  return _op;
}

// -----------------------------------------------------------------------

// Klasa Not -------------------------------------------------------------


BaseFormula::Type Not::getType() const
{
  return T_NOT;
}

// -----------------------------------------------------------------------

// Klasa BinaryConjective ------------------------------------------------

BinaryConjective::BinaryConjective( const Formula & op1,  const Formula & op2)
  :_op1(op1),
   _op2(op2)
{}

const Formula & BinaryConjective::getOperand1() const
{
  return _op1;
  }

const Formula & BinaryConjective::getOperand2() const
{
  return _op2;
}
  

// Klasa And ---------------------------------------------------------------

BaseFormula::Type And::getType() const
{
  return T_AND;
}

// -------------------------------------------------------------------------

// Klasa Or ----------------------------------------------------------------

BaseFormula::Type Or::getType() const
{
  return T_OR;
}

// -------------------------------------------------------------------------

// Klasa Imp ---------------------------------------------------------------

BaseFormula::Type Imp::getType() const
{
  return T_IMP;
}

// -------------------------------------------------------------------------

// Klasa Iff ---------------------------------------------------------------

BaseFormula::Type Iff::getType() const
{
  return T_IFF;
}

// -------------------------------------------------------------------------
  
// Klasa Quantifier --------------------------------------------------------

Quantifier::Quantifier(const Variable & v, const Formula & op)
  :_v(v),
   _op(op)
{}

const Variable & Quantifier::getVariable() const
{
  return _v;
}

const Formula & Quantifier::getOperand() const
{
  return _op;
}

// ------------------------------------------------------------------------

// Klasa Forall -----------------------------------------------------------

BaseFormula::Type Forall::getType() const
{
  return T_FORALL;
}

// ------------------------------------------------------------------------

// Klasa Exists -----------------------------------------------------------

BaseFormula::Type Exists::getType() const
{
  return T_EXISTS;
}

// -----------------------------------------------------------------------
// -----------------------------------------------------------------------

/* Test program */
int main()
{

  Signature s;

  s.addFunctionSymbol("c", 0);
  s.addFunctionSymbol("g", 1);
  s.addFunctionSymbol("f", 3);
  s.addFunctionSymbol("h", 1);
  s.addPredicateSymbol("p", 2);
  s.addPredicateSymbol("q", 2);
  


  Term xt = make_shared<VariableTerm>("x");
  Term ut = make_shared<VariableTerm>("u");
  Term vt = make_shared<VariableTerm>("v");
  Term zt = make_shared<VariableTerm>("z");
  Term ct = make_shared<FunctionTerm>(s, "c");
  
  /* g(x) */
  Term gt = make_shared<FunctionTerm>(s, "g", vector<Term> { xt });
  
  /* p(x, g(x)) */
  Formula px = make_shared<Atom>(s, "p", vector<Term> {xt, gt});
  
  /* p(u, v) */
  Formula puv = make_shared<Atom>(s, "p", vector<Term> {ut, vt});
  
  /* q(u, v) */
  Formula quv = make_shared<Atom>(s, "q", vector<Term> {ut, vt});

  /* q(c, z) */
  Formula qcz = make_shared<Atom>(s, "q", vector<Term> {ct, zt});
  
  /* h(x) */
  Term hx = make_shared<FunctionTerm>(s, "h", vector<Term> { xt });

  Term yt = make_shared<VariableTerm>("y");
  Term wt = make_shared<VariableTerm>("w");
  
  /* g(z) */
  Term gz = make_shared<FunctionTerm>(s, "g", vector<Term> { zt });

  /* g(y) */
  Term gy = make_shared<FunctionTerm>(s, "g", vector<Term> { yt });

  /* f(x, h(x), y) */
  Term f1 = make_shared<FunctionTerm>(s, "f", vector<Term> { xt, hx, yt });
 
  /* f(g(z), w, z) */
  Term f2 = make_shared<FunctionTerm>(s, "f", vector<Term> { gz, wt, zt });
  
  /* Parovi termova: (g(y), x) i (f(x, h(x), y), f(g(z), w, z)) */
  TermPairs pairs { { gy, xt }, { f1, f2 } };

  Substitution sub;

  /* Ako su unifikabilni, prikazujemo najopstiji unifikator,
     kao i rezultat primene tog unifikatora na termove */
  if(unify(pairs, sub))
    {
      cout << "Unifiable: " << sub << endl;
      cout << "Check: " << gy->substitute(sub) << " = " << xt->substitute(sub) << " , " << f1->substitute(sub) << " = " << f2->substitute(sub) << endl;
    }
  /* Ako nisu unifikabilni, prikazujemo poruku o tome */
  else
    {
      cout << "Not unifiable" << endl;
    }

  /* g(x0) */
  Term x0 = make_shared<VariableTerm>("x0");
  Term g1 = make_shared<FunctionTerm>(s, "g", vector<Term> { x0 });

  /* g(x1) */
  Term x1 = make_shared<VariableTerm>("x1");
  Term g2 = make_shared<FunctionTerm>(s, "g", vector<Term> { x1 });

  /* g(x2) */
  Term x2 = make_shared<VariableTerm>("x2");
  Term g3 = make_shared<FunctionTerm>(s, "g", vector<Term> { x2 });
  
  Term x3 = make_shared<VariableTerm>("x3");

  /* Parovi (x1, g(x0)), (x2, g(x1)), (x3, g(x2)) */
  TermPairs tp { { x1, g1 }, { x2, g2 }, { x3, g3 } };
  
  sub.clear();
  if(unify(tp, sub))
    {
      cout << "Unifiable" << endl;
      cout << sub << endl;
    }
  else
    {
      cout << "Not unifiable" << endl;
    }

  
  /* (g(h(x)), g(g(c))) */
  Term ghx = make_shared<FunctionTerm>(s, "g", vector<Term> { make_shared<FunctionTerm>(s, "h", vector<Term> { xt }) });
  Term ggc = make_shared<FunctionTerm>(s, "g", vector<Term> { make_shared<FunctionTerm>(s, "g", vector<Term> { ct }) });
  TermPairs tp2 { { ghx, ggc } };
  sub.clear();
  if(unify(tp2, sub))
    {
      cout << "Unifiable" << endl;
      cout << sub << endl;
    }
  else
    {
      cout << "Not unifiable" << endl;
    }

  /* Klauze: {p(x, g(x))}, {~p(u, v), q(u, v)}, {~q(c,z)} */

  /*
  CNF cnf(3);
  
  cnf[0].push_back(px);
  cnf[1].push_back(make_shared<Not>(puv));
  cnf[1].push_back(quv);
  cnf[2].push_back(make_shared<Not>(qcz));

  cout << cnf << endl;
  

  

  if(resolution(cnf))
    {
      cout << "CNF satisfiable!" << endl;
    }
  else
    {
      cout << "CNF unsatisfiable!" << endl;
    }
  
  */
  
  /*  Primer nepotpunosti binarne rezolucije (paradoks berberina): 

  (E x) (A y) (p(x, y) <=> ~p(y, y))
  (A y) (p(c, y) <=> ~p(y, y))
  (A y) (~p(c, y) \/ ~p(y, y)) /\ (p(y, y) \/ p(c, y)).
 
  Klauze: { ~p(c, x), ~p(x, x) }, { p(y,y), p(c, y) }

  Moguce je izvesti praznu klauzu samo uz pravilo grupisanja.
  */
  
  /*
  CNF cnf(2);

  Formula pcx = make_shared<Atom>(s, "p", vector<Term> { ct, xt });
  Formula pxx = make_shared<Atom>(s, "p", vector<Term> { xt, xt });
  Formula pcy = make_shared<Atom>(s, "p", vector<Term> { ct, yt });
  Formula pyy = make_shared<Atom>(s, "p", vector<Term> { yt, yt });

  cnf[0].push_back(make_shared<Not>(pcx));
  cnf[0].push_back(make_shared<Not>(pxx));
  cnf[1].push_back(pcy);
  cnf[1].push_back(pyy);

  cout << cnf << endl;
    
  if(resolution(cnf))
    {
      cout << "CNF satisfiable!" << endl;
    }
  else
    {
      cout << "CNF unsatisfiable!" << endl;
    }
  */
  
  return 0;
}
