#include <stdio.h>
#include <stdlib.h>


/* NAPOMENA: Prilikom promene liste (dodavanje novog
   elementa, brisanje elementa, premestanje elemenata, itd.)
   postoji mogucnost da glava liste bude promenjena, tj. 
   da to postane neki drugi cvor (sa drugom adresom). U tom
   slucaju se pokazivac na glavu liste mora azurirati. Kada
   promenu liste obavljamo u posebnoj funkciji (kao sto je bio
   slucaj u prethodnom primeru, gde smo za dodavanje i brisanje
   imali posebne funkcije) onda je potrebno da se pozivajucoj
   funkciji vrati informacija o promeni adrese glave, kako bi
   pozivajuca funkcija mogla da azurira svoju pokazivacku 
   promenljivu. Ovo se moze uraditi na dva nacina:

   1) Pozvana funkcija koja vrsi promenu na listi vraca kao
      povratnu vrednost adresu glave nakon promene. Ova adresa
      moze biti ista kao i pre promene (ako glava nije dirana)
      a moze se i razlikovati. U svakom slucaju, ta adresa treba
      da se dodeli pokazivackoj promenljivoj koja cuva adresu
      glave u pozivajucoj funkciji. Zbog toga se funkcija za promenu
      liste uvek poziva na sledeci nacin:
         pok = funkcija_za_promenu(pok, ...);
      tj. promenljivoj cija se vrednost predaje kao adresa glave u 
      pozivu mora se dodeliti povratna vrednost funkcije, za slucaj 
      da je adresa glave interno promenjena.
   2) Pozvana funkcija koja vrsi promenu na listi prihvata kao
      argument pokazivac na pokazivacku promenljivu koja u pozivajucoj
     funkciji cuva adresu glave i koju eventalno treba azurirati. Sada
     pozvana funkcija moze interno da preko dobijenog pokazivaca promeni
     promenljivu pozivajuce funkcije direktno. Npr:
        funkcija_za_promenu(&pok, ...);
     Funkcija koja se poziva je po pravilu void tipa. Prednost drugog 
     metoda je jednostavnije pozivanje, dok je prednost prvog metoda
     jednostavnija sintaksa unutar pozvane funkcije.

     U prvom primeru sa ovog casa je demonstriran prvi pristup. U ovom
     primeru ilustrujemo drugi.
   */

/* Struktura koja predstavlja cvor liste */
typedef struct cvor {
int vrednost;          /* podatak koji cvor sadrzi */
struct cvor *sledeci;  /* pokazivac na sledeci cvor */
} Cvor;


/* Pomocna funkcija koja kreira cvor. Funkcija vrednost
   novog cvora inicijalizuje na broj, dok pokazivac na
   sledeci cvor u novom cvoru postavlja na NULL. Funkcija 
   ispisuje poruku o gresci i prekida program u slucaju
   da malloc() funkcija ne uspe da alocira prostor. 
   Funkcija vraca pokazivac na novokreirani cvor */
Cvor * napravi_cvor(int broj)
{
     Cvor *novi = NULL;
     if((novi = (Cvor *) malloc(sizeof(Cvor))) == NULL)
       {
           fprintf(stderr,"malloc() greska!\n");
           exit(1);
       }

     novi->vrednost = broj;
     novi->sledeci = NULL;
     return novi;
}

/* Funkcija dodaje novi cvor na pocetak liste. Funkcija
   kreira novi cvor koriscenjem funkcije napravi_cvor(). */
void dodaj_na_pocetak_liste(Cvor **glava, int broj)
{
    Cvor * novi = napravi_cvor(broj);
    novi->sledeci = *glava;
    *glava = novi;
}

/* Funkcija dodaje novi cvor na kraj liste. Funkcija
   kreira novi cvor koriscenjem funkcije napravi_cvor(). */
void dodaj_na_kraj_liste(Cvor **glava, int broj)
{
   Cvor * novi = napravi_cvor(broj);
   Cvor *tekuci = *glava;

  /* slucaj prazne liste. U tom slucaju je glava nove liste
     upravo novi cvor. */
  if(*glava == NULL)
  {
    *glava = novi;
     return;
  }

  /* Ako lista nije prazna, tada se krecemo duz liste sve dok
     ne dodjemo do poslednjeg cvora (tj. do cvora ciji pokazivac
     na sledeci pokazuje na NULL) */
   while(tekuci->sledeci != NULL)
    tekuci = tekuci->sledeci;
   
   /* Dodajemo novi element na kraj preusmeravanjem pokazivaca */
   tekuci->sledeci = novi;

}

/* Funkcija dodaje novi element u sortiranu listu tako da i nova
   lista ostane sortirana. Funkcija kreira novi cvor koriscenjem
   funkcije napravi_cvor(). */
void dodaj_sortirano(Cvor **glava, int broj)
{
   Cvor * novi = napravi_cvor(broj);
   Cvor *tekuci = *glava;

   /* u slucaju prazne liste glava nove liste je
      upravo novi element */
   if(*glava == NULL)
   {
    *glava = novi;
     return;
   }

   /* ako je novi element manji ili jednak od glave,
      tada novi element mora da bude nova glava */
   if((*glava)->vrednost >= novi->vrednost) 
     {
         novi->sledeci = *glava;
         *glava = novi;
         return;
     }

   /* u slucaju da je glava manja od novog elementa, tada se krecemo kroz
      listu sve dok se ne dodje do elementa ciji je sledeci element veci ili
      jednak od novog elementa, ili dok se ne dodje do poslednjeg elementa. */
   while(tekuci->sledeci != NULL && tekuci->sledeci->vrednost < novi->vrednost)
     tekuci = tekuci->sledeci;

   /* U svakom slucaju novi element dodajemo IZA tekuceg elementa */
   novi->sledeci = tekuci->sledeci;
   tekuci->sledeci = novi;

}

/* Funkcija trazi u listi element cija je vrednost jednaka 
   datom broju. Funkcija vraca pokazivac na cvor liste u 
   kome je sadrzan trazeni broj ili NULL u slucaju da takav 
   element ne postoji u listi */
Cvor * pretrazi_listu(Cvor *glava, int broj)
{
     for(; glava != NULL ; glava = glava->sledeci)
      if(glava->vrednost == broj)
        return glava;

     return NULL;
}

/* Funkcija brise iz liste sve cvorove koji sadrze dati broj.
   Funkcija vraca pokazivac na glavu liste (koji moze biti 
   promenjen u slucaju da se obrise stara glava) */
void obrisi_element(Cvor **glava, int broj)
{
    Cvor *tekuci;
    Cvor *pomocni;

    /* Brisemo sa pocetka liste sve eventualne cvorove
       koji su jednaki datom broju, i azuriramo pokazivac
       na glavu */
    while(*glava != NULL && (*glava)->vrednost == broj)
      {
         pomocni = (*glava)->sledeci;
         free(*glava);
         *glava = pomocni;
      }

     /* Ako je nakon toga lista ostala prazna prekidamo
        funkciju */ 
     if(*glava == NULL) return;

     /* Od ovog trenutka se u svakom koraku nalazimo 
        na tekucem cvoru koji je razlicit od trazenog
        broja (kao i svi levo od njega). Poredimo 
        vrednost sledeceg cvora (ako postoji) sa trazenim
        brojem i brisemo ga ako je jednak, a prelazimo na
        sledeci cvor ako je razlicit. Ovaj postupak ponavljamo
        dok ne dodjemo do poslednjeg cvora. */
     tekuci = *glava;
     while(tekuci->sledeci != NULL)
       if(tekuci->sledeci->vrednost == broj)
         {
           pomocni = tekuci->sledeci;
           tekuci->sledeci = tekuci->sledeci->sledeci;
           free(pomocni);
         }
       else tekuci = tekuci->sledeci;

     return;
}

/* Funkcija prikazuje elemente liste pocev 
   od glave ka kraju liste */
void prikazi_listu(Cvor *glava)
{
   putchar('[');
   for(;glava != NULL; glava = glava->sledeci)
     printf("%d ", glava->vrednost);
   putchar(']');

   putchar('\n');
}

/* Funkcija oslobadja dinamicku memoriju zauzetu 
   od strane liste. */
void oslobodi_listu(Cvor **glava)
{
  Cvor *pomocni;

  while(*glava != NULL)
  {
    /* moramo najpre zapamtiti adresu sledeceg
       elementa, a tek onda osloboditi glavu */
    pomocni = (*glava)->sledeci;
    free(*glava);
    *glava = pomocni;
  }

}


/* test program */
int main()
{

Cvor *glava = NULL;
Cvor *pomocni = NULL;
int broj;


/* Testiranje dodavanja na pocetak */
printf("-------------------------------------------------------\n");
printf("---------- Testiranje dodavanja na pocetak ------------\n");
do {
printf("-------------------------------------------------------\n");
printf("Prikaz trenutnog sadrzaja liste:\n");
prikazi_listu(glava);
printf("-------------------------------------------------------\n");
printf("Dodati element na pocetak liste (ctrl-D za kraj unosa):\n");
} while(scanf("%d", &broj) > 0 &&  
                    (dodaj_na_pocetak_liste(&glava, broj), 1) );

printf("-------------------------------------------------------\n");
putchar('\n');

/* Testiranje pretrage elementa */
printf("-------------------------------------------------------\n");
printf("---------------- Testiranje pretrage ------------------\n");
printf("-------------------------------------------------------\n");
printf("Prikaz trenutnog sadrzaja liste:\n");
prikazi_listu(glava);
printf("-------------------------------------------------------\n");
printf("Uneti broj koji se trazi: ");
scanf("%d",&broj);

if((pomocni = pretrazi_listu(glava, broj)) == NULL)
    printf("Trazeni broj nije u listi\n");
else
   printf("Trazeni element %d je u listi\n", pomocni->vrednost); 
printf("-------------------------------------------------------\n");
putchar('\n');

oslobodi_listu(&glava);

/* Testiranje dodavanja na kraj */
printf("-------------------------------------------------------\n");
printf("------------ Testiranje dodavanja na kraj -------------\n");
do{
printf("-------------------------------------------------------\n");
printf("Prikaz liste:\n");
prikazi_listu(glava);
printf("-------------------------------------------------------\n");
printf("Dodati element na kraj liste (ctrl-D za kraj unosa):\n");
} while(scanf("%d",&broj) > 0 &&  
                   (dodaj_na_kraj_liste(&glava, broj), 1));
printf("-------------------------------------------------------\n");
putchar('\n');


/* Testiranje brisanja elemenata */
printf("-------------------------------------------------------\n");
printf("--------------- Testiranje brisanja -------------------\n");
printf("-------------------------------------------------------\n");
printf("Prikaz trenutnog sadrzaja liste:\n");
prikazi_listu(glava);
printf("-------------------------------------------------------\n");
printf("Uneti broj koji se brise: ");
scanf("%d", &broj);

obrisi_element(&glava, broj);
printf("-------------------------------------------------------\n");
printf("Lista nakon izbacivanja:\n");
prikazi_listu(glava);
printf("-------------------------------------------------------\n");
putchar('\n');

oslobodi_listu(&glava);

/* Testiranje sortiranog dodavanja */
printf("-------------------------------------------------------\n");
printf("----------- Testiranje sortiranog dodavanja -----------\n");
do{
printf("-------------------------------------------------------\n");
printf("Prikaz liste:\n");
prikazi_listu(glava);
printf("-------------------------------------------------------\n");
printf("Dodati element sortirano (ctrl-D za kraj unosa):\n");
} while(scanf("%d",&broj) > 0 &&  
              (dodaj_sortirano(&glava, broj), 1));
printf("-------------------------------------------------------\n");
putchar('\n');

oslobodi_listu(&glava);

return 0;

}
