.intel_syntax noprefix

.data
four:	.double 4.0
two:	.double 2.0
six:	.double 6.0

.text

.global simpson

#############################################################################
##
## Funkcija racuna integral funkcije x^3 na intervalu [a,b] Simpsonovom
## kvadraturnom formulom, pri cemu je interval [a,b] podeljen na 2n intervala
## jednake duzine. Funkcija koristi paralelne SSE2 instrukcije.
## 
## void simpson(double a, double b, int n, double * r);
##
## -- double a  --  [ebp+8]  -- donja granica intervala
## -- double b  --  [ebp+16] -- gornja granica intervala
## -- int n     --  [ebp+24] -- polovina broja intervala na koji se deli [a,b]
## -- double *r --  [ebp+28] -- adresa na koju se upisuje rezultat
#############################################################################
simpson:
	## Prolog funkcije
	enter 0,0
	push edi
	
	## Proverava se da li procesor podrzava SSE2 instrukcije.
        mov     eax, 1
        cpuid
        test    edx, 0x4000000
        jz not_supported

        ## Cuva se sadrzaj SSE registara (i registara koprocesora).
        mov     edx, esp
        and     esp, 0xfffffff0
        sub     esp, 512
        fxsave  [esp]

	## Kvadraturna formula Simpsona se moze napisati u ovom obliku:
	##
	## I = h/3 (f(a) - f(b) + 4 * S1 + 2 * S2)
	##
	## gde su S1 i S2 respektivno sume neparnih i parnih (ne ukljucujuci
	## f(a)) sabiraka. Prvo cemo izracunati razliku f(a) - f(b), a zatim
	## cemo paralelno racunati S1 i S2. 

	## Ucitavamo n u registar ecx
	mov ecx, [ebp+24]

	## Ucitavamo a u nizi deo registra xmm1
	movsd xmm1, [ebp+8]

	## Ucitavamo b u nizi deo registra xmm2
	movsd xmm2, [ebp+16]
	
	## Izracunavamo a^3 i smestamo ga u xmm7
	movsd xmm3, xmm1
	mulsd xmm3, xmm1
	mulsd xmm3, xmm1
	movsd xmm7, xmm3
	
	## Izracunavamo b^3 i oduzimamo ga od xmm7 (sada je u xmm7 a^3 - b^3) 
	movsd xmm3, xmm2
	mulsd xmm3, xmm2
	mulsd xmm3, xmm2
	subsd xmm7, xmm3

	## Racunamo h = (b-a)/2n i smestamo ga u oba dela registra xmm6
	movsd xmm6, xmm2
	subsd xmm6, xmm1
	mov eax, ecx
	shl eax, 1
	cvtsi2sd xmm3, eax
	divsd xmm6, xmm3
	shufpd xmm6, xmm6, 0b00

	## Racunamo a + h i smestamo u nizi deo registra xmm1. a + 2h smestamo
	## u visi deo registra xmm1
	addsd xmm1, xmm6
	movsd xmm3, xmm1
	addsd xmm3, xmm6
	shufpd xmm1, xmm3, 0b00

	## Racunamo (2h, 2h) u registru xmm6
	addpd xmm6, xmm6

	## Registar xmm0 cemo koristiti za parcijalne sume.
	xorpd xmm0, xmm0	
	
next_two:
	## Racunamo po jedan sabirak suma S1 i S2 paralelno, i dodajemo na
	## tekuce vrednosti suma.
	movapd xmm2, xmm1
	mulpd xmm2, xmm1
	mulpd xmm2, xmm1
	addpd xmm0, xmm2

	## Racunamo sledece dve tacke u kojima se racuna vrednost funkcije.
	addpd xmm1, xmm6
	
	loop next_two

compute_sum:
	## U nizi deo registra xmm1 smestamo 4. U visi deo registra xmm1
	## smestamo 2.
	movlpd xmm1, four
	movhpd xmm1, two

	## Mnozimo sumu S1 sa 4, a sumu S2 sa 2. Nakon toga, na sumu S1 jos
	## dodajemo sabirak a^3 - b^3.
	mulpd xmm0, xmm1
	addsd xmm0, xmm7

	## Sabiramo dve parcijalne sume.
	movapd xmm1, xmm0
	shufpd xmm1, xmm1, 0b11
	addsd xmm0, xmm1

	## Mnozimo sumu sa h/3 = 2h/6
	mulsd xmm0, xmm6
	divsd xmm0, six

	## Upisujemo rezultat u za to predvidjenu lokaciju.
	mov edi, [ebp+28]
	movsd [edi], xmm0
	
finish:
        ## Vraca se prethodni sadrzaj u registre koprocesora i SSE
	## registre.
        fxrstor [esp]
        mov     esp, edx
	
done:	
	## Epilog funkcije
	pop edi
	leave
	ret

not_supported:
	## Ako SSE2 instrukcije nisu podrzane, prekidamo program exit(1)
	## pozivom. 
	mov eax, 1
	mov ebx, 1
	int 0x80
	