#include <stdio.h>
#include <stdlib.h>

#define MAX_CVOROVA 100

/* NAPOMENA: Graf je uredjeni par (V,E) gde je V neprazan
   skup cvorova, a E je skup grana -- parova (u,v), gde su 
   u i v cvorovi iz V (kazemo da grana (u,v) povezuje cvorove
   u i v iz V). Ako su grane uredjeni parovi, tada za graf
   kazemo da je USMEREN, a ako su grane neuredjeni parovi, 
   tada za graf kazemo da je NEUSMEREN. Jasno je da je u 
   neusmerenom grafu grana (u,v) isto sto i grana (v,u), dok
   kod usmerenog grafa to nije slucaj.
   
   Put u grafu je niz cvorova v1,v2,...,vn, pri cemu u grafu
   postoje grane (v1,v2), (v2,v3), ..., (vn-1, vn). Ako jos
   vazi da je v1 = vn, tada se ovakav put naziva ciklus.

   Lako se primeti da su liste i stabla koje smo prethodno
   ucili samo specijalni oblici grafova. Medjutim, u 
   programiranju postoji jos jedna bitna razlika: za razliku
   od lista i stabala koja su prevashodno koriscena kao 
   kontejneri za podatke (u svaki cvor smo smestali neki 
   podatak odredjenog tipa), grafovi se pre svega koriste za
   modeliranje nekih problema iz prakse, kako bi se ti problemi
   predstavili na apstraktan nacin i resili primenom algoritama
   nad odgovarajucim grafom. Npr. cvorovima u grafu se mogu 
   predstaviti raskrsnice u gradu, a granama ulice koji ih 
   povezuju. Tada se zadatak nalazenja najkrace marsute od tacke
   A do tacke B u gradu moze svesti na nalazenje najkraceg puta
   od cvora u do cvora v u grafu. Slicno se mogu resavati problemi
   sa racunarskim mrezama (racunari se predstave cvorovima, a 
   veze medju njima granama), itd.

   Grafovi se u racunaru mogu predstavljati na vise nacina. 
   Najjednostavniji nacin je da se graf sa n cvorova predstavi 
   matricom n x n, gde ce element na poziciji (i,j) biti jednak
   1 ako i samo ako postoji grana izmedju i-tog i j-tog cvora,
   u suprotnom ce biti 0. Jasno je da ce neusmerenim grafovima
   odgovarati simetricne matrice. Prednost ovakvog predstavljanja
   je jednostavnost. Nedostatak ovakvog pristupa je neracionalno
   koriscenje memorije u slucaju "retkih" grafova (grafova sa malo
   grana), gde ce najveci broj elemenata matrice biti 0. U ovom 
   slucaju se pribegava predstavljaju grafova tzv. listama
   povezanosti: kreiramo listu (ili niz) cvorova, pri cemu svakom
   cvoru dodeljujemo listu grana koje iz tog cvora izlaze. Ovakav
   pristup stedi memoriju, a omogucava i efikasno dodavanje novih
   cvorova i grana. Nedostatak mu je komplikovanost.

   Mi cemo u primeru koji sledi koristiti prvi pristup -- grafove
   cemo predstavljati matricama povezanosti. Pri tom cemo zbog
   jednostavnosti pretpostaviti da je broj cvorova ogranicen nekom
   fiksiranom konstantom, tako da cemo odgovarajuce matrice alocirati
   staticki. Kako bismo jos dodatno pojednostavili liste argumenata
   pojedinih funkcija, vecina promenljivih u ovom primeru bice 
   deklarisane kao globalne, pa ce kao takve biti dostupne svim 
   funkcijama. Primer koji sledi pre svega treba da demonstrira neke
   osnovne algoritme nad grafovima, pa ce za tu svrhu ovako jednostavna
   implementacija biti dovoljna. 
*/

/* Matrica povezanosti grafa. Pretpostavka je da broj
   cvorova grafa nece biti veca od MAX_CVOROVA */
int graf[MAX_CVOROVA][MAX_CVOROVA];
int broj_cvorova;

/* Niz 0/1 vrednosti kojim se obelezavaju poseceni cvorovi */
int posecen[MAX_CVOROVA];


/* NAPOMENA: Osnovni problem koji se javlja kod grafova jeste kako
   polazeci od nekog cvora, kretanjem kroz puteva grafa posetiti
   sve cvorove. Ovakav postupak se naziva obilazak grafa. U nastavku
   demonstriramo dva osnovna nacina obilaska grafa -- obilazak u
   dubinu i obilazak u sirinu. Prilikom obilaska vrsi se odgovarajuca
   obrada cvorova i grana koje se prolaze i koja zavisi od samog
   problema. */

/* Pomocna funkcija koja utvrdjuje da li postoji neposecen cvor. 
   Funkcija vraca indeks prvog takvog cvora, ili -1 u slucaju da
   su svi cvorovi vec poseceni */ 
int postoji_neposecen()
{
  int i;

  for(i = 0; i < broj_cvorova; i++)
    if(!posecen[i]) return i;
  
  return -1;
  
}

/* Pomocna funkcija koja sluzi za inicijalizaciju raznih nizova
   (poput posecen[], dolazna_numeracija[] itd.) Za sve nizove
   se pretpostavlja da su duzine broj_cvorova.  */
void inicijalizuj(int niz[])
{
  int i;
  
  for(i = 0 ; i < broj_cvorova ; i++)
    niz[i] = 0;
  
}

/* NAPOMENA: za potrebu demonstracije obilaska grafa, obrada koja
   se prilikom obilaska vrsi bice numerisanje cvorova grafa u 
   poretku posecivanja.  */

/* Staticki podaci koji se koriste za smestanje rezultata
   prilikom numeracije cvorova (i za DFS i za BFS). */
int dolazna_numeracija[MAX_CVOROVA];
int odlazna_numeracija[MAX_CVOROVA];

/* Donji brojaci treba da se postave na nulu pre pozivanja
   funkcija za numeraciju */ 
int brojac_dolazna=0;
int brojac_odlazna=0;

/* Funkcija obilazi graf DFS algoritmom, i tom prilikom
   vrsi dolaznu i odlaznu numeraciju cvorova. Pre poziva
   ove funkcije obavezno je postaviti brojace dolazne i
   odlazne numeracije na 0 */ 
void DFS_numeracija(int polazni_cvor)
{
  int i;
  
  /* Obelezavanje */
  posecen[polazni_cvor] = 1;
  
  /* Ulazna obrada */
  dolazna_numeracija[polazni_cvor] = ++brojac_dolazna;
  
  /* Rekurzija (za sve cvorove ka kojima postoji grana
     i koji jos nisu obelezeni) */
  for(i = 0 ; i < broj_cvorova; i++)
    if(graf[polazni_cvor][i] && !posecen[i])
      DFS_numeracija(i);
  
  /* Izlazna obrada */
  odlazna_numeracija[polazni_cvor] = ++brojac_odlazna;
  
}


/* Funkcija vrsi BFS obilazak i numerise cvorove u poretku
   obilaska. Pre poziva ove funkcije treba inicijalizovati
   posecen[] i brojac_dolazna */
void BFS_numeracija(int polazni_cvor)
{

  /* Improvizovani red koji koristimo za smestanje
     cvorova koji cekaju da budu obradjeni. Cvorovi
     se smestaju na kraj reda (kraj niza), a uzimaju
     sa pocetka reda (niza). Zato imamo dva indeksa
     koji pokazuju na pocetak i kraj reda. Duzina
     ovog reda je MAX_CVOROVA sto je dovoljno jer
     ce svaki cvor na red biti postavljen tacno jednom */
  int red[MAX_CVOROVA];
  int sledeci_na_redu = 0;
  int poslednji_na_redu = -1;
  int i;
  
  /* Inicijalno se na redu nalazi samo polazni cvor */
  posecen[polazni_cvor] = 1;
  red[++poslednji_na_redu] = polazni_cvor;
  
  /* Dokle god imamo cvorove u redu...*/
  while(sledeci_na_redu <= poslednji_na_redu)
    {
      /* Vrsimo obradu cvora (numeracija) */
      dolazna_numeracija[red[sledeci_na_redu]] = ++brojac_dolazna;
      
      /* Za sve cvorove ka kojima postoji grana i koji nisu
	 poseceni vrsimo obelezavanje i dodavanje u red */
      for(i = 0; i < broj_cvorova; i++)
	if(graf[red[sledeci_na_redu]][i] && !posecen[i])
	  {
	    red[++poslednji_na_redu] = i;
	    posecen[i] = 1;
	  }
      /* Prelazimo na sledeci u redu */
      sledeci_na_redu++;
    }
  
}



/* Test program */
int main()
{
  int i,j;
  int aciklican;
  
  /* Unos matrice povezanosti grafa */
  printf("Unesite broj cvorova grafa (<=%d): ", MAX_CVOROVA);
  scanf("%d",&broj_cvorova);
  
  for(i = 0; i < broj_cvorova; i++)
    for(j = 0; j < broj_cvorova; j++)
      {
	if(i == j) continue;
	
	printf("Da li postoji grana (%d,%d) (0/1): ", i, j);
	scanf("%d", &graf[i][j]);
      }
  
  /* Testiramo DFS numeraciju */
  inicijalizuj(posecen);
  
  brojac_dolazna = 0;
  brojac_odlazna = 0;
  
  while((i = postoji_neposecen()) >= 0)
    DFS_numeracija(i);
  
  printf("Dolazna DFS numeracija: ");
  for(i = 0; i < broj_cvorova; i++)
    printf("%d ",dolazna_numeracija[i]);
  
  putchar('\n');
  printf("Odlazna DFS numeracija: ");
  for(i = 0; i < broj_cvorova; i++)
    printf("%d ", odlazna_numeracija[i]);
  putchar('\n');
  
  
  /* Testiramo BFS numeraciju */
  inicijalizuj(posecen);
  brojac_dolazna = 0;
  
  while((i = postoji_neposecen()) >= 0)
    BFS_numeracija(i);
  
  printf("Dolazna BFS numeracija: ");
  for(i = 0; i < broj_cvorova; i++)
    printf("%d ",dolazna_numeracija[i]);
  putchar('\n');
  
  return 0;
}
