#include <stdio.h>

/* Funkcija izdvaja iz podatka x bit na poziciji i (pozicije se broje sa desna
   u levo pocev od 0), i smesta ga u najnizi bit podatka na koji pokazuje b.
   Ostali bitovi rezultujuceg podatka se postavljaju na 0. */
void procitaj_bit(unsigned x, int i, unsigned * b)
{
  *b = (x >> i) & 1;
}

/* Funkcija postavlja bit na poziciji i u podatku na koji pokazuje x na 
   vrednost najnizeg bita u podatku b */
void postavi_bit(unsigned * x, int i, unsigned b)
{
  *x = (b << i) | (*x & ~(1 << i));
}

/* Funkcija simulira POLUSABIRAC, tj. kolo koje sabira dva bita ne uzimajuci
   u obzir prethodni prenos. Vrednosti bitova koji se sabiraju nalaze se
   u najnizim bitovima podataka x i y, vrednost zbira se smesta u najnizi
   bit podatka na koji pokazuje S, dok se vrednost prenosa smesta u najnizi
   bit podatka na koji pokazuje C. */
void polusabirac(unsigned x, unsigned y, unsigned * S, unsigned * C)
{
  *S = x ^ y;
  *C = x & y;
}

/* Funkcija simulira POTPUNI SABIRAC, tj. kolo koje sabira dva bita uzimajuci
   u obzir prethodni prenos. Parametri su kao i u prethodnoj funkciji, uz
   dodatni parametar pc ciji najnizi bit cuva vrednost prethodnog prenosa */
void potpuni_sabirac(unsigned x, unsigned y, unsigned pc,
		     unsigned * S, unsigned * C)
{
  unsigned S1, C1, C2;
  
  
  polusabirac(x, y, &S1, &C1);
  polusabirac(S1, pc, S, &C2);
  *C = C1 | C2;
}

/* Funkcija vrsi sabiranje dva neoznacena cela broja x i y ne koristeci
   operator +, simulacijom algoritma sabiranja koji se obavlja u hardveru.
   Vrednost zbira se smesta na lokaciju na koju pokazuje S. Ukoliko
   je prilikom sabiranja nastupilo prekoracenje, podatak na koji pokazuje
   C postavlja se na 1, a u suprotnom na 0 */
void saberi_neoznacene(unsigned x, unsigned y, unsigned * S, unsigned * C)
{
  int i;
  unsigned bx, by;
  unsigned bs, bc, pc = 0;
  
  
  for(i = 0; i < 32; i++)
    {
      /* Ucitavamo i-te bitove podataka x i y. */
      procitaj_bit(x, i, &bx);
      procitaj_bit(y, i, &by);

      /* Sabiramo bitove, uz uvazavanje prethodnog prenosa */
      potpuni_sabirac(bx, by, pc, &bs, &bc);

      /* Postavljamo i-ti bit rezultata, a dobijeni prenos cuvamo za sledecu
	 iteraciju */
      postavi_bit(S, i, bs);
      pc = bc;
    }
  
  /* Zavrsni prenos je ujedno i indikator prekoracenja */
  *C = pc;
  
}

/* Funkcija implementira POLUODUZIMAC (bez uzimanja u obzir prethodne
   pozajmice) */
void poluoduzimac(unsigned x, unsigned y, unsigned * S, unsigned * C)
{
  *S = x ^ y;
  *C = (x ^ 1) & y;
}

/* Funkcija implementira POTPUNI ODUZIMAC (uz uzimanje u obzir prethodne
   pozajmice) */
void potpuni_oduzimac(unsigned x, unsigned y, unsigned pc,
		      unsigned * S, unsigned * C)
{
  unsigned S1, C1, C2;
  
  
  poluoduzimac(x, y, &S1, &C1);
  poluoduzimac(S1, pc, S, &C2);
  *C = C1 | C2;
}

/* Funkcija oduzima dva neoznacena cela broja x i y bez koriscenja operatora
   -, simulirajuci algoritam oduzimanja koji se koristi u hardveru. */
void oduzmi_neoznacene(unsigned x, unsigned y, unsigned * S, unsigned * C)
{
  int i;
  unsigned bx, by;
  unsigned bs, bc, pc = 0;
  
  
  for(i = 0; i < 32; i++)
    {

      /* Citmo i-ti bit u podacima x i y */
      procitaj_bit(x, i, &bx);
      procitaj_bit(y, i, &by);

      /* Oduzimamo bitove uzimajuci u obzir prethodnu pozajmicu */
      potpuni_oduzimac(bx, by, pc, &bs, &bc);

      /* Smestamo dobijenu razliku na poziciju i u rezultatu. */
      postavi_bit(S, i, bs);

      /* Nova pozajmica postaje prethodna pozajmica, za sledecu iteraciju */
      pc = bc;
    }
  
  *C = pc;
}

/* Funkcija pomera registre C, A, P lancano u desno za jednu poziciju */
void pomeri_registre_u_desno(unsigned * C, unsigned * A, unsigned * P)
{
  *P >>= 1;
  *P |= (*A & 1) << 31;
  *A >>= 1;
  *A |= (*C & 1) << 31;
  *C >>= 1;
}



/* Funkcija pomera registre A i P lancano u levo za jednu poziciju */
void pomeri_registre_u_levo(unsigned * A, unsigned * P)
{
  *A <<= 1;
  *A |= (*P & 0x80000000) >> 31;
  *P <<= 1;
}


/* Funkcija mnozi dva neoznacena cela broja x i y, a njihov 64-bitni 
   proizvod smesta na lokacije na koje pokazuju RL (nizih 32 bita) i 
   RH (visih 32 bita). */
void pomnozi_neoznacene(unsigned x, unsigned y, unsigned * RL, unsigned * RH)
{

  unsigned M = x;
  unsigned P = y;
  unsigned A = 0;
  unsigned C = 0;
  int i;

  for(i = 0; i < 32; i++)
    {
      /* Ako je bit najmanje tezine u mnoziocu 1, tada se mnozenik M dodaje
	 na parcijalnu sumu A. */
      if(P & 1 != 0)
	{
	  saberi_neoznacene(A, M, &A, &C);
	}
      /* Pomeramo lancano registre C, A i P za jednu poziciju u desno */
      pomeri_registre_u_desno(&C, &A, &P);
    }

  /* Smestamo rezultat u za to predvidjene lokacije */
  *RL = P;
  *RH = A;
}


/* Funkcija deli dva neoznacena cela broja x i y. Kolicnik smesta na adresu
   na koju pokazuje Q, a ostatak na lokaciju na koju pokazuje R */
void podeli_neoznacene(unsigned x, unsigned y, unsigned * Q, unsigned * R)
{
  unsigned P = x, M = y;
  unsigned A = 0;
  int i;
  
  for(i = 0; i < 32; i++)
    {
      unsigned C;
      /* Pomeramo lancano u levo za jednu poziciju registre A i P */
      pomeri_registre_u_levo(&A, &P);
      
      /* Oduzimamo M od A i razliku smestamo u A */
      oduzmi_neoznacene(A, M, &A, &C);
      
      /* Ako je oduzimanje uspesno (nije bilo prekoracenja), tada se 
	 ukljucuje bit najmanje tezine u P. U suprotnom se ponistava
	 oduzimanje (sabiranjem sa M), a bit najmanje tezin u P ostaje
	 jednak 0. */ 
      if(C == 0)
	{
	  P |= 1;
	}
      else
	{
	  saberi_neoznacene(A, M, &A, &C);
	}
    }
  
  /* Smestaju se rezultati u za to predvidjene lokacije */
  *Q = P;
  *R = A;
}

/* Funkcija izracunava -x za dati oznaceni ceo broj x. Rezultat se smesta 
   na adresu na koju pokazuje R */
void promeni_znak(int x, int * R)
{
  unsigned C;
  x = ~x;
  saberi_neoznacene(x, 1, R, &C);
}


/* Funkcija ispituje da li je broj x >= 0 */
int pozitivan(int x)
{
  return (x & 0x80000000) == 0;
}

/* Funkcija ispituje da li je broj x < 0 */
int negativan(int x)
{
  return (x & 0x80000000) != 0;
}

/* Funkcija sabira dva oznacena cela broja x i y. Rezultat smesta na 
   adresu na koju pokazuje S, dok se informacija o prekoracenju smesta
   na lokaciju na koju pokazuje C */
void saberi_oznacene(int x, int y, int * S, int * C)
{
  unsigned C1;
  saberi_neoznacene(x, y, S, &C1);
  *C = (pozitivan(x) && pozitivan(y) && negativan(*S))
    ||
    (negativan(x) && negativan(y) && pozitivan(*S));
}

/* Funkcija oduzima oznacene brojeve x i y. Oduzimanje se vrsi svodjenjem
   na oduzimanje neoznacenih celih brojeva, slicno kao i kod sabiranja. */
void oduzmi_oznacene(int x, int y, int * S, int * C)
{
  unsigned C1;
  oduzmi_neoznacene(x, y, S, &C1);
  *C = (pozitivan(x) && negativan(y) && negativan(*S)) ||
    (negativan(x) && pozitivan(y) && pozitivan(*S));
}

/* Funkcija vrsi lancano aritmeticko siftovanje u desno za jednu poziciju
   int-ova cije se adrese zadaju kao argumenti funkcije */
void pomeri_registre_u_desno_aritmeticki(unsigned * A, unsigned * P, unsigned * P1)
{
  int * A1 = (int*)A; // Da bi pomeranje bilo aritmeticko, najvisi
		      // registar se tumaci kao int (konverzija pokazivaca)
  
  *P1 >>= 1;
  *P1 |= (*P & 1) << 31;
  *P >>= 1;
  *P |= (*A1 & 1) << 31;
  *A1 >>= 1; 
}

/* Mnozenje dva oznacena broja x i y Butovim algoritmom. */
void pomnozi_oznacene(int x, int y, int * RL, int * RH)
{
  int M = x, P = y, P1 = 0, A = 0;
  int i;
  
  for(i = 0; i < 32; i++)
    {
      int C;
      
      /* U slucaju kombinacije 10 vrsi se oduzimanje */
      if(!(P1 & 0x80000000) && (P & 1))
	{
	  oduzmi_oznacene(A, M, &A, &C);
	}
      
      /* U slucaju kombinacije 01 vrsi se sabiranje */
      if((P1 & 0x80000000) && !(P & 1))
	{
	  saberi_oznacene(A, M, &A, &C);
	}
      /* U kombinacijama 00 i 11 ne radi se nista */

      /* Pomeramo aritmeticki registre A, P i P1 lancano u desno */
      pomeri_registre_u_desno_aritmeticki(&A, &P, &P1);
    }
  
  /* Smestamo rezultate na za to predvidjene lokacije */
  *RH = A; 
  *RL = P;
}

/* Deljenje dva oznacena cela broja x i y. */
void podeli_oznacene(int x, int y, int * Q, int * R)
{
  int M = y, P = x, A;
  int C;
  int i;
  int pozP, pozA, pozM;

  /* Pamtimo znak deljenika P */
  pozP = pozitivan(P);

  /* Pamtimo znak delioca M */
  pozM = pozitivan(M);
  
  /* A se postavlja ili na sve nule ili na sve jedinice u binarnom zapisu,
     sto je ekvivalentno oznacenom prosirivanju podatka P */
  if(pozP)
    A = 0;
  else
    A = -1;
   
  for(i = 0; i < 32; i++)
    {
      /* Pomeramo lancano u levo registre A i P */
      pomeri_registre_u_levo(&A, &P);

      /* Pamtimo znak trenutne vrednosti registra A */
      pozA = pozitivan(A);
      
      /* Ako su A i M istog znaka, oduzimamo M od A */
      if((pozA && pozM) || (!pozA && !pozM))
	{
	  oduzmi_oznacene(A, M, &A, &C);
	}
      else /* U suprotnom dodajemo M na A */
	{
	  saberi_oznacene(A, M, &A, &C);
	}
      
      /* Ako je operacija uspesna, ukljucujemo bit najmanje tezine u P */
      if((pozA && pozitivan(A)) ||
	 (!pozA && negativan(A)) || 
	 (!pozA && A == 0 && (P  & (0xffffffff << i)) == 0))
	{
	  P |= 1;
	}
      else /* U suprotnom, ponistavamo prethodnu operaciju, a bit u P ostaje 0 */
	{
	  if((pozA && pozM) || (!pozA && !pozM))
	    {
	      saberi_oznacene(A, M, &A, &C);
	    }
	  else
	    {
	      oduzmi_oznacene(A, M, &A, &C);
	    }	  
	}      
    }

  /* Na kraju se po potrebi menja znak kolicnika */
  if((pozP && pozM)
     ||
     (!pozP && !pozM))
    *Q = P;
  else
    promeni_znak(P, Q);
  *R = A;
}


int main()
{
  int x,y, s, c;
  int p, q;

  scanf("%d %d", &x, &y);
  
  saberi_oznacene(x, y, &s, &c);  
  printf("Zbir: %d\tPrekoracenje: %d\n", s, c);
  
  oduzmi_oznacene(x, y, &s, &c);  
  printf("Razlika: %d\tPrekoracenje: %d\n", s, c);

  pomnozi_oznacene(x, y, &p, &q);
  printf("Visi deo proizvoda: %d\tNizi deo proizvoda: %d\n", q, p);
  
  podeli_oznacene(x, y, &p, &q);
  printf("Celobrojni kolicnik: %d\tOstatak pri deljenju: %d\n", p, q);
  
  return 0;
}
