#include <iostream>
#include <vector>
#include <vector>
#include <cstdlib>

using namespace std;

/* Strukture podataka */

/* Varijable su predstavljene nenegativnim celim brojevima, pocev od
   nule. */
typedef unsigned Variable;  

/* Literali su takodje predstavljeni nenegativnim celim brojevima, po
   sledecem principu: varijabli 0 odgovaraju literali 0 i 1 (pozitivan
   i negativan literal respektivno), varijabli 1 odgovaraju literali
   2 i 3, varijabli 2 odgovaraju literali 4 i 5, i td. Dakle, parni 
   brojevi predstavljaju pozitivne literale, a neparni negativne.
   Jasno je da varijabli (i) odgovaraju literali 2*i i 2*i + 1. Ovo 
   omogucava jednostavno dobijanje varijable od literala i obrnuto:
   siftovanjem varijable u levo za jednu poziciju dobija se pozitivan
   literal kojo odgovara toj varijabli, a ukljucivanjem najnizeg bita
   nakon siftovanja dobija se negativni literal. Obrnuta tranformacija
   je takodje jednostavna: samo se literal (pozitivan ili negativan)
   siftuje u desno za jednu poziciju i dobije se odgovarajuca varijabla */
typedef unsigned Literal;

/* Enumeratorski tip koji predstavlja polaritet literala */
enum Polarity { P_POSITIVE, P_NEGATIVE };

/* Dobijanje literala datog polariteta iz varijable */
Literal litFromVar(Variable v, Polarity p)
{
  return p == P_POSITIVE ? v << 1 : (v << 1) | 1;
}

/* Dobijanje varijable iz literala proizvoljnog polariteta */
Variable varFromLit(Literal l)
{
  return l >> 1;
}

/* Ispitivanje da li je literal pozitivan (ako je broj paran, tj. ako
   je najnizi bit 0) */
bool isPositive(Literal l)
{
  return !(l & 1);
}

/* Ispitivanje da li je literal negativan (ako je broj neparan, tj. ako
   je najnizi bit 1) */
bool isNegative(Literal l)
{
  return l & 1;
}

/* Dobijanje suprotnog literala (invertovanjem najnizeg bita) */
Literal oppositeLiteral(Literal l)
{
  return l ^ 1;
}


/* NAPOMENA: S obzirom da se u DIMACS formatu varijable predstavljaju 
   celim brojevima pocev od 1 (a ne od nule, kao kod nas), dok se
   literali predstavljaju oznacenim celim brojevima razlicitim od nule
   (pozitivni brojevi za pozitivne literale, negativni za negativne),
   potrebno je da za svrhu ucitavanja i ispisa imamo funkcije koje
   tranformisu nas zapis literala u DIMACS-ov. */

/* Tranformise se literal u oznaceni ceo broj kojim se taj literal 
   predstavlja u DIMACS-u. Ovo se radi tako sto se dobije varijabla,
   njena vrednost se uveca za jedan (jer se u DIMACS-u varijable 
   numerisu pocev od 1), a zatim se eventualno promeni znak (ako je
   literal bio negativan). Tako se, npr. od literala 5 dobije varijabla
   2, sto je treca DIMACS varijabla, a literal je negativan, tako da 
   je rezultat -3. */
int intFromLit(Literal l)
{
  return isPositive(l) ? (int)varFromLit(l) + 1 : -(int)(varFromLit(l) + 1);
}

/* Obrnuta konverzija iz celog oznacenog broja (kako se literali predstavljaju
   u DIMACS-u) u literal (onako kako je kod nas predstavljen) */
Literal litFromInt(int i)
{
  return i > 0 ? litFromVar(i - 1, P_POSITIVE) : litFromVar(-i - 1, P_NEGATIVE);
}

/* Klauza ce biti predstavljena vektorom (dinamickim nizom) literala */
typedef vector<Literal> Clause;

/* Formula je vektor (niz) klauza */
typedef vector<Clause> Formula;

/* Pomocna funkcija koja ispisuje klauzu u DIMACS obliku. Funkcija postoji
   za potrebe detaljnijeg izvestaja u toku rada resavaca (prikazivanje
   klauza koje su u konfliktu, iz kojih slede propagacije i sl.) i nije
   sustinski znacajna za sam rad resavaca. */
void printClause(const Clause & c, ostream & ostr)
{
  for(Clause::const_iterator i = c.begin(); i != c.end(); i++)
    ostr << intFromLit(*i) << " ";
  ostr << endl;
}

/* Semantika iskazne logike propisuje da se svakoj iskaznoj varijabli 
   valuacijom dodeljuje jedna od vrednosti {true, false}. Medjutim, 
   s obzirom da je DPLL algoritam po svojoj prirodi inrekementalan,
   tj. valuacija se gradi postupno, tako sto se jednoj po jednoj  
   varijabli dodeljije neka od ove dve vrednosti, postojace i one
   varijable koje u datom trenutku nemaju definisanu vrednost u
   toj (parcijalnoj) valuaciji. Zbog toga nam je potrebno da prosirimo
   bool tip tako da bude trovalentan: tacno, netacno i nedefinisano.
   U ovu svrhu uvodimo enumeratorski tip ExtendedBoolean */
enum ExtendedBoolean { B_TRUE, B_FALSE, B_UNDEFINED };

/* Operator vraca suprotnu vrednost ExtendedBoolean tipa (suprotno od
   nedefinisanog je nedefinisano) */
ExtendedBoolean operator ! (ExtendedBoolean b)
{
    switch(b)
    {
    case B_TRUE:
      return B_FALSE;
      break;
    case B_FALSE:
      return B_TRUE;
      break;
    }
     
  return B_UNDEFINED;
}

/* Klasa Valuacija predstavlja parcijalnu valuaciju. Ova struktura podataka
   se sastoji iz dva dela: prvi deo je vektor ExtendedBoolean vrednosti 
   duzine num_of_vars (broj varijabli sa kojima baratamo). Varijabli (i)
   odgovara vrednost na poziciji (i) u ovom vektoru i moze biti tacno, netacno
   ili nedefinisano. Medjutim, s obzirom da je DPLL u svojoj sustini zasnovan
   na backtrack-u, potrebno je da mozemo da u odredjenom trenutku ponistimo
   neke dodele vrednosti promenljivima koje su prethodno ucinjene. Zbog 
   toga je potrebno da valuacija bude organizovana kao stek struktura -- 
   kako se kojoj varijabli dodeli vrednost, tako se odgovarajuci literal 
   (pozitivan ako je dodeljena vrednost tacno, a negativan ako je dodeljena
   vrednost netacno) postavi na stek. Kada je potrebno uraditi backtrack,
   tada se prosto sa vrha steka skinu odgovarajuci literali (istovremeno
   ponistavajuci njihove vrednosti u tabeli vrednosti, tj. postavljanjem
   vrednosti na nedefinisano). Backtrack se vrsi sve do poslednje tacke
   u kojoj je doneta neka "odluka", tj. tacka u kojoj je izvrsen proizvoljan
   izbor (koji nije rezultat rezonovanja) varijable i njene vrednosti 
   (polariteta literala) koja ce joj biti dodeljena (Decide pravilo, videti
   dole). Tada se umesto te odluke postavi suprotna, i pretrega se nastavlja
   u suprotnoj grani. Zbog toga struktura podataka Valuation mora podrzavati
   tzv. nivoe odlucivanja (eng. decision levels) -- svaki put kada se primeni
   Decide pravilo, na steku se uspostavi novi nivo odlucivanja, a uz svaki
   literal koji se nakon toga stavi na stek pridruzuje se oznaka nivoa 
   odlucivanja kome literal pripada. Kada se izvrsi Backtrack, tada se prosto
   sa steka skinu svi literali koji se nalaze u tekucem nivou odlucivanja,
   nakon cega se taj nivo odlucivanja ponistava, a vracamo se u prethodni 
   nivo odlucivanja. */
class Valuation {
private:
  vector<ExtendedBoolean> _values;
  vector< pair<Literal, unsigned> > _stack;
  unsigned _curr_level;
public:
  Valuation(unsigned num_of_vars)
    :_values(num_of_vars, B_UNDEFINED),
     _curr_level(0)
  {}
  
  unsigned currentLevel() const
  {
    return _curr_level;
  }

  /* Postavlja na stek novi literal u tekucem nivou odlucivanja i 
     postavlja vrednost odgovarajuce varijable u tabeli vrednosti.
     Ako je decide == true, tada se uspostavlja novi nivo odlucivanja
     na steku (tj. l je literal odlucivanja) */
  void push(Literal l, bool decide = false)
  {
    if(decide)
      _curr_level++;

    _stack.push_back(make_pair(l, _curr_level));
    _values[varFromLit(l)] = isPositive(l) ? B_TRUE : B_FALSE;
  }


  /* Funkcija vrsi backtrack i nakon toga vraca prvi literal iz upravo
     ponistenog nivoa odlucivanja -- to je upravo tzv. "decision literal",
     tj. onaj koji je postavljen kao rezultat primene Decide pravila */
  Literal backtrack()
  {
    Literal l;
    while(_stack.back().second == _curr_level)
      {
	_values[varFromLit(_stack.back().first)] = B_UNDEFINED;
	l = _stack.back().first;
	_stack.pop_back();
      }
    _curr_level--;
    return l;
  }

  /* Pronalazi prvu varijablu sa nedefinisanom vrednoscu (vraca false 
     ako su sve varijable definisane, tj. ako je valuacija potpuna) */
  bool findFirstUndefinedVariable(Variable & v) const
  {
    for(unsigned i = 0; i < _values.size(); i++)
      if(_values[i] == B_UNDEFINED)
	{
	  v = i;
	  return true;
	}
    return false;
  }

  /* Vraca vrednost varijable u tekucoj parcijalnoj valuaciji */
  ExtendedBoolean variableValue(Variable v) const
  {
    return _values[v];
  }

  /* Vraca vrednost literala u tekucoj parcijalnoj valuaciji */
  ExtendedBoolean literalValue(Literal l) const
  {
    return isPositive(l) ? 
      _values[varFromLit(l)] : 
      !_values[varFromLit(l)];
  }

  /* Proverava da li je klauza netacna u tekucoj parcijalnoj valuaciji,
     tj. da li su svi njeni literali netacni (ponisteni) */
  bool isClauseFalse(const Clause & c) const
  {
    for(Clause::const_iterator i = c.begin(); i != c.end(); i++)
      {
	if(literalValue(*i) == B_TRUE || literalValue(*i) == B_UNDEFINED)
	  return false;
      }
    return true;
  }
  
  /* Proverava da li je data klauza jedinicna u tekucoj parcijalnoj 
     valuaciji -- klauza je jedinicna ako su svi njeni literali netacni
     osim jednog koji je nedefinisan. Funkcija vraca dati literal ako
     klauza jeste jedinicna. */
  bool isClauseUnit(const Clause & c, Literal & l) const
  {
    bool found = false;

    for(Clause::const_iterator i = c.begin(); i != c.end(); i++)
      {
	if(literalValue(*i) == B_TRUE)
	  return false;

	if(literalValue(*i) == B_UNDEFINED)
	  {
	    if(!found)
	      {
		found  = true;
		l = *i;
	      }
	    else
	      {
		return false;
	      }
	  }
      }
    return found;
  }

  /* Prikazuje tekucu valuaciju */
  void printValuation(ostream & ostr) const
  {
    for(unsigned i = 0; i < _values.size(); i++)
      {
	if(_values[i] == B_UNDEFINED)
	  ostr << "U ";
	else if(_values[i] == B_TRUE)
	  ostr << (int)(i + 1) << " ";
	else
	  ostr <<  -(int)(i + 1) << " ";
      }
    ostr << 0 << endl;
  }

  /* Prikazuje valuaciju u obliku steka, sa oznacenim nivoima odlucivanja */
  void printStack(ostream & ostr) const
  {
    unsigned level  = 0;
    for(unsigned i = 0; i < _stack.size(); i++)
      {
	if(_stack[i].second > level)
	  {
	    ostr << "| ";
	    level++;
	  }
	ostr << intFromLit(_stack[i].first)  << " ";
      }
    ostr << endl;
  }
  
};


/* Funkcije za citanje ulaznog fajla u DIMACS formatu */

/* Funkcija preskace beline i vraca ASCII kod prvog 
   karaktera koji nije belina */
int skipSpaces(istream & istr)
{
  int c;
  while((c = istr.get()) == ' ' || c == '\t' || c == '\n');
  return c;
}

/* Funkcija preskace ostatak linije (zgodno za preskakanje komentara) */
void skipRestOfLine(istream & istr)
{
  while(istr.get() != '\n');
}

/* Funkcija cita DIMACS format iz fajla datog ulaznim tokom istr. Ucitanu 
   formulu smesta u f, a broj varijabli u num_of_vars. */
bool readDIMACS(Formula & f, unsigned & num_of_vars, istream & istr)
{
  unsigned num_of_clauses;
  int c;

  // Preskace komentare
  while((c = skipSpaces(istr)) == 'c')
    skipRestOfLine(istr);

  // Cita liniju p cnf nvars nclauses
  if(c != 'p')
    return false;
  else
    {
      string s;
      istr >> s;
      if(s != "cnf")
	return false;
      
      istr >> num_of_vars;
      istr >> num_of_clauses;
    }

  // Citamo klauze
  for(unsigned i = 0; i < num_of_clauses; i++)
    {
      Clause c;
      int n;
      istr >> n; 
      while(!istr.eof() && !istr.fail() && n != 0)
	{
	  c.push_back(litFromInt(n));
	  istr >> n;
	}
      
      if(istr.eof() || istr.fail())
	return false;

      f.push_back(c);
    }
  return true;
}

/* Klasa Solver predstavlja DPLL zasnovan iterativni SAT resavac */
class Solver {
private:
  /* Stanje resavaca je predstavljeno formulom koju resavamo i trenutnom
     parcijalnom valuacijom (sa definisanim nivoima odlucivanja). U ovoj
     jednostavnoj implementaciji, formula se nece menjati tokom rada
     resavaca, ali u naprednijim implementacijama formula se moze
     menjati dodavanjem novih klauza za koje se ispostavi da slede iz
     postojecih (ucenje klauza),  kao i brisanjem redundantnih klauza 
     (zaboravljanje klauza) */
  Formula _formula;
  Valuation _val;

public:
  Solver(const Formula & f, unsigned num_of_vars)
    :_formula(f),
     _val(num_of_vars)
  {}
  
  /* Funkcija proverava da li postoji konflikt parcijalne valuacije sa
     nekom od klauza. Postavlja i na indeks klauze koje je u konfliktu . */
  bool checkConflict(unsigned & i)
  {
    for(i = 0; i < _formula.size(); i++)
      if(_val.isClauseFalse(_formula[i]))
	{
	  return true;
	}
    return false;
  }

  /* Funkcija proverava da li postoje jedinicne klauza, tj, da li 
     je moguce primeniti jedinicnu propagaciju. Postavlja l na jedinicni
     literal, a i na indeks jedinicne klauze. */
  bool checkUnit(Literal & l, unsigned & i)
  {
    for(i = 0; i < _formula.size(); i++)
      if(_val.isClauseUnit(_formula[i], l))
	{
	  return true;
	}

    return false;
  }

  /* Funkcija donosi odluku i izboru jedinicnog literala koji ce biti
     postavljen na stek primenom Decide pravila. Funkcija vraca false
     ako to nije moguce, tj. ako su sve varijable vec dobile vrednost.
     U okviru ove funkcije se obicno implementiraju razlicite strategije
     izbora varijable i njenog polariteta */
  bool chooseDecisionLiteral(Literal & l)
  {
    Variable v;
    if(!_val.findFirstUndefinedVariable(v))
      return false;
    else
      {
	l = litFromVar(v, P_POSITIVE);
	return true;
      }
  }

  /* Da li je moguce primeniti backtrack pravilo */ 
  bool canBacktrack()
  {
    return _val.currentLevel() > 0;
  }

  /* Primena jedinicne propagacije datog literala */
  void applyUnitPropagation(Literal l)
  {
#ifndef NDEBUG
    cerr << "Unit Propagate: literal: " << intFromLit(l) << endl;
#endif
    
    _val.push(l);
  }

  /* Primena Decide pravila (sa izabranim literalom) */
  void applyDecide(Literal l)
  {
    _val.push(l, true);
    
#ifndef NDEBUG
    cerr << "Decide: level: " 
    	 << _val.currentLevel() 
    	 << " literal: " 
    	 << intFromLit(l) << endl;
#endif
  }


  /* Primena backtrack pravila */
  void applyBacktrack()
  {
    
    Literal l = _val.backtrack();
    
#ifndef NDEBUG
    cerr << "Backtrack: level: " 
    	 << _val.currentLevel()  
    	 << " literal: " 
    	 << intFromLit(l) << endl;
#endif
    _val.push(oppositeLiteral(l));
  }

  /* Resavanje */
  bool solve()
  {
    Literal l;
    unsigned i;

    while(true)
      {

	/* Ako je u konfliktu tada, moramo primeniti backtrack. Ako ovo
	   nije moguce (konflikt na nultom nivou), tada je formula 
	   nezadovoljiva */
	if(checkConflict(i))
	  {
#ifndef NDEBUG
	    cerr << "Conflict: ";
	    printClause(_formula[i], cerr);
#endif
	    if(canBacktrack())
	      {
		applyBacktrack();
		
#ifndef NDEBUG
		cerr << "Stack: ";
		_val.printStack(cerr);
#endif
	      }
	    else
	      return false;
	  }	
	/* Ako nije u konfliktu, tada prvo pokusavamo da primenimo pravilo
	   jedinicne propagacije, ako je to moguce */
	else if(checkUnit(l, i))
	  {
#ifndef NDEBUG
	    cerr << "Unit clause: ";
	    printClause(_formula[i], cerr);
#endif
	    applyUnitPropagation(l);

#ifndef NDEBUG
	    cerr << "Stack: ";
	    _val.printStack(cerr);
#endif
	  }
	/* Ako nije moguce, tada nam ostaje samo da donesemo proizvoljnu
	   odluku (Decide pravilo). Ukoliko to nije moguce znaci da 
	   su sve varijable vec dobile vrednost, a nismo u konfliktu ni
	   sa jednom klauzom -- to znaci da je formula zadovoljena datom
	   valuacijom. */
	else if(chooseDecisionLiteral(l))
	  {
	    applyDecide(l);
#ifndef NDEBUG
	    cerr << "Stack: ";
	    _val.printStack(cerr);
#endif
	  }
	else
	  return true;
      }
  }

  const Valuation & getValuation() const
  {
    return _val;
  }
 
};

int main()
{
  unsigned num_of_vars;
  Formula f;
  
  if(!readDIMACS(f, num_of_vars, cin))
    {
      cerr << "Error reading input file" << endl;
      exit(1);
    }

  Solver solver(f, num_of_vars);

  if(!solver.solve())
    {
      cout << "UNSAT" << endl;
    }
  else
    {
      cout << "SAT" << endl;
      solver.getValuation().printValuation(cout);
    }

  return 0;
}
