/* Skupovi brojeva su predstavljeni tekstualnim fajlovima koji sadrze brojeve
   i imena drugih fajlova. Ako se broj x nalazi u fajlu S, tada broj x pripada 
   skupu S. Ako se ime fajla S1 nalazi u fajlu S2, tada je skup S1 podskup
   skupa S2. Program za dati skup S i dati broj x proverava da
   li x pripada S. */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>


/* NAPOMENA: Red (eng. queue) je struktura podataka nad kojom su 
   definisane sledece operacije:
   1) Dodavanje elementa -- kazemo da je element dodat na kraj reda
      (eng. enqueue() operacija)
   2) Uklanjanje elementa koji je prvi dodat -- kazemo da je 
      element skinut sa pocetka reda (eng. dequeue() operacija)
   3) Ocitavanje vrednosti elementa koji je na pocetku reda (eng.
      front() operacija)

   Red spada u FIFO strukture (eng. First In First Out). Moze se 
   implementirati na vise nacina. Najjednostavniji nacin je da se
   definise kao niz. Medjutim, tada je ogranicen max. broj elemenata
   u redu dimenzijom niza. Zbog toga se obicno pribegava koriscenju
   lista za implementaciju reda, gde se enqueue() operacija svodi
   na dodavanje na kraj liste, a dequeue() operacija se svodi na uklanjanje
   glave liste. Obe operacije se izvode u konstantnom vremenu, pod uslovom
   da se osim pokazivaca na glavu liste cuva i pokazivac na poslednji element
   u listi.

   U ovom zadatku ce red biti koriscen na sledeci nacin: inicijalno se u red
   stavlja ime fajla koji predstavlja dati skup S. Iz reda se zatim u petlji
   uzima sledeci fajl i otvara za citanje. Prilikom citanja, svaki put kada se 
   u fajlu koji se trenutno cita naidje na ime drugog fajla, to ime ce biti
   dodato u red. Kada se zavrsi sa citanjem tekuceg fajla, isti se zatvara,
   skida se sa reda sledeci fajl i citanje se nastavlja u njemu. Ovaj proces
   se ponavlja dok se ne naidje na trazeni broj, ili dok se ne isprazni red.

*/


#define MAX 100   
   
/* Struktura koja predstavlja cvor liste */
typedef struct cvor {
  char ime_fajla[MAX];          /* Sadrzi ime fajla */
  struct cvor *sledeci;  /* pokazivac na sledeci cvor */
} Cvor;


/* Funkcija kreira novi cvor, upisuje u njega etiketu
   i vraca njegovu adresu */
Cvor * napravi_cvor(char * ime_fajla)
{
  Cvor *novi = NULL;
  if((novi = (Cvor *) malloc(sizeof(Cvor))) == NULL)
    {
      fprintf(stderr,"malloc() greska!\n");
      exit(1);
    }
  
  strcpy(novi->ime_fajla, ime_fajla);
  novi->sledeci = NULL;
  return novi;
}


/* Funkcija dodaje na kraj reda novi fajl */
void dodaj_u_red(Cvor **pocetak, Cvor **kraj, char *ime_fajla)
{
  Cvor * novi = napravi_cvor(ime_fajla);
  
  if(*kraj != NULL)
    {
      (*kraj)->sledeci = novi;
      *kraj = novi;
    }
  else /* ako je red prazan */
    {
      *pocetak = novi;
      *kraj = novi;
    } 
}

/* Funkcija skida sa pocetka reda fajl. Ako je poslednji argument
   pokazivac razlicit od NULL, tada u niz karaktera na koji on
   pokazuje upisuje ime fajla koji je upravo skinut sa reda
   dok u suprotnom ne radi nista. Funkcija vraca 0 ako je red
   prazan (pa samim tim nije bilo moguce skinuti vrednost sa 
   reda) ili 1 u suprotnom. */
int skini_sa_reda(Cvor **pocetak, Cvor ** kraj, char * ime_fajla)
{
     Cvor * pomocni;

     if(*pocetak == NULL)
       return 0;
     
     if(ime_fajla != NULL)
       strcpy(ime_fajla, (*pocetak)->ime_fajla);
     
     
     pomocni = *pocetak;
     *pocetak = (*pocetak)->sledeci;
     free(pomocni);
     
     if(*pocetak == NULL)
       *kraj = NULL;
     
     return 1;
}

/* Funkcija vraca pokazivac na string koji sadrzi ime fajla
   na pocetku reda. Ukoliko je red prazan, vraca NULL */
char * pocetak_reda(Cvor *pocetak)
{
  if(pocetak == NULL)
    return NULL;
  else
    return pocetak->ime_fajla;
}

/* Funkcija prikazuje red  */
void prikazi_red(Cvor *pocetak)
{
   for(;pocetak != NULL; pocetak = pocetak->sledeci)
        printf("%s, ", pocetak->ime_fajla);
   printf("\n");
}


/* Funkcija prazni red */
void oslobodi_red(Cvor **pocetak, Cvor **kraj)
{
  Cvor *pomocni;
  
  while(*pocetak != NULL)
    {
      pomocni = *pocetak;
      *pocetak = (*pocetak)->sledeci;
      free(pomocni);
    }
  *kraj = NULL;
}

/* Glavni program */
int main(int argc, char ** argv)
{
  Cvor * pocetak = NULL, * kraj = NULL;
  FILE *in = NULL;
  char ime_fajla[MAX];
  int x, y;
  int pronadjen = 0;

  /* NAPOMENA: Pretpostavlja se da nema cirkularnih zavisnosti, tj. da ne moze
     da se desi slucaj: S2 podskup od S1, S3 podskup S2,...,S1 podskup od Sn,
     u kom slucaju bismo imali beskonacnu petlju. */

  /* Ime fajla i broj se zadaju sa komandne linije */
  if(argc < 3)
    {
      fprintf(stderr, "koriscenje: %s fajl broj\n", argv[0]);
      exit(1);
    }
  
  /* Dodajemo skup S u red */
  dodaj_u_red(&pocetak, &kraj, argv[1]);

  /* Trazeni broj x */
  x = atoi(argv[2]);
  
  /* Dokle god broj nije pronadjen, a ima jos fajlova u redu */
  while(!pronadjen && skini_sa_reda(&pocetak, &kraj, ime_fajla))
    {
      /* Otvaramo sledeci fajl sa reda */
      if((in = fopen(ime_fajla, "r")) == NULL)
	{
	  fprintf(stderr, "Greska prilikom otvaranja fajla: %s\n", ime_fajla);
	  exit(1);
	}
      
      /* Dokle god nismo stigli do kraja fajla i nismo pronasli broj */
      while(!feof(in) && !pronadjen)
	{
	  /* Ako je broj sledeci podatak u fajlu */
	  if(fscanf(in, "%d", &y) == 1)
	    {
	      /* Ako je nadjen trazeni broj */
	      if(x == y)
		{
		  printf("Broj x=%d pripada skupu!\n", x);
		  pronadjen = 1;
		}
	    }
	  /* Ako je ime fajla sledeci podatak u fajlu */
	  else if(fscanf(in, "%s", ime_fajla) == 1)
	    {
	      /* Dodajemo ga u red */
	      dodaj_u_red(&pocetak, &kraj, ime_fajla);
	    }
	    
	}
      /* Zatvaramo fajl */
      fclose(in); 
    }

  /* Ako do kraja nije pronadjen */
  if(!pronadjen)
    printf("Broj x=%d ne pripada skupu!\n", x);


  /* Oslobadjamo red */
  oslobodi_red(&pocetak, &kraj);

  return 0;
}
