#include <stdio.h>
#include <string.h>
#include <ctype.h>

/* Program demonstrira algoritme za racunske operacije nad oznacenim
   celim brojevima u proizvoljnoj osnovi u potpunom komplementu */


/* POMOCNE FUNKCIJE I FUNKCIJE SA PRETHODNIH CASOVA KOJE CEMO
   KORISTITI */

#define MAX_STR 1000

/* Funkcija uzima karakter i prevodi ga u vrednost cifre. Za cifre se
   koriste karakteri '0' - '9' i 'A' - 'Z'  */
unsigned karakter_u_cifru(char c)
{
  return isdigit(c) ? c - '0' : c - 'A' + 10;
}

/* Funkcija uzima vrednost cifre i prevodi je u karakter kojim se ta 
   cifra oznacava */
char cifra_u_karakter(unsigned d)
{
  return d < 10 ? d + '0' : d - 10 + 'A';
}

/* Konvertuje zapis broja x iz osnove b u binarni zapis, koristeci algoritam
   zasnovan na sumi */
void konvertuj_u_binarni(char * x, unsigned b, unsigned * r)
{
  int i;
  unsigned xc;
  *r = 0;
  for(i = 0; x[i] != '\0'; i++)
    {
      xc = karakter_u_cifru(x[i]);
      *r = *r * b + xc;
    }
}

/* Pomocna funkcija obrce string */
void obrni_string(char * s)
{
  int n = strlen(s);
  int i, j;
  for(i = 0, j = n - 1; i < j; i++, j--)
    {
      unsigned t = s[i];
      s[i] = s[j];
      s[j] = t;
    }
}

/* Konvertuje binarni zapis broja x u zapis sa osnovom b, algoritmom
   zasnovanom na deljenju */
void konvertuj_iz_binarnog(unsigned x, unsigned b, char * r)
{
  unsigned i = 0;
  while(x != 0)
    {
      r[i++] = cifra_u_karakter(x % b);
      x = x / b;
    }
  r[i] = '\0';
  obrni_string(r);
}

/* Funkcija dodaje k nula iza zapisa broja (ekvivalentno mnozenju sa b^k) */
void dodaj_k_pratecih_nula(char * x, unsigned k)
{
  int n = strlen(x);
  while(k)
    {
      x[n++] = '0';
      k--;
    }
  x[n] = '\0';   
}

/* Funkcija dodaje k nula ispred zapisa broja (ne utice na vrednost broja) */
void dodaj_k_vodecih_nula(char * x, unsigned k)
{
  int n = strlen(x);
  x[n + k] = '\0';
  while(n)
    {
      n--;
      x[n + k] = x[n]; 
    }
  while(k)
    {
      k--;
      x[k] = '0';
    }
}

/* Funkcija brise vodece nule iz zapisa broja */
void obrisi_vodece_nule(char * x)
{
  int i = 0, j = 0;
  while(x[i] == '0') i++; /* Preskacemo sve vodece nule */

  while(x[i] != '\0') /* Prepisujemo ostale cifre na pocetak */
    x[j++] = x[i++];

  if(j != 0)
    x[j] = '\0';
  else /* Specijalni slucaj: kada su sve cifre nule, treba ostaviti jednu */
    {
      x[0] = '0';
      x[1] = '\0';
    }
}

/* Funkcija dodaje odgovarajuci broj vodecih nula ispred "kraceg" broja, kako
   bi imali isti broj cifara */
void izjednaci_sirine_zapisa(char * x, char * y)
{
  int nx = strlen(x), ny = strlen(y);
  if(nx < ny)
    dodaj_k_vodecih_nula(x, ny - nx);
  else if(nx > ny)
    dodaj_k_vodecih_nula(y, nx - ny);
}

/* Funkcija sabira dva broja u osnovi b. Pretpostavlja se da su brojevi 
   iste duzine i da su predstavljeni stringovima x i y. Rezultat se smesta
   u string na koji pokazuje r, dok se informacija o prekoracenju upisuje u
   lokaciju na koju pokazuje c */
void sabiranje(char * x, char * y, unsigned b, char * r, unsigned * c)
{
  unsigned xc, yc, rc, pc = 0;
  int i;
  int n = strlen(x);
  for(i = n - 1; i >= 0 ; i--)
    {
      xc = karakter_u_cifru(x[i]);
      yc = karakter_u_cifru(y[i]);
      rc = (xc + yc + pc) % b;
      pc = (xc + yc + pc) / b;
      r[i] = cifra_u_karakter(rc);
    }
  r[n] = '\0';
  *c = pc;
}

/* Funkcija oduzima dva broja u osnovi b. Pretpostavlja se da su brojevi 
   iste duzine i da su predstavljeni stringovima x i y. Rezultat se smesta
   u string na koji pokazuje r, dok se informacija o prekoracenju upisuje u
   lokaciju na koju pokazuje c */
void oduzimanje(char * x, char * y, unsigned b, char * r, unsigned * c)
{
  unsigned xc, yc, rc, pc = 0;
  int i;
  int n = strlen(x);
  for(i = n - 1; i >= 0 ; i--)
    {
      xc = karakter_u_cifru(x[i]);
      yc = karakter_u_cifru(y[i]);
      unsigned p = xc < (yc + pc);
      rc = !p ? xc - (yc + pc) : xc + b - (yc + pc);
      pc = p;
      r[i] = cifra_u_karakter(rc);
    }
  r[n] = '\0';
  *c = pc;
}

/* FUNKCIJE ZA RAD SA OZNACENIM BROJEVIMA */

/* Funkcija ispituje da li je dati broj nenegativan. */
int nenegativan(char *x , unsigned b)
{
  /* Slucaj parne osnove (gledamo samo prvu cifru) */
  if(b % 2 == 0)
    {
      unsigned c = karakter_u_cifru(x[0]);
      if(c < b / 2)
	return 1;
      else
	return 0;
    }
  else /* Slucaj neparne osnove (trazimo prvu cifru razlicitu od b/2) */
    {
      int i;
      int n = strlen(x);
      for(i = 0; i < n; i++)
	{
	  unsigned c = karakter_u_cifru(x[i]);
	  if(c < b / 2)
	    return 1;
	  else if(c > b / 2)
	    return 0;
	}
      return 1;  /* Ako su sve cifre jednake b/2, tada je broj pozitivan */
    }
}

/* Funkcija implementira algoritam promene znaka, tj. odredjuje zapis
   broja -x u potpunom komplementu u osnovi b */
void promeni_znak(char * x, unsigned b, char * r)
{
  unsigned n = strlen(x);
  int i;
  unsigned pc = 1;
  for(i = n - 1; i >= 0; i--)
    {
      /* Svaku cifru najpre zamenimo njenim komplementom do 
	 najvece cifre u osnovi b (tj. do b - 1) */
      unsigned c = b - 1 - karakter_u_cifru(x[i]);

      /* Zatim dodamo pc (koji je inicijalno 1, tj. na najnizoj
	 poziciji svakako dodajemo 1) */
      unsigned rc = (c + pc) % b;

      /* Ako se dodavanjem pc dobija b, tada ce cifra biti 0, ali
	 cemo imati propagaciju jedinice na sledecu poziciju */
      pc = (c + pc) / b;

      /* Upisujemo cifru u rezultat */
      r[i] = cifra_u_karakter(rc);
    }
  /* Na kraju jos upisujemo terminirajucu nulu */
  r[n] = '\0';
}

/* Funkcija odredjuje zapis broja x u potpunom komplementu u osnovi b
   sa najmanjim mogucim brojem cifara */
void odredi_zapis(int x, unsigned b, char * r)
{
  /* Najpre ispitujemo da li je broj negativan. Ako jeste, pamtimo
     tu informaciju (neg = 1) i odredjujemo apsolutnu vrednost
     broja */
  int neg = 0;
  unsigned aps_x;
  if(x < 0)
    {
      aps_x = -x;
      neg = 1;
    }
  else
    aps_x = x;

  /* Odredjujemo zapis apsolutne vrednosti broja x u osnovi b
     kao neoznacenog broja */
  char s[MAX_STR];
  konvertuj_iz_binarnog(aps_x, b, s);

  /* Ako je dobijeni zapis zaista zapis nenegativnog broja, to
     znaci da se broj |x| moze zapisati sa toliko cifara u
     potpunom komplementu u osnovi b. Ako zapis predstavlja
     negativan broj, tada treba dopisati jednu vodecu nulu,
     i to ce biti zapis sa minimalnim brojem cifara */
  if(!nenegativan(s, b))
    {
      dodaj_k_vodecih_nula(s, 1);
    }

  /* Ako je broj bio negativan, tada treba zapisu s promeniti
     znak */
  if(neg)
    {
      promeni_znak(s, b, r);
      /* Specijalni slucaj: ako je u pitanju parna osnova, a x je
	 najmanji negativan broj koji se moze zapisati sa n cifara,
	 tada ce za |x| biti potrebna n + 1 cifra, sto ce nam nakon
	 promene znaka dati jednu cifru viska u zapisu broja x. Ova
	 cifra je najveca cifra u toj osnovi (b-1), a iza nje ce biti
	 cifra koja je >= b /2, pa cemo moci da skratimo ovaj zapis
	 za tu jednu cifru */
      if(b % 2 == 0 && karakter_u_cifru(r[0]) == b-1 && karakter_u_cifru(r[1]) >= b/2)
	{
	  /* Prepravljamo tu vodecu cifru u nulu, a zatim pozivamo funkciju
	     za brisanje 1 vodece nule */
	  r[0] = '0';
	  obrisi_vodece_nule(r);
	}
    }
  else /* U suprotnom samo vracamo dobijeni zapis */
    strcpy(r, s);
}

/* Funkcija dodaje k cifara b - 1, gde je b osnova  */
void dodaj_k_vodecih_najvecih_cifara(char * x, unsigned b, unsigned k)
{
  int n = strlen(x);
  x[n + k] = '\0';
  while(n)
    {
      n--;
      x[n + k] = x[n]; 
    }
  while(k)
    {
      k--;
      x[k] = cifra_u_karakter(b - 1);
    }
}

/* Funkcija prosiruje zapis broja u potpunom komplementu u osnovi b
   tako sto mu dodaje jos k vodecih cifara (0 za nenegativne, a 
   b - 1 za negativne brojeve) */
void prosiri_oznaceni_zapis(char * x, unsigned b, unsigned k)
{
  if(nenegativan(x, b))
    dodaj_k_vodecih_nula(x, k);
  else
    dodaj_k_vodecih_najvecih_cifara(x, b, k);
}

/* Funkcija sabira brojeve u potpunom komplementu u osnovi b */
void saberi_oznacene(char * x, char * y, unsigned b, char * r, unsigned * c)
{
  unsigned c1;
  /* Sabiramo ih kao neoznacene brojeve, ignorisuci prekoracenje */
  sabiranje(x, y, b, r, &c1);

  /* Oznaceno prekoracenje dobijamo tako sto ispitujemo znakove
     sabiraka i dobijenog zbira */
  if((nenegativan(x,b) && nenegativan(y,b) && !nenegativan(r,b)) ||
     (!nenegativan(x,b) && !nenegativan(y,b) && nenegativan(r,b)))
    *c = 1;
  else
    *c = 0;
}

/* Funkcija oduzima brojeve u potpunom komplementu u osnovi b */
void oduzmi_oznacene(char * x, char * y, unsigned b, char * r, unsigned * c)
{
  unsigned c1;
  /* Oduzimamo ih kao neoznacene brojeve, ignorisuci prekoracenje */
  oduzimanje(x, y, b, r, &c1);

  /* Oznaceno prekoracenje dobijamo tako sto ispitamo znakove 
     deljenika, delioca i razlike */
  if((!nenegativan(x,b) && nenegativan(y,b) && nenegativan(r,b)) ||
     (nenegativan(x,b) && !nenegativan(y,b) && !nenegativan(r,b)))
    *c = 1;
  else
    *c = 0;
}


int main()
{
  /*
  char r[1000];
  unsigned b;
  int x;

  scanf("%u", &b);
  scanf("%d", &x);
  odredi_zapis(x, b, r);
  printf("%s\n", r);
  */

  char x[MAX_STR], y[MAX_STR], r[MAX_STR];
  unsigned b, c;

  scanf("%u", &b);
  scanf("%s %s", x, y);

  saberi_oznacene(x, y, b, r, &c);
  printf("Zbir: %s (%d)\n", r, c);
  oduzmi_oznacene(x, y, b, r, &c);
  printf("Razlika: %s (%d)\n", r, c);
  
  
  return 0;
}
