/* Program broji pojavljivanje svake od reci u tekstu i ispisuje reci i
   njihove frekvencije u opadajucem poretku po frekvencijama */

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <ctype.h>

/* Makimalna duzina imena etikete */
#define MAX_ETIKETA 32

/* Definisemo stanja prilikom citanja html datoteke */
#define VAN_ETIKETE 1
#define U_ETIKETI 2

/* Struktura koja predstavlja cvor stabla */
typedef struct cvor {
char etiketa[MAX_ETIKETA];   /* Ime etikete */
int brojac;                  /* Brojac pojavljivanja etkete u tekstu */
struct cvor *levi;           /* Pokazivaci na levo */
struct cvor *desni;          /* i desno podstablo */
} Cvor;


/* Funkcija dodaje ime html etikete u binarno stablo,
   u rastucem leksikografskom poretku. Funkcija vraca
   koren stabla nakon modifikacije. U slucaju da je 
   etiketa sa istim imenom vec u stablu, uvecava se 
   brojac pojavljivanja. */ 
Cvor * dodaj_leksikografski(Cvor *koren, char *etiketa)
{
    int cmp;

    /* Izlaz iz rekurzije */
    if(koren == NULL)
    {
         if((koren = malloc(sizeof(Cvor))) == NULL)
          {
            fprintf(stderr,"malloc() greska\n");
            exit(1);
          }

         /* Ime etikete se kopira u novi cvor, a brojac
            se inicijalizuje na 1 (prvo pojavljivanje) */
         strcpy(koren->etiketa, etiketa);
         koren->brojac = 1;
         koren->levi = NULL;
         koren->desni = NULL;
         return koren;
    }

    /* Rekurzivni pozivi */
   if((cmp = strcmp(koren->etiketa,etiketa)) < 0)
     koren->desni = dodaj_leksikografski(koren->desni, etiketa);
   else if(cmp > 0)
     koren->levi = dodaj_leksikografski(koren->levi, etiketa);
   else  
     koren->brojac++; /* uvecanje brojaca za vec prisutne etikete */

   return koren;
}

/* Funkcija cita ulazni fajl i iz njega izdvaja sve
   otvorene html etikete (npr. <html>, <br> itd, ali ne
   i </html>, </body> itd.) i njihova imena (bez znakova
   (< i > kao i bez eventualnih atributa) ubacuje u stablo
   u leksikografskom poretku. Tom prilikom ce se prebrojati
   i pojavljivanja svake od etiketa. Funkcija vraca pokazivac
   na koren kreiranog stabla. */
Cvor * ucitaj(FILE *f)
{
Cvor * koren = NULL;
int c;
int stanje = VAN_ETIKETE;
int i;
char etiketa[MAX_ETIKETA];

/* NAPOMENA: prilikom izdvajanja etiketa pretpostavlja se da je
   html datoteka sintaksno ispravna, kao i da nakon znaka '<' 
   nema belina. Zato se otvorene etikete mogu lako prepoznati
   tako sto pronadjemo znak '<' i nakon toga izdvojimo sva 
   slova koja slede. Ako postoji neprazan niz slova nakon znaka
   '<', tada je to upravo ime etikete. Ako ne, tada je verovatno
   u pitanju zatvorena etiketa, npr. </html>, pa je ignorisemo. */

while((c = fgetc(f)) != EOF)
{
   switch(stanje)
   {
      /* u ovom stanju trazimo prvu sledecu pojavu znaka '<' */
      case VAN_ETIKETE:
           if(c == '<')
             {
                stanje = U_ETIKETI; /* sada smo u etiketi */
                i = 0;
             }
             break;
      /* u ovom stanju citamo slova koja slede, i nakon toga 
         ubacujemo procitanu etiketu u stablo */
      case U_ETIKETI:
           /* Ako je slovo, i nismo prekoracili duzinu niza
              dodajemo slovo u niz */
           if(isalpha(c) && i < MAX_ETIKETA - 1)
             etiketa[i++] = c;
            /* u suprotnom se vracamo u stanje VAN_ETIKETE */
           else {
             stanje = VAN_ETIKETE;
             /* Ako je niz slova nakon '<' bio neprazan... */
             if(i > 0)
             {
                etiketa[i]='\0';
                /* ubacujemo procitanu etiketu u stablo */
                koren = dodaj_leksikografski(koren, etiketa);
             }
           }
           break;
    }

}
 
return koren;

}

/* Funkcija dodaje u stablo etiketu ciji je broj pojavljivanja
   poznat (broj), i to u opadajucem poretku po broju pojavljivanja.
   Funkcija vraca pokazivac na koren tako modifikovanog stabla */
Cvor * dodaj_po_broju(Cvor * koren, char * etiketa, int broj)
{

    /* Izlaz iz rekurzije */
    if(koren == NULL)
    {
         if((koren = malloc(sizeof(Cvor))) == NULL)
          {
            fprintf(stderr,"malloc() greska\n");
            exit(1);
          }

         /* Kopiramo u novi cvor ime i broj pojavljivanja etikete */ 
         strcpy(koren->etiketa, etiketa);
         koren->brojac = broj;
         koren->levi = NULL;
         koren->desni = NULL;
         return koren;
    }

    /* NAPOMENA: s obzirom da dve ili vise etiketa mogu imati isti broj
       pojavljivanja, etiketa se mora dodavati u stablo, cak i ako ima
       isti broj pojavljivanja sa korenom. Zato cemo u levo podstablo
       dodavati etikete koje imaju veci broj pojavljivanja od korena,
       dok cemo u desno podstablo dodavati etikete koje imaju manji
       ili jednak broj pojavljivanja od korena. */

   /* Rekurzivni pozivi */
   if(koren->brojac >= broj)
     koren->desni = dodaj_po_broju(koren->desni, etiketa, broj);
   else if(koren->brojac < broj)
     koren->levi = dodaj_po_broju(koren->levi, etiketa, broj);

   return koren;

}

/* Funkcija pretpostavlja da je novo stablo ili prazno, ili 
   uredjeno prema opadajucem poretku broja pojavljavanja
   etiketa. Funkcija u takvo stablo rekurzivno dodaje sve
   etikete iz starog stabla, i vraca koren na tako modifikovano
   novo stablo. */
Cvor * resortiraj_stablo(Cvor * staro, Cvor * novo)
{

      /* Izlaz iz rekurzije - nemamo sta da dodamo u novo stablo */
      if(staro == NULL)
        return novo;

      /* Dodajemo etiketu iz korena starog stabla u novo stablo,
         sortirano opadajuce prema broju pojavljivanja */
      novo = dodaj_po_broju(novo, staro->etiketa, staro->brojac);

      /* Rekurzivno dodajemo u novo stablo sve cvorove iz levog
         i desnog podstabla starog stabla. */
      novo = resortiraj_stablo(staro->levi, novo);
      novo = resortiraj_stablo(staro->desni, novo);

      return novo;
}

/* Ispis stabla - s leva na desno */
void ispisi_stablo(Cvor * koren)
{
    if(koren == NULL)
     return;

    ispisi_stablo(koren->levi);
    printf("%s: %d\n", koren->etiketa, koren->brojac);
    ispisi_stablo(koren->desni);

}

/* Oslobadjanje dinamicki alociranog prostora */
void oslobodi_stablo(Cvor *koren)
{
  if(koren == NULL)
     return;

 oslobodi_stablo(koren->levi);
 oslobodi_stablo(koren->desni);
 free(koren);
}


/* glavni program */
int main(int argc, char **argv)
{

  FILE *in = NULL;     /* Ulazni fajl */
  Cvor * koren = NULL; /* Stablo u leksikografskom poretku */
  Cvor * resortirano = NULL;  /* stablo u poretku po broju pojavljivanja */


  /* Mora se zadati makar ime ulaznog fajla na komandnoj liniji.
     Opciono se kao drugi argument moze zadati opcija "-b"
     sto dovodi do ispisa po broju pojavljivanja (umesto leksikografski) */
  if(argc < 2)
  {
     fprintf(stderr, "koriscenje: %s ime_fajla [-b]\n", argv[0]);
     exit(1);
  }

  /* Otvaramo fajl */
  if((in = fopen(argv[1], "r")) == NULL)
  {
     fprintf(stderr, "fopen() greska\n");
     exit(1);
  }

  /* Formiramo leksikografsko stablo etiketa */
  koren = ucitaj(in);

  /* Ako je korisnok zadao "-b" kao drugi argument, tada
     se vrsi resortiranje, i ispisuje izvestaj sortiran
     opadajuce po broju pojavljivanja */
  if(argc == 3 && strcmp(argv[2], "-b") == 0)
    {
         resortirano = resortiraj_stablo(koren, resortirano);
         ispisi_stablo(resortirano);
    }
  /* U suprotnom se samo ispisuje izvestaj u leksikografskom poretku */
  else
       ispisi_stablo(koren);


/* Oslobadjamo stabla */
oslobodi_stablo(koren);
oslobodi_stablo(resortirano);

return 0;

}
