#include <stdio.h>
#include <stdlib.h>
#include <time.h>

/* NAPOMENA: Primer ilustruje sortiranje liste. Lista moze biti 
   sortirana na dva nacina. Prvi nacin je da se premestaju cvorovi 
   u listi (sto zahteva prevezivanje pokazivaca), a drugi je da se 
   premestaju  vrednosti  koje  cvorovi  sadrze.  Prvi  nacin  je 
   komplikovaniji, ali je cesto efikasniji, narocito ako su objekti 
   koji se cuvaju kao podaci u cvorovima veliki. 

   U oba slucaja se moze koristiti rekurzija, a moze se realizovati
   i iterativno. U ovom primeru ilustrujemo oba nacina sortiranja
   lista (prvi rekurzivno, a drugi iterativno) */

/* Struktura koja predstavlja cvor liste */
typedef struct cvor {
int vrednost;          /* podatak koji cvor sadrzi */
struct cvor *sledeci;  /* pokazivac na sledeci cvor */
} Cvor;


/* Pomocna funkcija koja kreira cvor. Funkcija vrednost
   novog cvora inicijalizuje na broj, dok pokazivac na
   sledeci cvor u novom cvoru postavlja na NULL. Funkcija 
   ispisuje poruku o gresci i prekida program u slucaju
   da malloc() funkcija ne uspe da alocira prostor. 
   Funkcija vraca pokazivac na novokreirani cvor */
Cvor * napravi_cvor(int broj)
{
     Cvor *novi = NULL;
     if((novi = (Cvor *) malloc(sizeof(Cvor))) == NULL)
       {
           fprintf(stderr,"malloc() greska!\n");
           exit(1);
       }

     novi->vrednost = broj;
     novi->sledeci = NULL;
     return novi;
}

/* Funkcija dodaje novi cvor na pocetak liste. Funkcija
   kreira novi cvor koriscenjem funkcije napravi_cvor().
   Funkcija vraca pokazivac na novu glavu liste */
Cvor * dodaj_na_pocetak_liste(Cvor *glava, int broj)
{
    Cvor * novi = napravi_cvor(broj);
    novi->sledeci = glava;
    return novi; 
}


/* Pomocna rekurzivna funkcija koja prikazuje
   listu. Definisana je kao static, cime se 
   sprecava njeno koriscenje od strane funkcija
   koje nisu definisane u ovom fajlu */
static void prikazi_listu_r(Cvor *glava)
{

/* Izlaz iz rekurzije */
if(glava == NULL) return;

/* Prikaz glave */
printf("%d ", glava->vrednost);

/* Prikaz repa (rekurzivnim pozivom) */
prikazi_listu_r(glava->sledeci);


}

/* Funkcija prikazuje elemente liste pocev od glave 
   ka kraju liste. Koristi rekurzivnu funkciju 
   prikazi_listu_r(). Ova funkcija prosto dodaje 
   uglaste zagrade i novi red na kraju ispisa. */
void prikazi_listu(Cvor *glava)
{
   putchar('[');
   
   prikazi_listu_r(glava); 
  
   putchar(']');

   putchar('\n');
}

/* Funkcija oslobadja dinamicku memoriju zauzetu 
   od strane liste. Funkcija koristi rekurziju. */
Cvor * oslobodi_listu(Cvor *glava)
{
  /* Izlaz iz rekurzije */
  if(glava == NULL) return NULL;

  /* Oslobadjamo rep liste */
  oslobodi_listu(glava->sledeci);

  /* Oslobadjamo glavu liste */
  free(glava);

  return NULL;
}

Cvor * dodaj_cvor_sortirano(Cvor * glava, Cvor * novi)
{
   /* u slucaju da je lista prazna, ili da je vrednost
      glave veca ili jednaka od vrednosti broja koji
      umecemo, tada se novi cvor umece na pocetak liste. */
   if(glava == NULL || glava->vrednost >= novi->vrednost) 
     {
        novi->sledeci = glava;
        return novi;
     }

   /* U slucaju da je lista neprazna, i da je vrednost 
      glave manja od vrednosti broja koji umecemo u listu,
      tada je sigurno da se novi element umece DESNO od glave,
      tj. u rep liste. Dakle, mozemo rekurzivno pozvati istu
      funkciju za rep liste. S obzirom da po induktivnoj 
      pretpostavci rep nakon toga ostaje sortiran, a svi
      elementi u repu ukljucujuci i element koji je dodat
      su jednaki ili veci od glave, tada ce i cela lista
      biti sortirana. Povratna vrednost rekurzivnog poziva
      predstavlja adresu glave modifikovanog repa liste,
      pa se ta adresa mora sacuvati u pokazivacu na sledeci 
      u glavi liste. */
   glava->sledeci = dodaj_cvor_sortirano(glava->sledeci, novi);

   /* vracamo staru - novu glavu */
   return glava;
}

/* Sortira listu i vraca adresu glave liste. Lista se sortira
   rekurzivno, prevezivanjem cvorova. */
Cvor * sortiraj_listu(Cvor * glava)
{
   Cvor * pomocni;

   /* Izlaz iz rekurzije */
   if(glava == NULL)
     return NULL;

   /* Odvajamo cvor glave iz liste, a za glavu proglasavamo
      sledeci cvor */
   pomocni = glava;
   glava = glava->sledeci;
   pomocni->sledeci = NULL;

   /* Sortiramo ostatak liste */
   glava = sortiraj_listu(glava);
   glava = dodaj_cvor_sortirano(glava, pomocni);

   /* Vracamo novu glavu */
   return glava;
}

/* Funkcija sortira listu i vraca adresu glave liste. Lista se
   sortira iterativno, zamenom vrednosti u cvorovima. */
Cvor * sortiraj_listu2(Cvor * glava)
{
   int t;
   Cvor * pi, * pj, * min;

   /* Slucaj prazne liste */
   if(glava == NULL)
    return glava;

   /* Simuliramo selection sort */
   for(pi = glava; pi->sledeci != NULL ; pi = pi->sledeci)
   {
      min = pi;
      for(pj = pi->sledeci; pj != NULL; pj = pj->sledeci)
       {
          if(pj->vrednost < min->vrednost)
           min = pj;
       }
     t = min->vrednost;
     min->vrednost = pi->vrednost;
     pi->vrednost = t;
   }

   /* Vracamo novu glavu */
   return glava;
}



/* test program */
int main()
{

Cvor *glava = NULL;
int n = 10;

/* Inicijalizujemo sekvencu slucajnih brojeva. */
srand(time(NULL));

/* Generisemo n slucajnih brojeva i dodajemo ih na pocetak liste */
while(n--)
{
   glava = dodaj_na_pocetak_liste(glava, (int) (100 * (rand()/(double)RAND_MAX)));
}

/* Prikazujemo generisanu listu */
printf("Generisana lista:\n");
prikazi_listu(glava);

/* Sortiramo listu */
glava = sortiraj_listu(glava);

/* Prikazujemo listu nakon sortiranja */
printf("Lista nakon sortiranja:\n");
prikazi_listu(glava);

/* Oslobadjamo listu */
glava = oslobodi_listu(glava);

return 0;

}
