#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>

#define MAX 1024


/* NAPOMENA: Binarna stabla se veoma efikasno mogu koristiti za brojanje
   pojavljivanja svakog pojedinacnog objekta u datom multiskupu objekata.
   Sledeci primer demonstrira kako se pomocu binarnog stabla moze brojati
   pojavljivanje svake od reci u tekstu.
*/


typedef struct cvor {
char rec[MAX]; /* rec */
int brojac;   /* broj pojavljivanja reci */
struct cvor * levi; 
struct cvor * desni; 

} Cvor;

/* Pomocna funkcija za kreiranje cvora. Funkcija vraca 
   adresu novokreiranog cvora */
Cvor * napravi_cvor (char * rec)
{

/* dinamicki kreiramo cvor */
Cvor * novi = (Cvor *) malloc (sizeof(Cvor));

/* u slucaju greske ... */
if (novi == NULL)
{
  fprintf (stderr,"malloc() greska\n");
  exit (1);
}

/* inicijalizacija */
strcpy(novi->rec, rec);
novi->brojac = 1;
novi->levi = NULL;
novi->desni = NULL;

/* vracamo adresu novog cvora */
return novi;

}

/* Funkcija dodaje novi cvor u stablo sa datim korenom.
   Ukoliko rec vec postoji u stablu, uvecava dati brojac.
   Cvor se kreira funkcijom napravi_cvor(). Funkcija 
   vraca koren stabla nakon ubacivanja novog cvora. */
Cvor * dodaj_u_stablo (Cvor * koren, char *rec)
{

/* izlaz iz rekurzije: ako je stablo bilo prazno,
   novi koren je upravo novi cvor */
if (koren == NULL)
  return napravi_cvor (rec);

/* Ako je stablo neprazno, i koren sadrzi leksikografski 
   manju rec od date reci, broj se umece u desno podstablo, 
   rekurzivnim pozivom */
if (strcmp(koren->rec, rec) < 0)
  koren->desni = dodaj_u_stablo (koren->desni, rec);
/* Ako je stablo neprazno, i koren sadrzi vecu vrednost
   od datog broja, broj se umece u levo podstablo,
   rekurzivnim pozivom */
else if (strcmp(koren->rec, rec) > 0)
  koren->levi = dodaj_u_stablo (koren->levi, rec);
else
/* Ako je data rec vec u stablu, samo uvecavamo brojac. */
  koren->brojac++;

/* Vracamo koren stabla */
return koren;

}

/* Funkcija pronalazi najfrekventniju rec, tj. cvor ciji
   brojac ima najvecu vrednost. Funkcija vraca NULL, ako
   je stablo prazno, odnosno adresu cvora koji sadrzi 
   najfrekventniju rec u suprotnom. */
Cvor * nadji_najfrekventniju (Cvor * koren)
{
Cvor *max, *max_levi, *max_desni;

/* Izlaz iz rekurzije */
if (koren == NULL)
 return NULL;
 
/* Odredjujemo najfrekventnije reci u levom i desnom podstablu */
max_levi = nadji_najfrekventniju(koren->levi);
max_desni = nadji_najfrekventniju(koren->desni);

/* Odredjujemo MAX(koren, max_levi, max_desni) */
max = koren;
if(max_levi != NULL && max_levi->brojac > max->brojac)
 max = max_levi;
if(max_desni != NULL && max_desni->brojac > max->brojac)
 max = max_desni;

/* Vracamo adresu cvora sa najvecim brojacem */
return max;

}

/* Prikazuje reci u leksikografskom poretku, kao i broj pojava svake
   od reci */
void prikazi_stablo(Cvor *koren)
{
  if(koren == NULL)
   return;

  prikazi_stablo(koren->levi);
  printf("%s: %d\n", koren->rec, koren->brojac);
  prikazi_stablo(koren->desni);

}


/* Funkcija oslobadja prostor koji je alociran za 
   cvorove stabla. Funkcija vraca NULL, zato sto je
   nakon oslobadjanja stablo prazno. */
void oslobodi_stablo (Cvor *koren)
{
/* Izlaz iz rekurzije */
if(koren == NULL)
  return;

oslobodi_stablo (koren->levi); 
oslobodi_stablo (koren->desni);
free(koren);

}

/* Funkcija ucitava sledecu rec iz fajla i upisuje je
   u niz na koji pokazuje rec, maksimalne duzine max.
   Funkcija vraca EOF ako nema vise reci, 0 u suprotnom.
   Rec je niz alfabetskih karaktera.*/ 
int sledeca_rec(FILE *f, char * rec, int max)
{
  int c;
  int i = 0;

  /* Dokle god ima mesta za jos jedan karakter u stringu,
     i dokle god nismo stigli do kraja fajla... */
  while(i < max - 1 && (c = fgetc(f)) != EOF)
  { 
     /* Ako je slovo, ubacujemo ga u rec */
     if(isalpha(c))
       rec[i++] = tolower(c); 
     /* U suprotnom, ako smo procitali bar jedno slovo
        prethodno, onda imamo rec. Inace idemo na sledecu
        iteraciju */
     else if(i > 0)
     	break;
  }

/* Zatvaramo string */
rec[i] = '\0';

/* Vracamo 0 ako imamo rec, EOF u suprotnom */
return i > 0 ? 0 : EOF;
}


/* test program */
int main(int argc, char **argv)
{

Cvor *koren = NULL, *max;
FILE *f;
char rec[MAX];

/* Proveravamo da li je navedeno ime datoteke */
if(argc < 2)
{
  fprintf(stderr, "Morate uneti ime datoteke sa tekstom!\n");
  exit(0);
}

/* Otvaramo datoteku */
if((f = fopen(argv[1], "r")) == NULL)
{
  fprintf(stderr, "fopen() greska\n");
  exit(1);
}

/* Ucitavamo reci iz datoteke. Rec je niz alfabetskih karaktera. */
while(sledeca_rec(f, rec, MAX) != EOF)
{
  koren = dodaj_u_stablo(koren, rec);
}
/* Zatvaramo datoteku */
fclose(f);

/* Prikazujemo sve reci i brojeve njihovih pojavljivanja */
prikazi_stablo(koren);

/* Pronalazimo najfrekventniju rec */
if((max = nadji_najfrekventniju(koren)) == NULL)
 printf("U tekstu nema reci!\n");
else
 printf("Najcesca rec %s (pojavljuje se %d puta)\n", max->rec, max->brojac);

/* Oslobadjamo stablo */
oslobodi_stablo(koren);

return 0;
}
