#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#define MAX 1024


/* NAPOMENA: Mapa je apstraktna struktura podataka koja u
   sebi sadrzi parove oblika (kljuc, vrednost). Pri tom su
   i kljuc i vrednost unapred odredjenog (ne obavezno istog)
   tipa (Na primer, kljuc nam moze biti string koji predstavlja
   ime studenta, a vrednost je npr. broj koji predstavlja njegov
   prosek). Operacije koje mapa mora efikasno da podrzava su:
   - dodavanje para (kljuc, vrednost) u mapu.
   - brisanje para (kljuc, vrednost) iz mape.
   - pronalazenje vrednosti za dati kljuc.
   - promena vrednosti za dati kljuc.

   Jedan od najcescih nacina implementacije mape je preko binarnog
   stabla pretrage. Svaki cvor ovog stabla sadrzace odgovarajuci
   par (kljuc, vrednost), tj. svaki cvor ce sadrzati dva podatka.
   Pri tom ce poredak u stablu biti odredjen poretkom koji je 
   definisan nad kljucevima. Ovim se postize da se pretraga po
   kljucu moze obaviti na uobicajen nacin, kao i sve ostale 
   navedene operacije. 

   Jasno je da ovako implementirana mapa nece efikasno podrzavati
   operaciju pretrage po vrednosti (npr. naci sve kljuceve koji imaju
   datu vrednost), zato sto poredak u stablu nije ni na koji nacin u
   vezi sa poretkom medju vrednostima. Medjutim, u mapi se najcesce
   i ne zahteva takva operacija.

   Sledi primer koji demonstrira implementaciju mapa pomocu binarnog
   stabla.
*/


/* Struktura koja predstavlja cvor stabla */
typedef struct cvor {
char naziv[MAX];
int cena;
struct cvor * levi;
struct cvor * desni;
} Cvor;

/* Funkcija kreira novi cvor i vraca njegovu adresu */
Cvor * napravi_cvor (char * naziv, int cena)
{

/* dinamicki kreiramo cvor */
Cvor * novi = (Cvor *) malloc (sizeof(Cvor));

/* u slucaju greske ... */
if (novi == NULL)
{
  fprintf (stderr,"malloc() greska\n");
  exit (1);
}

/* inicijalizacija */
strcpy(novi->naziv, naziv);
novi->cena = cena;
novi->levi = NULL;
novi->desni = NULL;

/* vracamo adresu novog cvora */
return novi;

}

/* Funkcija dodaje novi cvor u stablo sa datim korenom.
   U cvor se upisuje vrednost (naziv, cena). Ukoliko naziv
   vec postoji u stablu, tada se azurira njegova cena.
   Cvor se kreira funkcijom napravi_cvor(). Funkcija 
   vraca koren stabla nakon ubacivanja novog cvora. */
Cvor * dodaj_u_stablo (Cvor * koren, char * naziv, int cena)
{

/* izlaz iz rekurzije: ako je stablo bilo prazno,
   novi koren je upravo novi cvor */
if (koren == NULL)
  return napravi_cvor (naziv, cena);

/* Ako je stablo neprazno, i koren sadrzi naziv koji je 
   leksikografski manji od datog naziva, vrednost se umece 
   u desno podstablo, rekurzivnim pozivom */
if (strcmp(koren->naziv, naziv) < 0)
  koren->desni = dodaj_u_stablo (koren->desni, naziv, cena);
/* Ako je stablo neprazno, i koren sadrzi naziv koji je
   leksikografski  veci od datog naziva, vrednost se umece 
   u levo podstablo, rekurzivnim pozivom */
else if (strcmp(koren->naziv, naziv) > 0)
  koren->levi = dodaj_u_stablo (koren->levi, naziv, cena);
/* Ako je naziv korena jednak nazivu koja se umece, tada se samo
   azurira cena. */
else
  koren->cena = cena;

/* Vracamo koren stabla */
return koren;

}

/* Funkcija pretrazuje binarno stablo. Ukoliko 
   pronadje cvor sa vrednoscu naziva koja je jednaka
   datom nazivu, vraca adresu tog cvora. U 
   suprotnom vraca NULL */
Cvor * pretrazi_stablo (Cvor * koren, char * naziv)
{

/* Izlaz iz rekurzije: ako je stablo prazno,
   tada trazeni broj nije u stablu */
if (koren == NULL)
 return NULL;

/* Ako je stablo neprazno, tada se pretrazivanje
   nastavlja u levom ili desnom podstablu, u 
   zavisnosti od toga da li je trazeni naziv 
   respektivno manji ili veci od vrednosti naziva
   korena. Ukoliko je pak trazeni naziv jednak
   nazivu korena, tada se vraca adresa korena. */
if (strcmp(koren->naziv, naziv) < 0)
  return pretrazi_stablo (koren->desni, naziv);
else if (strcmp(koren->naziv, naziv) > 0)
  return pretrazi_stablo (koren->levi, naziv);
else 
  return koren;

}

/* cvor liste */
typedef struct cvor_liste {
  char naziv[MAX];
  int cena;
  struct cvor_liste * sledeci;
} Cvor_liste;

/* Pomocna funkcija koja kreira cvor liste */ 
Cvor_liste * napravi_cvor_liste(char * naziv, int cena)
{
  Cvor_liste * novi;

  if((novi = malloc(sizeof(Cvor_liste))) == NULL)
  {
    fprintf(stderr, "malloc() greska\n");
    exit(1);
  }
  
  strcpy(novi->naziv, naziv);
  novi->cena = cena;
  novi->sledeci = NULL;
  return novi;
}

/* Funkcija pronalazi sve nazive cija je cena manja ili jednaka od date,
   i formira listu koja sadrzi nadjene parove (naziv, cena) u leksikogra-
   fskom poretku po nazivima. Prilikom pocetnog poziva, treci argument
   treba da bude NULL. Funkcija obilazi stablo sa desna u levo, kako bi
   se prilikom dodavanja na pocetak liste poslednji dodao onaj koji je
   leksikografski najmanji (tj. on ce biti na pocetku). */
Cvor_liste * pronadji_manje (Cvor * koren, int cena, Cvor_liste * glava)
{
   if(koren == NULL)
    return glava;

  /* Dodajemo na pocetak liste sve cvorove desnog podstabla cija je cena
     manja od date. */
  glava = pronadji_manje(koren->desni, cena, glava);
 
  /* Dodajemo koren u listu, ako mu je cena manja od date */
  if(koren->cena <= cena)
   {
     Cvor_liste * novi = napravi_cvor_liste(koren->naziv, koren->cena);
     novi->sledeci = glava;
     glava = novi;
   }

  /* Dodajemo na pocetak liste sve cvorove levog podstabla cija je cena
     manja od date */
  glava = pronadji_manje(koren->levi, cena, glava);

  /* Vracamo glavu liste nakon svih modifikacija */
  return glava;

}

/* Funkcija prikazuje listu */
void prikazi_listu(Cvor_liste * glava)
{
   if(glava == NULL)
    return;

   printf("%s = %d\n", glava->naziv, glava->cena);
   prikazi_listu(glava->sledeci);
}


/* Funkcija oslobadja listu */
Cvor_liste * oslobodi_listu(Cvor_liste * glava)
{
  if(glava == NULL)
   return NULL;

  glava->sledeci = oslobodi_listu(glava->sledeci);
  free(glava);

  return NULL;
}

/* Funkcija oslobadja stablo. */
Cvor * oslobodi_stablo (Cvor *koren)
{

if(koren == NULL)
 return NULL;

koren->levi = oslobodi_stablo (koren->levi);
koren->desni = oslobodi_stablo (koren->desni);
free(koren);

return NULL;

}

/* test program */
int main(int argc, char ** argv)
{
Cvor * koren = NULL, * pomocni;
Cvor_liste * glava = NULL;
FILE *f;
char naziv[MAX];
int cena;

/* Proveravamo da li je navedeno ime datoteke */
if(argc < 2)
{
  fprintf(stderr, "Morate navesti ime datoteke!\n");
  exit(0);
}

/* Otvaramo datoteku */
if((f = fopen(argv[1], "r")) == NULL)
{
  fprintf(stderr, "fopen() greska\n");
  exit(1);
}

/* Ubacujemo proizvode u stablo */
while(fscanf(f, "%s %d", naziv, &cena) == 2)
{
   koren = dodaj_u_stablo(koren, naziv, cena);
}
fclose(f);

/* Testiranje pretrage po nazivu (efikasna operacija) */
printf("Uneti naziv proizvoda koji vas zanima: ");
scanf("%s", naziv);

if((pomocni = pretrazi_stablo(koren, naziv)) == NULL)
  printf("Trazeni proizvod ne postoji\n");
else
  printf("Cena trazenog proizvoda je: %d\n", pomocni->cena);

/* Testiranje pretrage po ceni (neefikasno) */
printf("Unesite maksimalnu cenu: ");
scanf("%d", &cena);
glava = pronadji_manje(koren, cena, NULL);
prikazi_listu(glava);

/* Oslobadjanje memorije */
glava = oslobodi_listu(glava);
koren = oslobodi_stablo(koren);

return 0;

}
