#include <stdio.h>
#include <stdlib.h>

/* NAPOMENA: Binarno stablo je skup cvorova koji su 
   povezani na sledeci nacin:
   1) Svaki cvor moze imati levog i desnog SINA (pri
      tom svaki od sinova moze biti i izostavljen, ili oba).
      Kazemo da je dati cvor RODITELJ svojim sinovima.
      Ako je cvor U roditelj cvoru V, tada pisemo da
      je U < V. Cvor koji nema ni levog ni desnog sina 
      naziva se LIST. 
   2) Postoji jedan jedinstveni cvor takav da nema roditelja.
      Ovaj cvor nazivamo KOREN stabla.
   3) U stablu nema ciklusa, tj. ne postoji niz cvorova 
      x1,x2,...,xn, takav da je x1 < x2 < ... < xn < x1.
   Inace, niz cvorova x1 < x2 < ... < xn nazivamo put u
   stablu. Kazemo da je cvor U predak cvora V ako u stablu
   postoji put od U do V. Specijalno, svaki cvor je predak
   samom sebi. Lako se moze pokazati da je koren predak svih
   cvorova u stablu. Rastojanje od korena do nekog cvora naziva 
   se visina cvora (koren je visine 0). Maksimalna visina cvora 
   u stablu naziva se visina stabla. 

   Stablo koje nema cvorova naziva se prazno stablo.

   Za svaki cvor V u stablu mozemo posmatrati stablo koje se
   sastoji od svih njegovih potomaka (svih cvorova kojima je on 
   predak). Ovo stablo se naziva podstablo sa datim korenom V.
   Podstablo sa njegovim levim sinom kao korenom nazivamo levim
   podstablom cvora V, dok podstablo sa njegovim desnim sinom
   kao korenom nazivamo desnim podstablom cvora V. Ako je neki
   od njegovih sinova izostavljen, tada kazemo da je odgovarajuce
   podstablo cvora V prazno stablo. Specijalno, ako je V koren, 
   tada njegovo levo i desno podstablo nazivamo levim i desnim 
   podstablom datog (citavog) stabla.
   S obzirom na gore receno, stablo se moze definisati i rekurzivno
   na sledeci nacin:

   1) Prazno stablo je stablo
   2) Ako su data dva stabla t1 i t2, i cvor r, tada je i (t1,r,t2)
      takodje stablo. t1 je tada levo podstablo, t2 je desno podstablo
      dok je cvor r koren tako formiranog stabla.

   U programiranju se stabla obicno koriste kao strukture podataka,
   tako sto svaki cvor sadrzi po jedan podatak odredjenog tipa. Na ovaj
   nacin se podaci mogu organizovati dvodimenzionalno, za razliku od
   jednodimenzionalne organizacije kod nizova i lista. Nacin rasporedjivanja
   podataka u stablu zavisi od konkretne primene stabla. S obzirom na 
   rekurzivnu prirodu stabla, uobicajeno je da se u programiranju
   stabla obradjuju rekurzivno.

   Posebna vrsta stabla su tzv. binarna pretrazivacka stabla. Ova 
   stabla imaju osobinu da za svaki cvor vazi sledece: svi cvorovi 
   njegovog levog podstabla sadrze podatke sa manjom vrednoscu od
   vrednosti podatka u tom cvoru, dok svi cvorovi njegovog desnog 
   podstabla sadrze podatke sa vecom vrednoscu od vrednosti podatka 
   u tom cvoru. Ovakva organizacija omogucava efikasno pretrazivanje.
   
   U nastavku demonstriramo upotrebu binarnog pretrazivackog stabla.
 
*/   

/* Struktura koja predstavlja cvor stabla */
typedef struct cvor {

int vrednost;        /* Vrednost koja se cuva */
struct cvor * levi;  /* Pokazivac na levo podstablo */
struct cvor * desni; /* Pokazivac na desno podstablo */

} Cvor;


/* NAPOMENA: Prazno stablo se predstavlja NULL pokazivacem. */

/* Pomocna funkcija za kreiranje cvora. Cvor se kreira
   dinamicki, funkcijom malloc(). U slucaju greske program
   se prekida i ispisuje se poruka o gresci. U slucaju 
   uspeha inicijalizuje se vrednost datim brojem, a pokazivaci
   na podstabla se inicijalizuju na NULL. Funkcija vraca 
   adresu novokreiranog cvora */
Cvor * napravi_cvor (int broj)
{

/* dinamicki kreiramo cvor */
Cvor * novi = (Cvor *) malloc (sizeof(Cvor));

/* u slucaju greske ... */
if (novi == NULL)
{
  fprintf (stderr,"malloc() greska\n");
  exit (1);
}

/* inicijalizacija */
novi->vrednost = broj;
novi->levi = NULL;
novi->desni = NULL;

/* vracamo adresu novog cvora */
return novi;

}

/* Funkcija dodaje novi cvor u stablo sa datim korenom.
   Ukoliko broj vec postoji u stablu, ne radi nista.
   Cvor se kreira funkcijom napravi_cvor(). Funkcija 
   vraca koren stabla nakon ubacivanja novog cvora. */
Cvor * dodaj_u_stablo (Cvor * koren, int broj)
{

/* izlaz iz rekurzije: ako je stablo bilo prazno,
   novi koren je upravo novi cvor */
if (koren == NULL)
  return napravi_cvor (broj);

/* Ako je stablo neprazno, i koren sadrzi manju vrednost
   od datog broja, broj se umece u desno podstablo, 
   rekurzivnim pozivom */
if (koren->vrednost < broj)
  koren->desni = dodaj_u_stablo (koren->desni, broj);
/* Ako je stablo neprazno, i koren sadrzi vecu vrednost
   od datog broja, broj se umece u levo podstablo,
   rekurzivnim pozivom */
else if (koren->vrednost > broj)
  koren->levi = dodaj_u_stablo (koren->levi, broj);

/* U slucaju da je koren jednak datom broju, tada
   broj vec postoji u stablu, i ne radimo nista */

/* Vracamo koren stabla */
return koren;

}

/* Funkcija pretrazuje binarno stablo. Ukoliko 
   pronadje cvor sa vrednoscu koja je jednaka
   datom broju, vraca adresu tog cvora. U 
   suprotnom vraca NULL */
Cvor * pretrazi_stablo (Cvor * koren, int broj)
{

/* Izlaz iz rekurzije: ako je stablo prazno,
   tada trazeni broj nije u stablu */
if (koren == NULL)
 return NULL;

/* Ako je stablo neprazno, tada se pretrazivanje
   nastavlja u levom ili desnom podstablu, u 
   zavisnosti od toga da li je trazeni broj 
   respektivno manji ili veci od vrednosti
   korena. Ukoliko je pak trazeni broj jednak
   korenu, tada se vraca adresa korena. */
if (koren->vrednost < broj)
  return pretrazi_stablo (koren->desni, broj);
else if (koren->vrednost > broj)
  return pretrazi_stablo (koren->levi, broj);
else 
  return koren;

}


/* Funkcija vraca adresu cvora sa najmanjom vrednoscu
   u stablu, ili NULL ako je stablo prazno */
Cvor * pronadji_najmanji (Cvor * koren)
{

/* Slucaj praznog stabla */
if (koren == NULL)
 return NULL;

/* Izlaz iz rekurzije: ako koren nema levog sina,
   tada je upravo koren po vrednosti najmanji cvor.*/
if (koren->levi == NULL)
 return koren;
/* U suprotnom je najmanja vrednost u stablu upravo
   najmanja vrednost u levom podstablu, pa se pretraga
   nastavlja rekurzivno u levom podstablu. */
else 
 return pronadji_najmanji (koren->levi);

}

/* Funkcija vraca adresu cvora sa najvecom vrednoscu
   u stablu, ili NULL ako je stablo prazno */
Cvor * pronadji_najveci (Cvor * koren)
{

/* Slucaj praznog stabla */
if (koren == NULL)
 return NULL;

/* Izlaz iz rekurzije: ako koren nema desnog sina,
   tada je upravo koren po vrednosti najveci cvor. */
if (koren->desni == NULL)
 return koren;
/* U suprotnom je najveca vrednost u stablu upravo
   najveca vrednost u desnom podstablu, pa se pretraga
   nastavlja rekurzivno u desnom podstablu. */
else 
 return pronadji_najveci (koren->desni);

}

/* Funkcija brise cvor sa datom vrednoscu iz stabla,
   ukoliko postoji. U suprotnom ne radi nista. Funkcija
   vraca koren stabla (koji moze biti promenjen nakon 
   brisanja) */
Cvor * obrisi_element (Cvor * koren, int broj)
{

    Cvor * pomocni=NULL;

    /* Izlaz iz rekurzije: ako je stablo prazno, ono ostaje
       prazno i nakon brisanja */
    if (koren == NULL)
       return NULL;

    /* Ako je vrednost broja veca od vrednosti korena,
       tada se broj eventualno nalazi u desnom podstablu,
       pa treba rekurzivno primeniti postupak na desno 
       podstablo. Koren ovako modifikovanog stabla je
       nepromenjen, pa vracamo stari koren */
    if (koren->vrednost < broj)
    {
       koren->desni = obrisi_element (koren->desni, broj);
       return koren;
    }
    
    /* Ako je vrednost broja manja od vrednosti korena,
       tada se broj eventualno nalazi u levom podstablu,
       pa treba rekurzivno primeniti postupak na levo 
       podstablo. Koren ovako modifikovanog stabla je
       nepromenjen, pa vracamo stari koren */
    if (koren->vrednost > broj)
    {   
       koren->levi = obrisi_element (koren->levi, broj);
       return koren;
    }

    /* Slede podslucajevi vezani za slucaj kada je vrednost
       korena jednaka broju koji se brise (tj. slucaj kada 
       treba obrisati koren) */

    /* Ako koren nema sinova, tada se on prosto brise, i
       rezultat je prazno stablo (vracamo NULL) */
    if (koren->levi == NULL && koren->desni == NULL)
     {
        free (koren);
        return NULL;
     }

     /* Ako koren ima samo levog sina, tada se brisanje
        vrsi tako sto obrisemo koren, a novi koren postaje
        levi sin */
     if (koren->levi != NULL && koren->desni == NULL)
     {
         pomocni = koren->levi;
         free (koren);
         return pomocni; 
     }

     /* Ako koren ima samo desnog sina, tada se brisanje 
        vrsi tako sto obrisemo koren, a novi koren postaje
        desni sin */
     if (koren->desni != NULL && koren->levi == NULL)
     {
         pomocni = koren->desni;
         free (koren);
         return pomocni; 
     }


     /* Slucaj kada koren ima oba sina. U tom slucaju se
        brisanje vrsi na sledeci nacin: najpre se potrazi
        sledbenik korena (u smislu poretka) u stablu. To
       je upravo po vrednosti najmanji cvor u desnom podstablu.
       On se moze pronaci npr. funkcijom pronadji_najmanji().
       Nakon toga se u koren smesti vrednost tog cvora, a u taj
       cvor se smesti vrednost korena (tj. broj koji se brise).
       Onda se prosto rekurzivno pozove funkcija za brisanje
       na desno podstablo. S obzirom da u njemu treba obrisati
       najmanji element, a on definitivno ima najvise jednog 
       potomka, jasno je da ce to brisanje biti obavljeno na 
       jedan od nacina koji je gore opisan. */ 
     pomocni = pronadji_najmanji (koren->desni);
     koren->vrednost =  pomocni->vrednost;
     pomocni->vrednost = broj;
     koren->desni = obrisi_element (koren->desni, broj);
     return koren;

}

/* Funkcija prikazuje stablo s leva u desno (tj. 
   prikazuje elemente u rastucem poretku) */
void prikazi_stablo (Cvor * koren)
{
/* izlaz iz rekurzije */
if(koren == NULL)
  return;

prikazi_stablo (koren->levi);
printf("%d ", koren->vrednost);
prikazi_stablo (koren->desni);
}

/* Funkcija oslobadja prostor koji je alociran za 
   cvorove stabla. Funkcija vraca NULL, zato sto je
   nakon oslobadjanja stablo prazno. */
Cvor * oslobodi_stablo (Cvor *koren)
{

/* Izlaz iz rekurzije */
if(koren == NULL)
 return NULL;

koren->levi = oslobodi_stablo (koren->levi);
koren->desni = oslobodi_stablo (koren->desni);
free(koren);

return NULL;

}

/* Funkcija kreira novo stablo identicno stablu koje je
   dato korenom. Funkcija vraca pokazivac na koren
   novog stabla. */
Cvor * kopiraj_stablo (Cvor * koren)
{
Cvor * duplikat = NULL;

/* Izlaz iz rekurzije: ako je stablo prazno,
   vracamo NULL */
if(koren == NULL)
   return NULL;

/* Dupliramo koren stabla i postavljamo ga
   da bude koren novog stabla */
duplikat = napravi_cvor (koren->vrednost);

/* Rekurzivno dupliramo levo podstablo i njegovu
   adresu cuvamo u pokazivacu na levo podstablo 
   korena duplikata. */
duplikat->levi = kopiraj_stablo (koren->levi);

/* Rekurzivno dupliramo desno podstablo i njegovu
   adresu cuvamo u pokazivacu na desno podstablo 
   korena duplikata. */
duplikat->desni = kopiraj_stablo (koren->desni);

/* Vracamo adresu korena duplikata */
return duplikat;

}

/* Funkcija modifikuje stablo dato korenom koren_1
   tako da sadrzi i sve elemente drugog stabla datog
   korenom koren_2 (drugim recima funkcija kreira uniju
   dva stabla, i rezultat se smesta u prvo stablo).
   Funkcija vraca pokazivac na koren tako modifikovanog
   prvog stabla. */
Cvor * kreiraj_uniju (Cvor * koren_1, Cvor * koren_2)
{

/* Ako je drugo stablo neprazno */
if(koren_2 != NULL) 
{
/* dodajemo koren drugog stabla u prvo stablo */
koren_1 = dodaj_u_stablo (koren_1, koren_2->vrednost);

/* rekurzivno racunamo uniju levog i desnog podstabla
   drugog stabla sa prvim stablom */
koren_1 = kreiraj_uniju (koren_1, koren_2->levi);
koren_1 = kreiraj_uniju (koren_1, koren_2->desni);
}

/* vracamo pokazivac na modifikovano prvo stablo */
return koren_1;
}

/* Funkcija modifikuje stablo dato korenom koren_1
   tako da sadrzi samo one elemente koji su i elementi
   stabla datog korenom koren_2 (drugim recima funkcija 
   kreira presek dva stabla, i rezultat se smesta u prvo 
   stablo). Funkcija vraca pokazivac na koren tako 
   modifikovanog prvog stabla. */
Cvor * kreiraj_presek (Cvor * koren_1, Cvor * koren_2)
{

/* Ako je prvo stablo prazno, tada je i rezultat prazno
   stablo */
if(koren_1 == NULL)
   return NULL;

/* Kreiramo presek levog i desnog podstabla sa drugim 
   stablom, tj. iz levog i desnog podstabla prvog stabla
   brisemo sve one elemente koji ne postoje u drugom 
   stablu */
koren_1->levi = kreiraj_presek (koren_1->levi, koren_2);
koren_1->desni = kreiraj_presek (koren_1->desni, koren_2);
 
/* Ako se koren prvog stabla ne nalazi u drugom stablu
   tada ga uklanjamo iz prvog stabla */
if(pretrazi_stablo (koren_2, koren_1->vrednost) == NULL)
  koren_1 = obrisi_element (koren_1, koren_1->vrednost);

/* Vracamo koren tako modifikovanog prvog stabla */
return koren_1;

}

/* Funkcija modifikuje stablo dato korenom koren_1
   tako da sadrzi samo one elemente koji nisu i elementi
   stabla datog korenom koren_2 (drugim recima funkcija 
   kreira razliku dva stabla, i rezultat se smesta u prvo 
   stablo). Funkcija vraca pokazivac na koren tako 
   modifikovanog prvog stabla. */
Cvor * kreiraj_razliku (Cvor * koren_1, Cvor * koren_2)
{

/* Ako je prvo stablo prazno, tada je i rezultat prazno
   stablo */
if(koren_1 == NULL)
   return NULL;

/* Kreiramo razliku levog i desnog podstabla sa drugim 
   stablom, tj. iz levog i desnog podstabla prvog stabla
   brisemo sve one elemente koji postoje i u drugom 
   stablu */
koren_1->levi = kreiraj_razliku (koren_1->levi, koren_2);
koren_1->desni = kreiraj_razliku (koren_1->desni, koren_2);
 
/* Ako se koren prvog stabla nalazi i u drugom stablu
   tada ga uklanjamo iz prvog stabla */
if(pretrazi_stablo (koren_2, koren_1->vrednost) != NULL)
  koren_1 = obrisi_element (koren_1, koren_1->vrednost);

/* Vracamo koren tako modifikovanog prvog stabla */
return koren_1;

}




/* test program */
int main()
{

Cvor * koren = NULL, * koren_2 = NULL;
Cvor * pomocni = NULL;
int broj;


/* Testiranje dodavanja u stablo */
printf("-------------------------------------------------------\n");
printf("---------- Testiranje dodavanja u stablo --------------\n");
do {
printf("-------------------------------------------------------\n");
printf("Prikaz trenutnog sadrzaja stabla:\n");
prikazi_stablo (koren);
putchar('\n');
printf("-------------------------------------------------------\n");
printf("Dodati element u stablo (ctrl-D za kraj unosa):\n");
} while(scanf("%d", &broj) > 0 &&  
                    (koren = dodaj_u_stablo (koren, broj)) );

printf("-------------------------------------------------------\n");
putchar('\n');

/* Testiranje pretrage elementa */
printf("-------------------------------------------------------\n");
printf("---------------- Testiranje pretrage ------------------\n");
printf("-------------------------------------------------------\n");
printf("Prikaz trenutnog sadrzaja stabla:\n");
prikazi_stablo (koren);
putchar('\n');
printf("-------------------------------------------------------\n");
printf("Uneti broj koji se trazi: ");
scanf("%d", &broj);

if((pomocni = pretrazi_stablo (koren, broj)) == NULL)
    printf("Trazeni broj nije u stablu\n");
else
   printf("Trazeni element %d je u stablu\n", pomocni->vrednost); 
printf("-------------------------------------------------------\n");
putchar('\n');


/* Testiranje brisanja elemenata */
printf("-------------------------------------------------------\n");
printf("--------------- Testiranje brisanja -------------------\n");
printf("-------------------------------------------------------\n");
printf("Prikaz trenutnog sadrzaja stabla:\n");
prikazi_stablo (koren);
putchar('\n');
printf("-------------------------------------------------------\n");
printf("Uneti broj koji se brise: ");
scanf("%d", &broj);

koren = obrisi_element (koren, broj);
printf("-------------------------------------------------------\n");
printf("Stablo nakon izbacivanja:\n");
prikazi_stablo (koren);
putchar('\n');
printf("-------------------------------------------------------\n");
putchar('\n');


/* Testiranje dupliranja, unije, preseka i razlike */
printf("-------------------------------------------------------\n");
printf("----- Testiranje dupliranja i skupovnih operacija -----\n");
do {
printf("-------------------------------------------------------\n");
printf("Prikaz trenutnog sadrzaja 2. stabla:\n");
prikazi_stablo (koren_2);
putchar('\n');
printf("-------------------------------------------------------\n");
printf("Dodati element u 2. stablo (ctrl-D za kraj unosa):\n");
} while(scanf("%d", &broj) > 0 &&  
                    (koren_2 = dodaj_u_stablo (koren_2, broj)) );

printf("-------------------------------------------------------\n");
putchar('\n');


pomocni = kopiraj_stablo(koren);
pomocni = kreiraj_uniju(pomocni, koren_2);

printf("-------------------------------------------------------\n");
printf("Prikaz 1. stabla:\n");
prikazi_stablo (koren);
putchar('\n');
printf("-------------------------------------------------------\n");
printf("Prikaz 2. stabla:\n");
prikazi_stablo (koren_2);
putchar('\n');
printf("-------------------------------------------------------\n");
printf("Unija ova dva stabla:\n");
prikazi_stablo (pomocni);
putchar('\n');
printf("-------------------------------------------------------\n");

pomocni = oslobodi_stablo(pomocni);

pomocni = kopiraj_stablo(koren);
pomocni = kreiraj_presek(pomocni, koren_2);

printf("Presek ova dva stabla:\n");
prikazi_stablo (pomocni);
putchar('\n');
printf("-------------------------------------------------------\n");

pomocni = oslobodi_stablo(pomocni);

pomocni = kopiraj_stablo(koren);
pomocni = kreiraj_razliku(pomocni, koren_2);

printf("Razlika ova dva stabla:\n");
prikazi_stablo (pomocni);
putchar('\n');
printf("-------------------------------------------------------\n");

pomocni = oslobodi_stablo (pomocni);

koren_2 = oslobodi_stablo (koren_2);

koren = oslobodi_stablo (koren);

return 0;

}
