/* Program proverava da li su etikete u datom HTML fajlu dobro uparene */



#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>

#define MAX 100

#define OTVORENA 1
#define ZATVORENA 2

#define VAN_ETIKETE 0
#define PROCITANO_MANJE 1
#define U_ETIKETI 2


/* NAPOMENA: Stek (eng. stack) je struktura podataka nad kojom su 
   definisane sledece operacije:
   1) Dodavanje elementa -- kazemo da je element potisnut na vrh 
     steka (eng. push() operacija)
   2) Uklanjanje elementa koji je poslednji dodat -- kazemo da je 
      element skinut sa vrha steka (eng. pop() operacija)
   3) Ocitavanje vrednosti elementa koji je poslednji dodat (eng.
      top() operacija)

   Stek spada u LIFO strukture (eng. Last In First Out). Moze se 
   implementirati na vise nacina. Najjednostavniji nacin je da se
   definise kao niz. Medjutim, tada je ogranicen max. broj elemenata
   na steku dimenzijom niza. Zbog toga se obicno pribegava koriscenju
   lista za implementaciju steka, gde se push() operacija svodi
   na dodavanje na pocetak, a pop() operacija se svodi na uklanjanje
   glave liste. Obe operacije se izvode u konstantnom vremenu.

   U ovom primeru demonstriramo koriscenje steka za proveru ispravnosti
   uparivanja etiketa u html dokumentu.

*/


/* Struktura koja predstavlja cvor liste */
typedef struct cvor {
char etiketa[MAX];     /* Sadrzi ime etikete */
struct cvor *sledeci;  /* pokazivac na sledeci cvor */
} Cvor;


/* Funkcija kreira novi cvor, upisuje u njega etiketu
   i vraca njegovu adresu */
Cvor * napravi_cvor(char * etiketa)
{
     Cvor *novi = NULL;
     if((novi = (Cvor *) malloc(sizeof(Cvor))) == NULL)
       {
           fprintf(stderr,"malloc() greska!\n");
           exit(1);
       }

     strcpy(novi->etiketa, etiketa);
     novi->sledeci = NULL;
     return novi;
}


/* Funkcija postavlja na vrh steka novu etiketu */
void potisni_na_stek(Cvor **vrh, char *etiketa)
{
    Cvor * novi = napravi_cvor(etiketa);
    novi->sledeci = *vrh;
    *vrh = novi; 
}

/* Funkcija skida sa vrha steka etiketu. Ako je drugi argument
   pokazivac razlicit od NULL, tada u niz karaktera na koji on
   pokazuje upisuje ime etikete koja je upravo skinuta sa steka
   dok u suprotnom ne radi nista. Funkcija vraca 0 ako je stek
   prazan (pa samim tim nije bilo moguce skinuti vrednost sa 
   steka) ili 1 u suprotnom. */
int skini_sa_steka(Cvor **vrh, char * etiketa)
{
     Cvor * pomocni;

     if(*vrh == NULL)
        return 0;

     if(etiketa != NULL)
       strcpy(etiketa, (*vrh)->etiketa);


     pomocni = *vrh;
     *vrh = (*vrh)->sledeci;
     free(pomocni);

     return 1;
}

/* Funkcija vraca pokazivac na string koji sadrzi etiketu
   na vrhu steka. Ukoliko je stek prazan, vraca NULL */
char * vrh_steka(Cvor *vrh)
{
     if(vrh == NULL)
       return NULL;
     else
      return vrh->etiketa;
}

/* Funkcija prikazuje stek pocev od vrha prema dnu */
void prikazi_stek(Cvor *vrh)
{
   for(;vrh != NULL; vrh = vrh->sledeci)
        printf("<%s>\n", vrh->etiketa);
}


/* Funkcija prazni stek */
void oslobodi_stek(Cvor **vrh)
{
  Cvor *pomocni;

  while(*vrh != NULL)
  {
    pomocni = *vrh;
    *vrh = (*vrh)->sledeci;
    free(pomocni);
  }
}

/* Funkcija iz fajla na koji pokazuje f cita sledecu
   etiketu, i njeno ime upisuje u niz na koji pokazuje
   pokazivac etiketa. Funkcija vraca EOF u slucaju da
   se dodje do kraja fajla pre nego sto se procita  
   etiketa, vraca OTVORENA ako je procitana otvorena
   etiketa, odnosno ZATVORENA ako je procitana zatvorena
   etiketa. */
int uzmi_etiketu(FILE *f, char * etiketa)
{
int c;
int stanje = VAN_ETIKETE;
int i = 0;
int tip;

while((c = fgetc(f)) != EOF)
{
   switch(stanje)
   {
      case VAN_ETIKETE:
           if(c == '<')
             {
                stanje = PROCITANO_MANJE;
             }
             break;
      case PROCITANO_MANJE:
          if(c == '/')
           {
             tip = ZATVORENA;
           }
          else if(isalpha(c))
           {
             tip = OTVORENA;
             etiketa[i++] = tolower(c);
           }
           stanje = U_ETIKETI;
           break;
      case U_ETIKETI:
           if(isalpha(c) && i < MAX - 1)
             etiketa[i++] = tolower(c);
           else {
             stanje = VAN_ETIKETE;
             etiketa[i]='\0';
             return tip;
           }
           break;
    }

}
 
 return EOF;

}

/* Test program */
int main(int argc, char **argv)
{

Cvor * vrh = NULL;
char etiketa[MAX];
int tip;
int u_redu = 1;
FILE * f;

/* Ime datoteke zadajemo na komandnoj liniji */
if(argc < 2)
{
  fprintf(stderr, "Koriscenje: %s ime_html_datoteke\n", argv[0]);
  exit(0);
}

/* Otvaramo datoteku */
if((f = fopen(argv[1], "r")) == NULL)
{
  fprintf(stderr, "fopen() greska!\n");
  exit(1);
}

/* Dokle god ima etiketa, uzimamo ih jednu po jednu sa ulaza */
while((tip = uzmi_etiketu(f, etiketa)) != EOF)
{
   /* Ako je otvorena etiketa, dodajemo je na stek. Izuzetak su
      etikete <br>, <hr> i <meta> koje nemaju sadrzaj, tako da 
      ih nije potrebno zatvoriti. NAPOMENA: U html-u postoje
      jos neke etikete koje koje nemaju sadzaj (npr link). 
      Pretpostavimo da njih nema u dokumentu, zbog jednostavnosti */
   if(tip == OTVORENA)
    {
     if(strcmp(etiketa, "br") != 0 && strcmp(etiketa, "hr") != 0 && strcmp(etiketa, "meta") != 0)
       potisni_na_stek(&vrh, etiketa);
    }
   /* Ako je zatvorena etiketa, tada je uslov dobre uparenosti da je u pitanju
      zatvaranje etikete koja je poslednja otvorena, a jos uvek nije zatvorena.
      Ova etiketa se mora nalaziti na vrhu steka. Ako je taj uslov ispunjen,
      tada je skidamo sa steka, jer je zatvorena. U suprotnom, obavestavamo 
      korisnika da etikete nisu pravilno uparene. */
   else if(tip == ZATVORENA)
    {
        if(vrh_steka(vrh) != NULL && strcmp(vrh_steka(vrh), etiketa) == 0)
         skini_sa_steka(&vrh, NULL);
        else
         {
         printf(vrh_steka(vrh) != NULL ? 
          "Etikete nisu pravilno uparene\n(nadjena etiketa </%s> a poslednja otvorena etiketa je <%s>)\n" :
          "Etikete nisu pravilno uparene\n(nadjena etiketa </%s> koja nije otvorena)\n",
                etiketa, vrh_steka(vrh));
         u_redu = 0;
         break;
         }
    }
}

/* Zatvaramo fajl */
fclose(f);

/* Ako nismo pronasli pogresno uparivanje...*/
if(u_redu)
{
 /* Nakon zavrsetka citanja etiketa uslov je da stek mora biti prazan. 
    Ako nije, tada znaci da postoje jos neke etikete koje su otvorene
    ali nisu zatvorene. */
 if(vrh_steka(vrh) == NULL)
   printf("Etikete su pravilno uparene!\n");
 else
   printf("Etikete nisu pravilno uparene\n(etiketa <%s> nije zatvorena)\n", vrh_steka(vrh));
} 

/* Oslobadjamo stek */
oslobodi_stek(&vrh);

return 0;

}
