/* Vrednosti kontrolnih e-signala */
`define DISABLE 0
`define ENABLE 1

/* Stanja konacnog automata kontrolne jedinice */
`define FETCH 0
`define REQUEST_INST 1
`define WAIT_INST 2
`define DECODE 3
`define EXECUTE 4
`define WRITE 5
`define REQUEST_LOAD 6
`define WAIT_LOAD 7
`define REQUEST_STORE 8
`define WAIT_STORE 9
`define HALT_STATE 10

/* Mnemonici instrukcija */
`define LDR0 8'b00000000
`define LDR1 8'b00000001
`define LDR2 8'b00000010
`define LDR3 8'b00000011
`define LDR4 8'b00000100
`define LDR5 8'b00000101
`define LDR6 8'b00000110
`define LDR7 8'b00000111
`define LDR8 8'b00001000
`define LDR9 8'b00001001
`define LDR10 8'b00001010
`define LDR11 8'b00001011
`define LDR12 8'b00001100
`define LDR13 8'b00001101
`define LDR14 8'b00001110
`define LDR15 8'b00001111

`define LDC0 8'b00010000
`define LDC1 8'b00010001
`define LDC2 8'b00010010
`define LDC3 8'b00010011
`define LDC4 8'b00010100
`define LDC5 8'b00010101
`define LDC6 8'b00010110
`define LDC7 8'b00010111
`define LDC8 8'b00011000
`define LDC9 8'b00011001
`define LDC10 8'b00011010
`define LDC11 8'b00011011
`define LDC12 8'b00011100
`define LDC13 8'b00011101
`define LDC14 8'b00011110
`define LDC15 8'b00011111

`define STR0 8'b00100000
`define STR1 8'b00100001
`define STR2 8'b00100010
`define STR3 8'b00100011
`define STR4 8'b00100100
`define STR5 8'b00100101
`define STR6 8'b00100110
`define STR7 8'b00100111
`define STR8 8'b00101000
`define STR9 8'b00101001
`define STR10 8'b00101010
`define STR11 8'b00101011
`define STR12 8'b00101100
`define STR13 8'b00101101
`define STR14 8'b00101110
`define STR15 8'b00101111

`define LDI 8'b10110000
`define STI 8'b10110001

`define MOV 8'b10000000
`define ADD 8'b10000001
`define ADC 8'b10000010
`define SUB 8'b10000011
`define SHL 8'b10000100
`define SHR 8'b10000101
`define SAR 8'b10000110
`define AND 8'b10000111
`define OR  8'b10001000
`define XOR 8'b10001001
`define CMP 8'b10001010
`define TST 8'b10001011

`define JMP 8'b01000000
`define JE  8'b01000001
`define JNE 8'b01000010
`define JA  8'b01000011
`define JB  8'b01000100
`define JAE 8'b01000101
`define JBE 8'b01000110
`define JG  8'b01000111
`define JL  8'b01001000
`define JGE 8'b01001001
`define JLE 8'b01001010

`define JMPI 8'b11000000
`define JEI  8'b11000001
`define JNEI 8'b11000010
`define JAI  8'b11000011
`define JBI  8'b11000100
`define JAEI 8'b11000101
`define JBEI 8'b11000110
`define JGI  8'b11000111
`define JLI  8'b11001000
`define JGEI 8'b11001001
`define JLEI 8'b11001010

`define HALT 8'b11111111

/* Signali za ALU jedinicu */
`define ALU_NOP    4'd0
`define ALU_ADD    4'd1
`define ALU_ADC    4'd2
`define ALU_SUB    4'd3
`define ALU_SHL    4'd4
`define ALU_SHR    4'd5
`define ALU_SAR    4'd6
`define ALU_AND    4'd7
`define ALU_OR     4'd8
`define ALU_XOR    4'd9

/* Flegovi */
`define CF 2'd0
`define ZF 2'd1
`define SF 2'd2
`define OF 2'd3

`include "psw_checker.v"

/* Kontrolna jedinica procesora. Radi u ritmu casovnika clk. Na ulazu
   ima komponente IR registra, kao i izlaz PSW registra (koji se koristi
   kod instrukcija skoka). Signali koji pocinju slovom 'e' su razni 
   signali kojima se kontrolisu ostale komponente procesora. mem_* signali
   se salju direktno memoriji (mem_status dolazi iz memorije). Signal
   halt_computer je izlazni signal kojim se gasi racunar */
module cu(clk, psw_out, ir_op_code, ir_dst_reg, ir_src_reg,
	  e_reg_a_in, e_reg_b_in, e_reg_a_out, e_reg_b_out, 
	  e_alu, 
	  e_ac_alu_in, e_ac_a_out, e_ac_b_out,
	  e_psw_in,
	  e_pc_a_in, e_pc_b_in, e_pc_a_out, e_pc_b_out, e_pc_inc,
	  e_ir_a_in, e_ir_b_in,
	  e_mar_a_in, e_mar_b_in, e_addr_out, e_calc_addr,
	  e_mdr_a_in, e_mdr_b_in, e_data_in, e_mdr_a_out, e_mdr_b_out, e_data_out, e_calc_const,  
	  alu_op, reg_a_sel, reg_b_sel, mem_rd, mem_wr, mem_sent, mem_status, halt_computer);
   
   input clk;
   input [3:0] psw_out;
   input [7:0] ir_op_code;
   input [3:0] 	ir_dst_reg;
   input [3:0] 	ir_src_reg;
   output reg  e_reg_a_in, e_reg_b_in, e_reg_a_out, e_reg_b_out, 
	       e_alu, 
	       e_ac_alu_in, e_ac_a_out, e_ac_b_out,
	       e_psw_in, 
	       e_pc_a_in, e_pc_b_in, e_pc_a_out, e_pc_b_out, e_pc_inc,
	       e_ir_a_in, e_ir_b_in,
	       e_mar_a_in, e_mar_b_in, e_addr_out, e_calc_addr,
	       e_mdr_a_in, e_mdr_b_in, e_data_in, e_mdr_a_out, e_mdr_b_out, e_data_out, e_calc_const;   	       

   output reg[3:0] alu_op;
   output reg[3:0] reg_a_sel;
   output reg[3:0] reg_b_sel;
   output    reg   mem_rd, mem_wr, mem_sent;
   input       mem_status;
   output reg  halt_computer;
   
   
   reg [3:0]   state;
   wire        cond;
   
   psw_checker _psw_checker(ir_op_code[3:0], psw_out, cond);
   
   /* Inicijalno su svi signali iskljuceni, a pocetno stanje je FETCH */
   initial
     begin
	state <= `FETCH;
	e_reg_a_in <= `DISABLE;
	e_reg_b_in <= `DISABLE;
	e_reg_a_out <= `DISABLE;
	e_reg_b_out <= `DISABLE;
	e_alu <= `DISABLE;
	e_psw_in <= `DISABLE;      
	e_ac_alu_in <= `DISABLE;
	e_ac_a_out <= `DISABLE;
	e_ac_b_out <= `DISABLE;
	e_pc_a_in <= `DISABLE;
	e_pc_b_in <= `DISABLE;
	e_pc_a_out <= `DISABLE;
	e_pc_b_out <= `DISABLE;
	e_pc_inc <= `DISABLE;	
	e_ir_a_in <= `DISABLE;
	e_ir_b_in <= `DISABLE;	
	e_mar_a_in <= `DISABLE;
	e_mar_b_in <= `DISABLE;
	e_addr_out <= `DISABLE;	
	e_calc_addr <= `DISABLE;
	e_mdr_a_in <= `DISABLE;
	e_mdr_b_in <= `DISABLE;	
	e_mdr_a_out <= `DISABLE;
	e_mdr_b_out <= `DISABLE;       
	e_data_in <= `DISABLE;
	e_data_out <= `DISABLE;
	e_calc_const <= `DISABLE;
	alu_op <= `ALU_NOP;
	reg_a_sel <= 0;
	reg_b_sel <= 0;	
	mem_rd <= `DISABLE;
	mem_wr <= `DISABLE;
	mem_sent <= `DISABLE;
	halt_computer <= `DISABLE;		
     end

   /* Konacni automat kontrolne jedinice. Na svakoj pozitivnoj ivici
      signala casovnika se, u zavisnosti od stanja automata, kao i 
      ulaznih signala (IR i PSW registri) aktiviraju odgovarajuci 
      signali i prelazi se u drugo stanje. */
   always @(posedge clk)
     begin
	case(state)
	  `FETCH:
	    begin
	       e_mar_a_in <= `ENABLE;
	       e_pc_a_out <= `ENABLE;
	       mem_rd <= `ENABLE;
	       state <= `REQUEST_INST;
	    end
	  `REQUEST_INST:
	    if(mem_status)
	      begin
		 e_addr_out <= `ENABLE;
		 e_data_in <= `ENABLE;		 
		 mem_sent <= `ENABLE;		 
		 state <= `WAIT_INST;
	      end
	  `WAIT_INST:
	    if(!mem_status)
	      begin
		 e_data_in <= `DISABLE;		   
		 e_addr_out <= `DISABLE;
		 mem_sent <= `DISABLE;
		 mem_rd <= `DISABLE;
		 state <= `DECODE;
	      end
	  `DECODE:
	    begin	       		    
	       e_ir_a_in <= `ENABLE;
	       e_mdr_a_out <= `ENABLE;		    		    
	       state <= `EXECUTE;	       
	    end 
	  `EXECUTE:
	    begin
	       case(ir_op_code)
		 `LDR0, `LDR1, `LDR2, `LDR3, `LDR4, `LDR5, `LDR6, `LDR7, `LDR8,
		 `LDR9, `LDR10, `LDR11, `LDR12, `LDR13, `LDR14, `LDR15:
		   begin
		      e_pc_a_out <= `ENABLE;
		      e_mdr_b_out <= `ENABLE;
		      e_calc_addr <= `ENABLE;
		      mem_rd <= `ENABLE;		      
		      state <= `REQUEST_LOAD;		      
		   end
	 	 `LDC0, `LDC1, `LDC2, `LDC3, `LDC4, `LDC5, `LDC6, `LDC7, `LDC8,
		   `LDC9, `LDC10, `LDC11, `LDC12, `LDC13, `LDC14, `LDC15:
		     begin
			e_calc_const <= `ENABLE;
			state <= `WRITE;		      
		     end
		 `LDI:
		   begin
		      reg_b_sel <= ir_src_reg;
		      e_reg_b_out <= `ENABLE;
		      e_mar_b_in <= `ENABLE;
		      mem_rd <= `ENABLE;		      
		      state <= `REQUEST_LOAD;		      
		   end		   
	 	 `STR0, `STR1, `STR2, `STR3, `STR4, `STR5, `STR6, `STR7, `STR8,
		   `STR9, `STR10, `STR11, `STR12, `STR13, `STR14, `STR15:
		     begin						
			e_pc_a_out <= `ENABLE;
			e_mdr_b_out <= `ENABLE;
			e_calc_addr <= `ENABLE;
			state <= `WRITE;			
		     end
		 `STI:
		   begin
		      reg_b_sel <= ir_src_reg;
		      e_reg_b_out <= `ENABLE;
		      e_mar_b_in <= `ENABLE;
		      state <= `WRITE;		      		      
		   end		 
		 `MOV, `ADD, `ADC, `SUB, `SHL, `SHR, `SAR, `AND, `OR, `XOR:
		   begin
		      reg_a_sel <= ir_dst_reg;
		      reg_b_sel <= ir_src_reg;
		      e_reg_a_out <= `ENABLE;
		      e_reg_b_out <= `ENABLE;
		      e_alu <= `ENABLE;	
		      e_ac_alu_in <= `ENABLE;
		      alu_op <= ir_op_code[3:0];		      
		      e_psw_in <= `ENABLE;
		      state <= `WRITE;		      		      		      
		   end
		 `CMP, `TST:
		   begin
		      reg_a_sel <= ir_dst_reg;
		      reg_b_sel <= ir_src_reg;
		      e_reg_a_out <= `ENABLE;
		      e_reg_b_out <= `ENABLE;
		      e_alu <= `ENABLE;	
		      alu_op <= ir_op_code == `CMP ? `ALU_SUB : `ALU_AND;		      
		      e_psw_in <= `ENABLE;
		      e_pc_inc <= `ENABLE;		      
		      state <= `FETCH;		      		      		      		      
		   end 
		 `JMP, `JE, `JNE, `JA, `JB, `JAE, `JBE, `JG, `JL, `JGE, `JLE:
		   begin
		      if(cond)
			begin
			   e_pc_b_in <= `ENABLE;
			   e_mdr_b_out <= `ENABLE;
			   state <= `FETCH;		      
			end
		      else
			begin
			   e_pc_inc <= `ENABLE;
			   state <= `FETCH;			   
			end
		   end 
		 `JMPI, `JEI, `JNEI, `JAI, `JBI, `JAEI, `JBEI, `JGI, `JLI, `JGEI, `JLEI:
		   begin
		      if(cond)
			begin
			   e_pc_a_in <= `ENABLE;
			   e_reg_a_out <= `ENABLE;
			   reg_a_sel <= ir_src_reg;			   
			   state <= `FETCH;		      
			end
		      else
			begin
			   e_pc_inc <= `ENABLE;
			   state <= `FETCH;			   
			end
		   end
		 `HALT:
		   begin
		      state <= `HALT_STATE;		      
		   end
	       endcase
	    end 
	  `REQUEST_LOAD:
	    if(mem_status)
	      begin
		 e_addr_out <= `ENABLE;
		 e_data_in <= `ENABLE;		 
		 mem_sent <= `ENABLE;		 
		 state <= `WAIT_LOAD;
	      end
	  `WAIT_LOAD:
	    if(!mem_status)
	      begin
		 e_data_in <= `DISABLE;		   
		 e_addr_out <= `DISABLE;
		 mem_sent <= `DISABLE;		 
		 mem_rd <= `DISABLE;
		 state <= `WRITE;
	      end
	  `REQUEST_STORE:
	    if(mem_status)
	      begin
		 e_addr_out <= `ENABLE;
		 e_data_out <= `ENABLE;		 
		 mem_sent <= `ENABLE;		 
		 state <= `WAIT_STORE;
	      end
	  `WAIT_STORE:
	    if(!mem_status)
	      begin
		 e_data_out <= `DISABLE;		   
		 e_addr_out <= `DISABLE;
		 mem_sent <= `DISABLE;		 
		 mem_wr <= `DISABLE;
		 e_pc_inc <= `ENABLE;		 
		 state <= `FETCH;
	      end
	  `WRITE:
	    begin
	       case(ir_op_code)
		 `LDR0, `LDR1, `LDR2, `LDR3, `LDR4, `LDR5, `LDR6, `LDR7, `LDR8,
		 `LDR9, `LDR10, `LDR11, `LDR12, `LDR13, `LDR14, `LDR15:
		   begin
		      e_mdr_a_out <= `ENABLE;
		      reg_a_sel <= ir_op_code[3:0];
		      e_reg_a_in <= `ENABLE;
		      e_pc_inc <= `ENABLE;		      
		      state <= `FETCH;		      
		   end
		 `LDC0, `LDC1, `LDC2, `LDC3, `LDC4, `LDC5, `LDC6, `LDC7, `LDC8,
		   `LDC9, `LDC10, `LDC11, `LDC12, `LDC13, `LDC14, `LDC15:
		     begin
			e_mdr_a_out <= `ENABLE;		       
			reg_a_sel <= ir_op_code[3:0];
			e_reg_a_in <= `ENABLE;			
			e_pc_inc <= `ENABLE;
			state <= `FETCH;	       
		     end
		 `LDI:
		   begin
		      e_mdr_a_out <= `ENABLE;
		      reg_a_sel <= ir_dst_reg;
		      e_reg_a_in <= `ENABLE;
		      e_pc_inc <= `ENABLE;
		      state <= `FETCH;		      		      
		   end		
		 `STR0, `STR1, `STR2, `STR3, `STR4, `STR5, `STR6, `STR7, `STR8,
		   `STR9, `STR10, `STR11, `STR12, `STR13, `STR14, `STR15:
		     begin
			reg_a_sel <= ir_op_code[3:0];
			e_reg_a_out <= `ENABLE;
			e_mdr_a_in <= `ENABLE;
			mem_wr <= `ENABLE;		      
			state <= `REQUEST_STORE;		      
		     end
		 `STI:
		   begin
		      reg_a_sel <= ir_dst_reg;
		      e_reg_a_out <= `ENABLE;
		      e_mdr_a_in <= `ENABLE;
		      mem_wr <= `ENABLE;		      
		      state <= `REQUEST_STORE;		      
		     end		   
		 `MOV, `ADD, `ADC, `SUB, `SHL, `SHR, `SAR, `AND, `OR, `XOR:				   
		   begin
		      reg_a_sel <= ir_dst_reg;
		      e_reg_a_in <= `ENABLE;
		      e_ac_a_out <= `ENABLE;
		      e_pc_inc <= `ENABLE;
		      state <= `FETCH;	       
		   end
	       endcase
	    end 
	  `HALT_STATE:
	    halt_computer <= `ENABLE;	  
	endcase	
     end
   
   /* Na silaznoj putanji casovnika se iskljucuju svi signali (osim onih
      koji kontrolisu memoriju, oni se iskljucuju eksplicitno u gornjem
      konacnom automatu, kada se dobije odgovarajuci signal od memorije) */
   always @(negedge clk)
     begin
	e_reg_a_in <= `DISABLE;
	e_reg_b_in <= `DISABLE;
	e_reg_a_out <= `DISABLE;
	e_reg_b_out <= `DISABLE;
	e_alu <= `DISABLE;
	e_psw_in <= `DISABLE;      
	e_ac_alu_in <= `DISABLE;
	e_ac_a_out <= `DISABLE;
	e_ac_b_out <= `DISABLE;
	e_pc_a_in <= `DISABLE;
	e_pc_b_in <= `DISABLE;
	e_pc_a_out <= `DISABLE;
	e_pc_b_out <= `DISABLE;
	e_pc_inc <= `DISABLE;	
	e_ir_a_in <= `DISABLE;
	e_ir_b_in <= `DISABLE;	
	e_mar_a_in <= `DISABLE;
	e_mar_b_in <= `DISABLE;
	e_calc_addr <= `DISABLE;
	e_mdr_a_in <= `DISABLE;
	e_mdr_b_in <= `DISABLE;	
	e_mdr_a_out <= `DISABLE;
	e_mdr_b_out <= `DISABLE;
	e_calc_const <= `DISABLE;	
     end

   /* DEBUG proces prikazuje informacije o promeni stanja. */
   always @(state)
     begin
	case(state)
	  `FETCH:
	    $display($time, ": state: FETCH");
	  `REQUEST_INST:
	    $display($time, ": state: REQUEST_INST");
	  `WAIT_INST:
	    $display($time, ": state: WAIT_INST");
	  `DECODE:
	    $display($time, ": state: DECODE");
	  `EXECUTE:
	    $display($time, ": state: EXECUTE");
	  `WRITE:
	    $display($time, ": state: WRITE");
	  `REQUEST_LOAD:
	    $display($time, ": state: REQUEST_LOAD");
	  `WAIT_LOAD:
	    $display($time, ": state: WAIT_LOAD");
	  `REQUEST_STORE:
	    $display($time, ": state: REQUEST_STORE");
	  `WAIT_STORE:
	    $display($time, ": state: WAIT_STORE");
	  `HALT_STATE:
	    $display($time, ": state: HALT_STATE");
	endcase
     end
   
endmodule // cu
