/* Operacije ALU jedinice */
`define ALU_NOP    4'd0
`define ALU_ADD    4'd1
`define ALU_ADC   4'd2
`define ALU_SUB    4'd3
`define ALU_SHL    4'd4
`define ALU_SHR    4'd5
`define ALU_SAR    4'd6
`define ALU_AND    4'd7
`define ALU_OR     4'd8
`define ALU_XOR    4'd9

/* Model ALU jedinice. Izracunava rezultat operacije kao i 
 flegove kada god je e_in ukljucen */
module alu(e_in, x, y, pc, op, r, o, s, z, c);
   input e_in;  // signal za aktivaciju kola
   input[31:0] x, y; // operandi
   input       pc;   // prethodni prenos
   input [3:0] op;   // operacija
   output reg[31:0] r;   // rezultat
   output o, s, z, c; // flegovi

   reg 	  o, c;
   
   
   assign s = r[31]; // sign flag je uvek najvisi bit rezultata
   assign z = ~|r;   // zero flag je 1 akko je rezultat 00000000

   // r, c i o racunamo u procesu koji se aktivira kad god se neki
   // od ulaznih signala promeni
   always @(e_in or x or y or pc or op)
     if(e_in)
       begin
	  case(op)
	    `ALU_NOP:
	      begin
		 r = y;
		 o = 1'b0;
		 c = 1'b0;		 
	      end
	    `ALU_ADD:
	      begin
		 {c, r} = x + y;
		 o = x[31] ~^ y[31] && x[31] ^ r[31];		 
	      end
	    `ALU_ADC:
	      begin
		 {c, r} = x + y + pc;
		 o = x[31] ~^ y[31] && x[31] ^ r[31];		 		 
	      end
	    `ALU_SUB:
	      begin
		 {c, r} = x - y;
		 o = r[31] ~^ y[31] && x[31] ^ y[31];		 
	      end
	    `ALU_SHL:
	      begin
		 {c, r} = { 1'b0, x } << y;
		 o = x[31] ^ r[31];		 
	      end
	    `ALU_SHR:
	      begin
		 {r, c} = { x, 1'b0 } >> y;
		 o = 1'b0;		 
	      end
	    `ALU_SAR:
	      begin
		 // Aritmeticko siftovanje >>> ne radi u iverilog-u. Zato
		 // ga simuliramo tako sto dopisemo 32 puta bit znaka sa
		 // leve strane, a onda sve to zajedno pomerimo u desno.
		 {r, c} =  { {32{x[31]}},  x, 1'b0 } >> y;
		 o = 1'b0;		 
	      end
	    `ALU_AND:
	      begin
		 r = x & y;
		 c = 1'b0;
		 o = 1'b0;		
	      end
	    `ALU_OR:
	      begin
		 r = x | y;
		 c = 1'b0;
		 o = 1'b0;		
	      end
	    `ALU_XOR:
	      begin
		 r = x ^ y;
		 c = 1'b0;
		 o = 1'b0;		
	      end
	  endcase
       end
   
   
endmodule // alu

   
