#include <iostream>
#include <vector>
#include <vector>
#include <cstdlib>

using namespace std;

typedef unsigned Variable;  

typedef unsigned Literal;

enum Polarity { P_POSITIVE, P_NEGATIVE };

Literal litFromVar(Variable v, Polarity p)
{
  return p == P_POSITIVE ? v << 1 : (v << 1) | 1;
}

Variable varFromLit(Literal l)
{
  return l >> 1;
}

bool isPositive(Literal l)
{
  return !(l & 1);
}

bool isNegative(Literal l)
{
  return l & 1;
}

Literal oppositeLiteral(Literal l)
{
  return l ^ 1;
}


int intFromLit(Literal l)
{
  return isPositive(l) ? (int)varFromLit(l) + 1 : -(int)(varFromLit(l) + 1);
}

Literal litFromInt(int i)
{
  return i > 0 ? litFromVar(i - 1, P_POSITIVE) : litFromVar(-i - 1, P_NEGATIVE);
}

typedef vector<Literal> Clause;

typedef vector<Clause> Formula;

void printClause(const Clause & c, ostream & ostr)
{
  for(Clause::const_iterator i = c.begin(); i != c.end(); i++)
    ostr << intFromLit(*i) << " ";
  ostr << endl;
}

enum ExtendedBoolean { B_TRUE, B_FALSE, B_UNDEFINED };

ExtendedBoolean operator ! (ExtendedBoolean b)
{
    switch(b)
    {
    case B_TRUE:
      return B_FALSE;
      break;
    case B_FALSE:
      return B_TRUE;
      break;
    }
     
  return B_UNDEFINED;
}

class Valuation {
private:
  vector<ExtendedBoolean> _values;
  vector< pair<Literal, unsigned> > _stack;
  unsigned _curr_level;
public:
  Valuation(unsigned num_of_vars)
    :_values(num_of_vars, B_UNDEFINED),
     _curr_level(0)
  {}
  
  unsigned currentLevel() const
  {
    return _curr_level;
  }

  void push(Literal l, bool decide = false)
  {
    if(decide)
      _curr_level++;

    _stack.push_back(make_pair(l, _curr_level));
    _values[varFromLit(l)] = isPositive(l) ? B_TRUE : B_FALSE;
  }


  Literal backtrack()
  {
    Literal l;
    while(_stack.back().second == _curr_level)
      {
	_values[varFromLit(_stack.back().first)] = B_UNDEFINED;
	l = _stack.back().first;
	_stack.pop_back();
      }
    _curr_level--;
    return l;
  }

  bool findFirstUndefinedVariable(Variable & v) const
  {
    for(unsigned i = 0; i < _values.size(); i++)
      if(_values[i] == B_UNDEFINED)
	{
	  v = i;
	  return true;
	}
    return false;
  }

  ExtendedBoolean variableValue(Variable v) const
  {
    return _values[v];
  }


  ExtendedBoolean literalValue(Literal l) const
  {
    return isPositive(l) ? 
      _values[varFromLit(l)] : 
      !_values[varFromLit(l)];
  }

  bool isClauseFalse(const Clause & c) const
  {
    for(Clause::const_iterator i = c.begin(); i != c.end(); i++)
      {
	if(literalValue(*i) == B_TRUE || literalValue(*i) == B_UNDEFINED)
	  return false;
      }
    return true;
  }
  
  void printValuation(ostream & ostr) const
  {
    for(unsigned i = 0; i < _values.size(); i++)
      {
	if(_values[i] == B_UNDEFINED)
	  ostr << "U ";
	else if(_values[i] == B_TRUE)
	  ostr << (int)(i + 1) << " ";
	else
	  ostr <<  -(int)(i + 1) << " ";
      }
    ostr << 0 << endl;
  }

  void printStack(ostream & ostr) const
  {
    unsigned level  = 0;
    for(unsigned i = 0; i < _stack.size(); i++)
      {
	if(_stack[i].second > level)
	  {
	    ostr << "| ";
	    level++;
	  }
	ostr << intFromLit(_stack[i].first)  << " ";
      }
    ostr << endl;
  }
  
};


int skipSpaces(istream & istr)
{
  int c;
  while((c = istr.get()) == ' ' || c == '\t' || c == '\n');
  return c;
}

void skipRestOfLine(istream & istr)
{
  while(istr.get() != '\n');
}

bool readDIMACS(Formula & f, unsigned & num_of_vars, istream & istr)
{
  unsigned num_of_clauses;
  int c;

  // Preskace komentare
  while((c = skipSpaces(istr)) == 'c')
    skipRestOfLine(istr);

  // Cita liniju p cnf nvars nclauses
  if(c != 'p')
    return false;
  else
    {
      string s;
      istr >> s;
      if(s != "cnf")
	return false;
      
      istr >> num_of_vars;
      istr >> num_of_clauses;
    }

  // Citamo klauze
  for(unsigned i = 0; i < num_of_clauses; i++)
    {
      Clause c;
      int n;
      istr >> n; 
      while(!istr.eof() && !istr.fail() && n != 0)
	{
	  c.push_back(litFromInt(n));
	  istr >> n;
	}
      
      if(istr.eof() || istr.fail())
	return false;

      f.push_back(c);
    }
  return true;
}

class Solver {
private:
  Formula _formula;
  Valuation _val;

public:
  Solver(const Formula & f, unsigned num_of_vars)
    :_formula(f),
     _val(num_of_vars)
  {}
  
  bool checkConflict(unsigned & i)
  {
    for(i = 0; i < _formula.size(); i++)
      if(_val.isClauseFalse(_formula[i]))
	{
	  return true;
	}
    return false;
  }

  bool chooseDecisionLiteral(Literal & l)
  {
    Variable v;
    if(!_val.findFirstUndefinedVariable(v))
      return false;
    else
      {
	l = litFromVar(v, P_POSITIVE);
	return true;
      }
  }

  bool canBacktrack()
  {
    return _val.currentLevel() > 0;
  }


  void applyDecide(Literal l)
  {
    _val.push(l, true);
    
#ifndef NDEBUG
    cerr << "Decide: level: " 
    	 << _val.currentLevel() 
    	 << " literal: " 
    	 << intFromLit(l) << endl;
#endif
  }


  void applyBacktrack()
  {
    
    Literal l = _val.backtrack();
    
#ifndef NDEBUG 
    cerr << "Backtrack: level: " 
    	 << _val.currentLevel()  
    	 << " literal: " 
    	 << intFromLit(l) << endl;
#endif  
    _val.push(oppositeLiteral(l));
  }

  bool solve()
  {
    Literal l;
    unsigned i;

    while(true)
      {

	if(checkConflict(i))
	  {
#ifndef NDEBUG
	    cerr << "Conflict: ";
	    printClause(_formula[i], cerr);
#endif	    
	    if(canBacktrack())
	      {
		applyBacktrack();
		
#ifndef NDEBUG
		cerr << "Stack: ";
		_val.printStack(cerr);
#endif
	      }
	    else
	      return false;
	  }	
	else if(chooseDecisionLiteral(l))
	  {
	    applyDecide(l);
#ifndef NDEBUG
	    cerr << "Stack: ";
	    _val.printStack(cerr);
#endif
	  }
	else
	  return true;
      }
  }

  const Valuation & getValuation() const
  {
    return _val;
  }
 
};

int main()
{
  unsigned num_of_vars;
  Formula f;
  
  if(!readDIMACS(f, num_of_vars, cin))
    {
      cerr << "Error reading input file" << endl;
      exit(1);
    }

  Solver solver(f, num_of_vars);

  if(!solver.solve())
    {
      cout << "UNSAT" << endl;
    }
  else
    {
      cout << "SAT" << endl;
      solver.getValuation().printValuation(cout);
    }

  return 0;
}
