#include <iostream>
#include <vector>
#include <cstdlib>
#include <set>
#include <algorithm>

using namespace std;


/*
  Zadatak sa kolokvijuma (april 2016.).

  Zadatak je trazio da se implementira pojednostavljena verzija DP
  procedure.  Ova procedura podrazumeva eliminisanje jedne po jedne
  promenljive iz CNF formule primenom pravila rezolucije. Ako se
  izvede prazna klauza, tada je formula nezadovoljiva. Ako se
  eliminisu sve klauze, tada je formula zadovoljiva.
 */

typedef unsigned Variable;
typedef unsigned Literal;

enum Polarity { P_POSITIVE, P_NEGATIVE };

Literal litFromVar(Variable v, Polarity p)
{
  return p == P_POSITIVE ? v << 1 : (v << 1) | 1;
}

Variable varFromLit(Literal l)
{
  return l >> 1;
}

bool isPositive(Literal l)
{
  return !(l & 1);
}

bool isNegative(Literal l)
{
  return l & 1;
}

Literal oppositeLiteral(Literal l)
{
  return l ^ 1;
}

int intFromLit(Literal l)
{
  return isPositive(l) ? (int)varFromLit(l) + 1 : -(int)(varFromLit(l) + 1);
}

Literal litFromInt(int i)
{
  return i > 0 ? litFromVar(i - 1, P_POSITIVE) : litFromVar(-i - 1, P_NEGATIVE);
}

typedef set<Literal> Clause;
typedef set<Clause> Formula;

void printClause(const Clause & c, ostream & ostr)
{
  for(Literal lit : c)
    ostr << intFromLit(lit) << " ";
  ostr << endl;
}


/* Ovaj deo na kolokvijumu nije trebalo pisati, jer se nije trazilo da 
   program moze da cita DIMACS ulaz */

int skipSpaces()
{
  int c;
  while((c = cin.get()) == ' ' || c == '\t' || c == '\n');
  return c;
}

void skipRestOfLine()
{
  while(cin.get() != '\n');
}

bool readDIMACS(Formula & f, unsigned & num_of_vars)
{
  unsigned num_of_clauses;
  int c;

  // Skip comment lines
  while((c = skipSpaces()) == 'c')
    skipRestOfLine();

  // read p line
  if(c != 'p')
    return false;
  else
    {
      string s;
      cin >> s;
      if(s != "cnf")
	return false;
      
      cin >> num_of_vars;
      cin >> num_of_clauses;
    }

  // Read clauses
  for(unsigned i = 0; i < num_of_clauses; i++)
    {
      Clause c;
      int n;
      cin >> n; 
      while(!cin.eof() && !cin.fail() && n != 0)
	{
	  c.insert(litFromInt(n));
	  cin >> n;
	}
      
      if(cin.eof() || cin.fail())
	return false;

      f.insert(c);
    }
  return true;
}

/************************************************************************/

class Solver {
private:
  Formula _formula;
  unsigned _num_of_vars;

public:
  Solver(const Formula & f, unsigned num_of_vars)
    :_formula(f),
     _num_of_vars(num_of_vars)
  {}

  bool containsLiteral(const Clause & c, Literal l)
  {
    return c.find(l) != c.end();
  }
  
  bool resolve(const Clause & c1, const Clause c2, Variable v,  Clause & r)
  {
    /* Pretpostavlja se da klauza c1 sadrzi v, a c2 ~v. Takodje, pretpostavlja
       se da c1 i c2 nisu tautologije */
    r = c1;
    r.erase(litFromVar(v, P_POSITIVE));
    
    for(Literal l : c2)
      if(l != litFromVar(v, P_NEGATIVE))
	{
	  if(r.find(oppositeLiteral(l)) == r.end())
	    r.insert(l);
	  else return false; // tautologija
	}
    return true;
  }

  bool eliminate(Variable v)
  {
    Formula nf;  // novi skup klauza
    
    Literal pv = litFromVar(v, P_POSITIVE);
    Literal nv = litFromVar(v, P_NEGATIVE);

    // Za svaku klauzu c1 iz F
    for(const Clause & c1 : _formula)
      {	       
	if(!containsLiteral(c1, pv))
	  {
	    // Ako c1 ne sadzi ni v ni ~v, dodajemo je u novi skup
	    if(!containsLiteral(c1, nv))
	      nf.insert(c1);

	    // U svakom slucaju, ako ne sadrzi v, preskacemo je
	    continue;
	  }

	// Ako c1 sadrzi v, tada za svaku klauzu c2 iz F...
	for(const Clause & c2 : _formula)
	  {

	    // ... koja sadrzi nv
	    if(!containsLiteral(c2, nv))
	      continue;

	    // radimo rezoluciju po v
	    Clause r;
	    // ako nije tautologija...
	    if(resolve(c1, c2, v, r))
	      {
		// ... i nije prazna klauza
		if(!r.empty())
		  nf.insert(r); // ... dodajemo je u skup
		else return false; // ako je prazna, tada je formula
				   // nezadovoljiva
	      }
	  }
      }
    _formula = move(nf);  // zamenjujemo stari skup klauza novim
    return true;    
  }

  bool checkSat()
  {
    // Eliminisemo promenljive jednu po jednu. Ako se tom prilikom
    // izvede prazna klauza, tada je formula nezadovoljiva.
    for(unsigned i = 0; i < _num_of_vars; i++)
      if(!eliminate(i))
	return false;
    return true;
  }
  
};

int main()
{
  unsigned num_of_vars;

  Variable p = 0;
  Variable q = 1;
  Variable r = 2;
  Literal pp = litFromVar(p, P_POSITIVE);
  Literal np = litFromVar(p, P_NEGATIVE);
  Literal pq = litFromVar(q, P_POSITIVE);
  Literal nq = litFromVar(q, P_NEGATIVE);
  Literal pr = litFromVar(r, P_POSITIVE);
  Literal nr = litFromVar(r, P_NEGATIVE);
  
  Formula f = { { pp, nq, pr }, { np, nq }, { pq, pp, nr }, { pq, np, nr } };

  // Ako zelimo da citamo iz dimacs fajla, tada treba da otkomentarisemo ovo
  /*  if(!readDIMACS(f, num_of_vars))
    {
      cerr << "Error reading input file" << endl;
      exit(1);
    }
  */
  
  Solver solver(f, num_of_vars);

  if(!solver.checkSat())
    {
      cout << "UNSAT" << endl;
    }
  else
    {
      cout << "SAT" << endl;
    }

  return 0;
}
