#include <iostream>
#include <memory>
#include <set>
#include <map>
#include <string>
#include <algorithm>

/*
   Resenje zadatka sa kolokvijuma (maj 2014). 
   
   U zadatku je trebalo odrediti NNF udaljenost date formule.  Ova
   udaljenost se definise kao zbir visina svih negacija u formuli. Pod
   visinom negacije ~F podrazumevamo visinu sintaksnog stabla njene
   podformule F.

 */

using namespace std;

class BaseFormula;

typedef shared_ptr<BaseFormula> Formula;


class BaseFormula : public enable_shared_from_this<BaseFormula> {

public:
  enum Type { T_TRUE, T_FALSE, T_ATOM, T_NOT, T_AND, T_OR, T_IMP };
  
  virtual Type getType() const = 0;
  virtual void printFormula(ostream & ostr) const = 0;

  /* Pomocna virtuelna metoda calculate() istovremeno racuna i visinu
     i nnf udaljenost podformule za koju se poziva. Ovim izbegavamo da
     se dva puta prolazi kroz stablo, sto kod cini efikasnijim */
  virtual void calculate(unsigned & height, unsigned & nnf_distance) const = 0;

  /* Funkcija nnfDistance() se sada svodi na poziv funkcije calculate()
     za datu formulu */
  unsigned nnfDistance() const
  {
    unsigned height, nnf_distance;
    calculate(height, nnf_distance);
    return nnf_distance;
  }
  
  virtual ~BaseFormula() {}
};

ostream & operator << (ostream & ostr, const Formula & f)
{
  f->printFormula(ostr);
  return ostr;
}


class AtomicFormula : public BaseFormula {
public:

  virtual void calculate(unsigned & height, unsigned & nnf_distance) const
  {
    /* Izlaz iz rekurzije: za listove u stablu visina je 0, kao i nnf udaljenost
       (jer nema negacija) */
    height = 0;
    nnf_distance = 0;
  }
};


class LogicConstant : public AtomicFormula {
public:

};

class True  : public LogicConstant {
public:
  virtual Type getType() const
  {
    return T_TRUE;
  }

  virtual void printFormula(ostream & ostr) const 
  {
    ostr << "TRUE";
  }
};

class False  : public LogicConstant {
public:
  virtual Type getType() const
  {
    return T_FALSE;
  }

  virtual void printFormula(ostream & ostr) const 
  {
    ostr << "FALSE";
  }
};

class Atom : public AtomicFormula {
private:
  string _name;
public:
  Atom(const string & name)
    :_name(name)
  {}

  Atom(string && name)
    :_name(move(name))
  {}
  
  const string & getName() const
  {
    return _name;
  }

  virtual Type getType() const
  {
    return T_ATOM;
  }

  virtual void printFormula(ostream & ostr) const 
  {
    ostr << _name;
  }

};

class UnaryConnective : public BaseFormula {
protected:
  Formula _op;
public:
  UnaryConnective(const Formula & op)
    :_op(op)
  {}
  
  const Formula & getOperand() const
  {
    return _op;
  }

};

class Not : public UnaryConnective {
public:
  using UnaryConnective::UnaryConnective;
  
  virtual Type getType() const
  {
    return T_NOT;
  }

  virtual void printFormula(ostream & ostr) const 
  {
    ostr << "(~" << _op << ")";
  }

  virtual void calculate(unsigned & height, unsigned & nnf_distance) const
  {
    unsigned op_height, op_nnf_distance;

    /* Najpre rekurzivno izracunamo visinu i nnf udaljenost podformule */
    _op->calculate(op_height, op_nnf_distance);

    /* nnf udaljenost nase formule je jednako zbiru nnf udaljenosti
       podformule (to je zbir visina svih negacija u podformuli)
       i visine negacije u korenu formule, a ona je jednaka visini
       podformule */
    nnf_distance = op_nnf_distance + op_height;
    height = op_height + 1;
  }
  
};

class BinaryConnective : public BaseFormula {
protected:
  Formula _op1;
  Formula _op2;
public:
  BinaryConnective(const Formula & op1, 
		   const Formula & op2)
    :_op1(op1),
     _op2(op2)
  {}
  
  const Formula & getOperand1() const
  {
    return _op1;
  }

  const Formula & getOperand2() const
  {
    return _op2;
  }

  virtual void calculate(unsigned & height, unsigned & nnf_distance) const
  {
    unsigned op1_height, op2_height, op1_nnf_distance, op2_nnf_distance;

    /* Najpre rekurzivno racunamo visine i nnf udaljenosti podfrmula */
    _op1->calculate(op1_height, op1_nnf_distance);
    _op2->calculate(op2_height, op2_nnf_distance);

    /* Visina stabla je jednaka vecoj od visina podstabala, plus 1 */
    height = max(op1_height, op2_height) + 1;

    /* nnf udaljenost je jednaka zbiru nnf udaljenosti leve i desne podformule */
    nnf_distance = op1_nnf_distance + op2_nnf_distance;
  }

};

class And : public BinaryConnective {
public:
  using BinaryConnective::BinaryConnective;
  
  virtual Type getType() const
  {
    return T_AND;
  }

  virtual void printFormula(ostream & ostr) const 
  {
    ostr << "(" << _op1 <<  " /\\ " << _op2 << ")";
  }

};

class Or : public BinaryConnective {
public:
  using BinaryConnective::BinaryConnective;


  virtual Type getType() const
  {
    return T_OR;
  }

  virtual void printFormula(ostream & ostr) const 
  {
    ostr << "(" << _op1 << " \\/ " << _op2 << ")";
  }

};

class Imp : public BinaryConnective {
public:
  using BinaryConnective::BinaryConnective;

  virtual Type getType() const
  {
    return T_IMP;
  }

  virtual void printFormula(ostream & ostr) const 
  {
    ostr << "(" << _op1 <<  " => " << _op2 << ")";
  }

};


int main()
{
  /* Kreiramo trazenu formulu, prikazujemo je na izlazu i odredjujemo
     njenu nnf udaljenost */
  
  Formula x = make_shared<Atom>("x");
  Formula y = make_shared<Atom>("y");
  Formula z = make_shared<Atom>("z");
  Formula nx = make_shared<Not>(x);
  Formula nz = make_shared<Not>(z);
  Formula ynz = make_shared<Or>(y, nz);
  Formula xynz = make_shared<Imp>(x, ynz);
  Formula nxynz = make_shared<Not>(xynz);
  Formula yz = make_shared<Imp>(y, z);
  Formula nyz = make_shared<Not>(yz);
  Formula xyz = make_shared<Or>(nx, nyz);
  Formula nxyz = make_shared<Not>(xyz);
  Formula f = make_shared<And>(nxynz, nxyz);

  cout << f << endl;
  cout << f->nnfDistance() << endl;
  
  
  return 0;
}
