#include <iostream>
#include <memory>
#include <set>
#include <map>
#include <string>
#include <algorithm>
#include <vector>

/*
   Resenje zadatka sa kolokvijuma (maj 2015). 
   
   U zadatku se trazilo:
   *) da se napise funkcija koja ispituje da li je formula f logicka
      posledica skupa D.
   *) da se napise funkcija koja ispituje da li je dati skup D
      nezavisan skup (skup formula D je nezavisan ako za svaku
      formulu f iz D vazi da f nije logicka posledica ostalih
      formula iz D). 
   *) da se za dati skup D pronadje jedan njegov minimalan ekvivalentan
      podskup. Za podskup D' skupa D kazemo da je ekvivalentan sa D ako
      za svaku formulu f vazi da je f logicka posledica D' akko je 
      logicka posledica od D. 
 */


using namespace std;


typedef set<string> AtomSet;

class Valuation {
private:
  map<string, bool> _map;
public:
  bool getValue(const string & x) const
  {
    auto it = _map.find(x);
    if(it != _map.end())
      return it->second;
    else
      throw "Unknown variable";
  }

  void setValue(const string & x, bool value)
  {
    _map[x] = value;
  }

  void initValuation(const AtomSet & s)
  {
    _map.clear();
    for(const auto & x : s)
      {
	_map.insert(make_pair(x, false));
      }
  }

  bool nextValuation()
  {
    for(auto it = _map.rbegin(), it_end = _map.rend(); it != it_end; ++it)
      {
	it->second = !it->second;
	if(it->second)
	  return true;
      }
    return false;
  }

};


class BaseFormula;

typedef shared_ptr<BaseFormula> Formula;


class BaseFormula : public enable_shared_from_this<BaseFormula> {

public:
  enum Type { T_TRUE, T_FALSE, T_ATOM, T_NOT, T_AND, T_OR, T_IMP };
  
  virtual Type getType() const = 0;
  virtual void printFormula(ostream & ostr) const = 0;
  virtual bool eval(const Valuation & v) const = 0;
  virtual void getAtoms(AtomSet & s) const = 0;
  
  virtual ~BaseFormula() {}
};

ostream & operator << (ostream & ostr, const Formula & f)
{
  f->printFormula(ostr);
  return ostr;
}

/* Skup formula cemo predstavljati vektorom. Iako ovo nije
   najprecizniji nacin da se cuva skup formula (jer vektor moze
   sadrzati duplikate, a skup ne moze), svakako je najjednostavniji.
 */
typedef vector<Formula> FormulaSet;

/* Ispis skupa formula */
ostream & operator << (ostream & ostr, const FormulaSet & fs)
{
  ostr << "{ ";
  for(const auto & f : fs)
    ostr << f << " ";
  ostr << "}";

  return ostr;
}

/* Funkcija ispituje da li je formula b logicka posledica skupa formula fs */
bool isConsequence(const FormulaSet &  fs, const Formula &  b)
{
  Valuation v;
  AtomSet atoms;

  /* Sakupljamo sve atome koji se pojavljuju u formulama */
  for(const auto & a : fs)
    a->getAtoms(atoms);
  b->getAtoms(atoms);

  /* Inicijalizujemo valuaciju */
  v.initValuation(atoms);
  
  do {

    /* Odredjujemo da li valuacija v zadovoljava sve formule iz fs */
    bool s_sat = true;
    for(const auto & a : fs)
      if(!a->eval(v))
	{
	  s_sat = false;
	  break;
	}

    /* Ako zadovoljava sve formule iz fs, a ne zadovoljava b, onda b nije
       logicka posledica */
    if(s_sat && !b->eval(v))
      return false;
    
  } while(v.nextValuation());

  /* Jeste logicka posledica */
  return true;
} 


/* Funkcija ispituje da li je skup fs nezavisan */
bool isIndependant(const FormulaSet & fs)
{
  FormulaSet d = fs;
  for(unsigned i = 0; i < d.size(); i++)
    {
      Formula f = d[i];

      /* Izbacujemo formulu f=d[i] iz skupa */
      swap(d[i], d.back());
      d.pop_back();

      /* Ako je f logicka posledica ostatka skupa d, tada polazni skup
	 nije nezavisan */
      if(isConsequence(d, f))
	return false;

      /* Vracamo f u skup (na isto mesto u vektoru) */
      d.push_back(f);
      swap(d[i], d.back());
    }

  /* Skup je nezavisan */
  return true;
}

/* Funkcija pronalazi jedan minimalni nezavisni podskup skupa fs */
FormulaSet findMinimal(const FormulaSet & fs)
{
  FormulaSet d = fs;
  int i = 0;

  while(i < (int)d.size())
    {
      // Za svaku formulu f = d[i]
      Formula f = d[i];

      // izbacujemo f iz skupa
      swap(d[i], d.back());
      d.pop_back();

      // Ako f nije logicka posledica ostatka skupa d, tada je f
      // neophodno zadrzati u skupu, pa ga vracamo na svoje mesto
      // i prelazimo na sledecu formulu. U suprotnom, formula f
      // je suvisna, i mozemo je izbaciti trajno iz skupa (indeks
      // i ostaje na istoj poziciji na kojoj se sada nalazi sledeca
      // formula koju treba proveriti). 
      if(!isConsequence(d, f))
	{
	  d.push_back(f);
	  swap(d[i], d.back());
	  i++;
	}
    }
  return d;
}


class AtomicFormula : public BaseFormula {
public:

};


class LogicConstant : public AtomicFormula {
public:

  virtual void getAtoms(AtomSet & s) const
  {
    // No atoms
  }

};

class True  : public LogicConstant {
public:
  virtual Type getType() const
  {
    return T_TRUE;
  }

  virtual void printFormula(ostream & ostr) const 
  {
    ostr << "TRUE";
  }

  virtual bool eval(const Valuation & v) const
  {
    return true;
  }

};

class False  : public LogicConstant {
public:
  virtual Type getType() const
  {
    return T_FALSE;
  }

  virtual void printFormula(ostream & ostr) const 
  {
    ostr << "FALSE";
  }

  virtual bool eval(const Valuation & v) const
  {
    return false;
  }
};

class Atom : public AtomicFormula {
private:
  string _name;
public:
  Atom(const string & name)
    :_name(name)
  {}

  Atom(string && name)
    :_name(move(name))
  {}
  
  const string & getName() const
  {
    return _name;
  }

  virtual Type getType() const
  {
    return T_ATOM;
  }

  virtual void printFormula(ostream & ostr) const 
  {
    ostr << _name;
  }
  
  virtual void getAtoms(AtomSet & s) const
  {
    s.insert(_name);
  }

  virtual bool eval(const Valuation & v) const
  {
    return v.getValue(_name);
  }

  
};

class UnaryConnective : public BaseFormula {
protected:
  Formula _op;
public:
  UnaryConnective(const Formula & op)
    :_op(op)
  {}
  
  const Formula & getOperand() const
  {
    return _op;
  }

  virtual void getAtoms(AtomSet & s) const
  {
    _op->getAtoms(s);
  }

};

class Not : public UnaryConnective {
public:
  using UnaryConnective::UnaryConnective;
  
  virtual Type getType() const
  {
    return T_NOT;
  }

  virtual void printFormula(ostream & ostr) const 
  {
    ostr << "(~" << _op << ")";
  }

  virtual bool eval(const Valuation & v) const
  {
    return !_op->eval(v);
  }

};

class BinaryConnective : public BaseFormula {
protected:
  Formula _op1;
  Formula _op2;
public:
  BinaryConnective(const Formula & op1, 
		   const Formula & op2)
    :_op1(op1),
     _op2(op2)
  {}
  
  const Formula & getOperand1() const
  {
    return _op1;
  }

  const Formula & getOperand2() const
  {
    return _op2;
  }

  virtual void getAtoms(AtomSet & s) const
  {
    _op1->getAtoms(s);
    _op2->getAtoms(s);
  }

};

class And : public BinaryConnective {
public:
  using BinaryConnective::BinaryConnective;
  
  virtual Type getType() const
  {
    return T_AND;
  }

  virtual void printFormula(ostream & ostr) const 
  {
    ostr << "(" << _op1 <<  " /\\ " << _op2 << ")";
  }


  virtual bool eval(const Valuation & v) const
  {
    return _op1->eval(v) && _op2->eval(v);
  }

};

class Or : public BinaryConnective {
public:
  using BinaryConnective::BinaryConnective;


  virtual Type getType() const
  {
    return T_OR;
  }

  virtual void printFormula(ostream & ostr) const 
  {
    ostr << "(" << _op1 << " \\/ " << _op2 << ")";
  }

  virtual bool eval(const Valuation & v) const
  {
    return _op1->eval(v) || _op2->eval(v);
  }

};

class Imp : public BinaryConnective {
public:
  using BinaryConnective::BinaryConnective;

  virtual Type getType() const
  {
    return T_IMP;
  }

  virtual void printFormula(ostream & ostr) const 
  {
    ostr << "(" << _op1 <<  " => " << _op2 << ")";
  }


  virtual bool eval(const Valuation & v) const
  {
    return !_op1->eval(v) || _op2->eval(v);
  }

};


int main()
{
  
  Formula p = make_shared<Atom>("p");
  Formula q = make_shared<Atom>("q");
  Formula r = make_shared<Atom>("r");
  Formula s = make_shared<Atom>("s");
  Formula np = make_shared<Not>(p);
  Formula nq = make_shared<Not>(q);
  Formula npq = make_shared<Or>(np, q);
  Formula pq = make_shared<Imp>(p, q);
  Formula rs = make_shared<Imp>(r, s);
  Formula nrs = make_shared<Not>(rs);
  Formula nrs_pq = make_shared<Imp>(nrs, pq);
  Formula nq_np = make_shared<Imp>(nq, np);

  FormulaSet fs { npq, nrs_pq, nq_np };


  cout << fs << endl;
  
  FormulaSet min_fs = findMinimal(fs);

  cout << min_fs << endl;

  
  return 0;
}
