#include <iostream>
#include <set>
#include <map>
#include <vector>
#include <memory>
#include <algorithm>
#include <iterator>

using namespace std;

/* Skup atoma (iskaznih promenljivih) */
typedef set<unsigned> AtomSet;

/* Valuacija je predstavljena mapom koja iskaznim promenljivama
   (predstavljenim unsigned tipom) dodeljuje bool vrednosti */
class Valuation {
private:
  map<unsigned, bool> _map;
public:

  void setInitial(const AtomSet & atoms);
  bool nextValuation();

  bool getValue(unsigned p) const;
  void setValue(unsigned p, bool b);
 
  void printValuation(ostream & ostr) const;  
};

ostream & operator << (ostream & ostr, const Valuation & v);


/* Deklaracija osnovne klase */
class BaseFormula;

/* Tip podatka koji predstavlja "rucku" za baratanje formulama
   (deljeni pokazivac na cvor u sintaksnom stablu). NAPOMENA:
   Deljeni pokazivaci se, nakon konstrukcije, sintaksno koriste
   na isti nacin kao i obicni pokazivaci (dostupni su operatori
   *, ->, kao i metoda get() kojom se dobija obican pokazivac
   koji je sadrzan u tom deljenom pokazivacu). 
   Za vise informacija o koriscenju deljenih pokazivaca, videti:
   http://www.cplusplus.com/reference/memory/shared_ptr/ */
typedef shared_ptr<BaseFormula> Formula;

/* Tipovi podataka za predstavljanje formule u obliku 
   liste listi literala. Ovi tipovi se mogu koristiti
   i za CNF i za DNF. Podrazumeva se da pokazivaci na
   formule u LiteralList pokazuju ili na atome ili na
   negacije atoma. */
typedef vector<Formula> LiteralList;
typedef vector<LiteralList> LiteralListList;

/* Apstraktna klasa za predstavljanje formula */
class BaseFormula : public enable_shared_from_this<BaseFormula> {

public:
  /* Tip formule (tj. vodeceg veznika formule) */
  enum Type { T_TRUE, T_FALSE, T_ATOM, T_NOT, T_AND, T_OR, T_IMP, T_IFF };
  
  virtual void printFormula(ostream & ostr) const = 0;
  virtual Type getType() const = 0;
  virtual unsigned complexity() const = 0; 
  virtual bool equalTo(const Formula & f) const = 0;
  virtual void getAtoms(AtomSet & set) const = 0;
  virtual bool eval(const Valuation & v) const = 0;
  virtual Formula substitute(const Formula & a, const Formula & b) = 0;
  virtual Formula simplify() = 0;
  virtual Formula nnf() = 0;
  virtual LiteralListList listCNF() = 0;
  virtual LiteralListList listDNF() = 0;
  virtual ~BaseFormula() {}

  void printTruthTable(ostream & ostr) const;
  bool isSatisfiable(Valuation & v) const;
  bool isTautology() const;
};

ostream & operator << (ostream & ostr, const Formula & f);

/* Atomicke formule (True, False i Atom) */
class AtomicFormula : public BaseFormula {

public:
  virtual unsigned complexity() const;
  virtual Formula substitute(const Formula & a, const Formula & b);
  virtual Formula simplify();
  virtual Formula nnf();
};

/* Logicke konstante (True i False) */
class LogicConstant : public AtomicFormula {

public:
  virtual bool equalTo(const Formula & f) const;
  virtual void getAtoms(AtomSet & set) const; 
};

class True : public LogicConstant {

public:
  virtual void printFormula(ostream & ostr) const;
  virtual Type getType() const;
  virtual bool eval(const Valuation & v) const; 
  virtual LiteralListList listCNF();
  virtual LiteralListList listDNF();
};

class False : public LogicConstant {

public:
  virtual void printFormula(ostream & ostr) const;
  virtual Type getType() const;
  virtual bool eval(const Valuation & v) const;
  virtual LiteralListList listCNF();
  virtual LiteralListList listDNF();
};

class Atom : public AtomicFormula {
private:
  unsigned _var_num;
public:
  Atom(unsigned num);
  unsigned getVariableNumber() const; 
  virtual void printFormula(ostream & ostr) const;
  virtual Type getType() const;
  virtual bool equalTo(const Formula &  f) const;  
  virtual void getAtoms(AtomSet & set) const; 
  virtual bool eval(const Valuation & v) const; 
  virtual LiteralListList listCNF();
  virtual LiteralListList listDNF();
};

/* Unarni veznik (obuhvata samo Not) */
class UnaryConnective : public BaseFormula {
protected:
  Formula _op;
public:
  UnaryConnective(const Formula & op);
  const Formula & getOperand() const;
  virtual unsigned complexity() const;
  virtual bool equalTo(const Formula & f) const;
  virtual void getAtoms(AtomSet & set) const;
 
};

class Not : public UnaryConnective {
public:

  // Uvozimo konstruktor iz bazne klase
  using UnaryConnective::UnaryConnective;
  
  virtual void printFormula(ostream & ostr) const;
  virtual Type getType() const;
  virtual bool eval(const Valuation & v) const; 
  virtual Formula substitute(const Formula & a, const Formula & b);
  virtual Formula simplify();
  virtual Formula nnf();
  virtual LiteralListList listCNF();
  virtual LiteralListList listDNF();
  
};

/* Binarni veznik (And, Or, Imp i Iff) */
class BinaryConnective : public BaseFormula {
protected:
  Formula _op1; 
  Formula _op2;
public:
  BinaryConnective(const Formula & op1, const Formula & op2);
  const Formula & getOperand1() const;
  const Formula & getOperand2() const; 
  virtual unsigned complexity() const;
  virtual bool equalTo(const Formula & f) const;
  virtual void getAtoms(AtomSet & set) const;
 
};

class And : public BinaryConnective {
public:
  // Uvozimo konstruktor iz bazne klase
  using BinaryConnective::BinaryConnective;
  
  virtual void printFormula(ostream & ostr) const;
  virtual Type getType() const;
  virtual bool eval(const Valuation & v) const;
  virtual Formula substitute(const Formula & a, const Formula & b);
  virtual Formula simplify();
  virtual Formula nnf();
  virtual LiteralListList listCNF();
  virtual LiteralListList listDNF();

};

class Or : public BinaryConnective {
public:
  // Uvozimo konstruktor iz bazne klase
  using BinaryConnective::BinaryConnective;

  virtual void printFormula(ostream & ostr) const; 
  virtual Type getType() const;
  virtual bool eval(const Valuation & v) const; 
  virtual Formula substitute(const Formula & a, const Formula & b);
  virtual Formula simplify();
  virtual Formula nnf();
  virtual LiteralListList listCNF();
  virtual LiteralListList listDNF();

};

class Imp : public BinaryConnective {
public:
  // Uvozimo konstruktor iz bazne klase
  using BinaryConnective::BinaryConnective;

  virtual void printFormula(ostream & ostr) const;
  virtual Type getType() const; 
  virtual bool eval(const Valuation & v) const; 
  virtual Formula substitute(const Formula & a, const Formula & b);
  virtual Formula simplify();
  virtual Formula nnf();
  virtual LiteralListList listCNF();
  virtual LiteralListList listDNF();
};

class Iff : public BinaryConnective {
public:
  // Uvozimo konstruktor iz bazne klase
  using BinaryConnective::BinaryConnective;

  virtual void printFormula(ostream & ostr) const;
  virtual Type getType() const;
  virtual bool eval(const Valuation & v) const; 
  virtual Formula substitute(const Formula & a, const Formula & b);
  virtual Formula simplify();
  virtual Formula nnf();
  virtual LiteralListList listCNF();
  virtual LiteralListList listDNF();
};


// DEFINICIJE FUNKCIJA 

// FUNKCIJE ZA ODREDJIVANJE TIPA FORMULE ---------------------------------

BaseFormula::Type True::getType() const
{
  return T_TRUE;
}

BaseFormula::Type False::getType() const 
{
  return T_FALSE;
}

BaseFormula::Type Atom::getType() const 
{
  return T_ATOM;
}

BaseFormula::Type Not::getType() const
{
  return T_NOT;
}

BaseFormula::Type And::getType() const
{
  return T_AND;
}

BaseFormula::Type Or::getType() const
{
  return T_OR;
}

BaseFormula::Type Imp::getType() const
{
  return T_IMP;
}

BaseFormula::Type Iff::getType() const
{
  return T_IFF;
}

// --------------------------------------------------------------------

// FUNKCIJE ZA STAMPANJE FORMULA -----------------------------------------

void True::printFormula(ostream & ostr) const
{
  ostr << "TRUE";
}

void False::printFormula(ostream & ostr) const
{
  ostr << "FALSE";
}

void Atom::printFormula(ostream & ostr) const
{
  ostr << "p_" << _var_num;
}

void Not::printFormula(ostream & ostr) const
{
  ostr << "(~" << _op << ")";
}

void And::printFormula(ostream & ostr) const
{
  ostr << "(" << _op1 <<  " /\\ " << _op2 << ")";
}

void Or::printFormula(ostream & ostr) const
{
  ostr << "(" << _op1 << " \\/ " << _op2 << ")";
}

void Imp::printFormula(ostream & ostr) const
{
  ostr << "(" << _op1 <<  " => " << _op2 << ")";  
}

void Iff::printFormula(ostream & ostr) const
{
    ostr << "(" << _op1 << " <=> " << _op2 << ")";
}
// -----------------------------------------------------------------------

// FUNKCIJE ZA ISPITIVANJE SINTAKSNE JEDNAKOSTI FORMULA ----------------

bool LogicConstant::equalTo(const Formula & f) const
{
  return f->getType() == getType();
}

bool Atom::equalTo(const Formula & f) const
{
  return f->getType() == T_ATOM && 
    ((Atom *)f.get())->getVariableNumber() == _var_num;
}

bool UnaryConnective::equalTo(const Formula & f) const
{
  return f->getType() == getType() && 
    _op->equalTo(((UnaryConnective *)f.get())->getOperand());
}

bool BinaryConnective::equalTo(const Formula & f) const 
{
  return f->getType() == getType() && 
    _op1->equalTo(((BinaryConnective *)f.get())->getOperand1()) 
    &&  
    _op2->equalTo(((BinaryConnective *)f.get())->getOperand2());
}

// --------------------------------------------------------------------

// FUNKCIJE ZA IZRACUNAVANJE SLOZENOSTI FORMULA ------------------------

unsigned AtomicFormula::complexity() const
{
  return 0;
}

unsigned UnaryConnective::complexity() const
{
  return _op->complexity() + 1;
}

unsigned BinaryConnective::complexity() const
{
  return _op1->complexity() + _op2->complexity() + 1;
}

// -------------------------------------------------------------------



// FUNKCIJE ZA IZDVAJANJE SKUPA ATOMA FORMULE -------------------------

void LogicConstant::getAtoms(AtomSet & set) const
{}

void Atom::getAtoms(AtomSet & set) const
{
  set.insert(_var_num);
}

void UnaryConnective::getAtoms(AtomSet & set) const 
{
  _op->getAtoms(set);
}

void BinaryConnective::getAtoms(AtomSet & set) const
{
  _op1->getAtoms(set);
  _op2->getAtoms(set);
}

// -----------------------------------------------------------------------

// FUNKCIJE SUBSTITUCIJE ---------------------------------------------

Formula AtomicFormula::substitute(const Formula & a, const Formula & b)
{
  /* Kod atomickih formula, substitucija se dogadja samo 
     kada je formula koju menjamo (A) sintaksno identicna
     samoj atomickoj formuli. U suprotnom, formula ostaje
     nepromenjena. Kod svih ostalih (neatomickih) formula,
     ukoliko formula koja se menja (A) nije identicki 
     jednaka sa samom formulom, tada se substitucija 
     primenjuje rekurzivno na podformule. */
  if(equalTo(a))
    return b;
  else
    return shared_from_this();
}

Formula Not::substitute(const Formula & a, const Formula & b)
{
  if(equalTo(a))
    return b;
  else
    {
      return make_shared<Not>(_op->substitute(a, b));
    }
  
}

Formula And::substitute(const Formula & a, const Formula & b)
{
  if(equalTo(a))
    return b;
  else
    {
      return make_shared<And>(_op1->substitute(a, b), _op2->substitute(a,b));
    }
}


Formula Or::substitute(const Formula & a, const Formula & b)
{
  if(equalTo(a))
    return b;
  else
    {
      return make_shared<Or>(_op1->substitute(a, b), _op2->substitute(a,b));
    }
  
}

Formula Imp::substitute(const Formula & a, const Formula & b)
{
  if(equalTo(a))
    return b;
  else
    {
      return make_shared<Imp>(_op1->substitute(a, b), _op2->substitute(a,b));
    }
}

Formula Iff::substitute(const Formula & a, const Formula & b)
{
  if(equalTo(a))
    return b;
  else
    {
      return make_shared<Iff>(_op1->substitute(a, b), _op2->substitute(a,b));
    }
  
}
// ----------------------------------------------------------------------

// FUNKCIJE ZA ODREDJIVANJE INTERPRETACIJE FORMULE U DATOJ VALUACIJI --

bool True::eval(const Valuation & v) const
{
  return true;
}

bool False::eval(const Valuation & v) const
{
  return false;
}

bool Atom::eval(const Valuation & v) const
{
  return v.getValue(_var_num);
}

bool Not::eval(const Valuation & v) const
{
  return !_op->eval(v);
}

bool And::eval(const Valuation & v) const
{
  return _op1->eval(v) && _op2->eval(v);
}

bool Or::eval(const Valuation & v) const
{
  return _op1->eval(v) || _op2->eval(v);
}

bool Imp::eval(const Valuation & v) const
{
  return !_op1->eval(v) || _op2->eval(v);
}

bool Iff::eval(const Valuation & v) const
{
  return _op1->eval(v) == _op2->eval(v);
}

// ---------------------------------------------------------------------


// FUNKCIJE VEZANE ZA SEMANTIKU (zadovoljivost, tautologicnost, 
// logicke posledice, ekvivalencija, istinitonosne tablice...)

void BaseFormula::printTruthTable(ostream & ostr) const
{
  Valuation v;
  AtomSet atoms;
  
  /* Odredjujemo skup atoma formule */
  getAtoms(atoms);
  
  /* Kreiramo inicijalnu valuaciju za dati skup atoma (svim
     atomima se dodeljuje false vrednost inicijalno) */
  v.setInitial(atoms);
  
  /* Prolazimo kroz sve valuacije, prikazujemo valuaciju i vrednost
     formule u toj valuaciji */
  do {
    ostr << v << " | " << eval(v) << endl;      
  } while(v.nextValuation());
    
}

/* Funkcija ispituje zadovoljivost date formule. 
   Preko izlaznog parametra v vraca valuaciju koja
   zadovoljava formulu, ako takva postoji */
bool BaseFormula::isSatisfiable(Valuation & v) const
{
  AtomSet atoms;
  
  getAtoms(atoms);

  v.setInitial(atoms);
  
  do {
    if(eval(v))
      return true;
    
    } while(v.nextValuation());
  
    return false;
}

/* Funkcija ispituje tautologicnost formule */
bool BaseFormula::isTautology() const
{
  Valuation v;
  AtomSet atoms;
  
  getAtoms(atoms);
  
  v.setInitial(atoms);
  
  do {
    if(!eval(v))
      return false;
    
  } while(v.nextValuation());
  
  return true;
}

/* Funkcija proverava da li je b logicka posledica formule a */
bool isConsequence(const Formula &  a, const Formula &  b)
{
  Valuation v;
  AtomSet atoms;
  
  a->getAtoms(atoms);
  b->getAtoms(atoms);
  
  v.setInitial(atoms);
  
  do {
    /* Ako postoji valuacija u kojoj je A tacna a B netacna... */
    if(a->eval(v) && !b->eval(v))
      return false;
    
  } while(v.nextValuation());
  
  return true;
} 

bool isEquivalent(const Formula &  a, const Formula &  b)
{
  Valuation v;
  AtomSet atoms;
  
  a->getAtoms(atoms);
  b->getAtoms(atoms);
  
  v.setInitial(atoms);
  
  do {
    /* Ako postoji valuacija u kojoj formule imaju 
       razlicite interpretacije */
    if(a->eval(v) != b->eval(v))
      return false;
    
  } while(v.nextValuation());
  
  return true;
} 

// ---------------------------------------------------------------------

// FUNKCIJE ZA SIMPLIFIKACIJU -------------------------------------------

/* Simplifikacija atomicke formule je trivijalna */
Formula AtomicFormula::simplify()
{
  return shared_from_this();
}

Formula Not::simplify()
{
  /* Negacija se uproscava prema pravilima: ~True === False,
     ~False === True. Kod svih formula, simplifikacija se prvo primeni
     rekurzivno na podformule, pa se zatim primenjuju pravila. */
  Formula simp_op = _op->simplify();
  
  if(simp_op->getType() == T_TRUE)
    return make_shared<False>();
  else if(simp_op->getType() == T_FALSE)
    return make_shared<True>();
  else
    return make_shared<Not>(simp_op);
}

Formula And::simplify()
{
  /* Simplifikacija konjukcije po pravilima A /\ True === A, 
     A /\ False === False i sl. */
  Formula simp_op1 = _op1->simplify();
  Formula simp_op2 = _op2->simplify();
  
  if(simp_op1->getType() == T_TRUE)
    return simp_op2;
  else if(simp_op2->getType() == T_TRUE)
    return simp_op1;
  else if(simp_op1->getType() == T_FALSE ||
	  simp_op2->getType() == T_FALSE)
    return make_shared<False>();
  else
    return make_shared<And>(simp_op1, simp_op2);
}

Formula Or::simplify()
{
  /* Simplifikacija disjunkcije po pravilima: A \/ True === True,
     A \/ False === A, i sl. */
  Formula simp_op1 = _op1->simplify();
  Formula simp_op2 = _op2->simplify();
  
  if(simp_op1->getType() == T_FALSE) 
    return simp_op2;
  else if(simp_op2->getType() == T_FALSE)
    return simp_op1;
  else if(simp_op1->getType() == T_TRUE ||
	  simp_op2->getType() == T_TRUE)
    return make_shared<True>();
  else
    return make_shared<Or>(simp_op1, simp_op2);
}

Formula Imp::simplify()
{
  /* Simplifikacija implikacije po pravilima: A ==> True === True,
     A ==> False === ~A, True ==> A === A, False ==> A === True */
  Formula simp_op1 = _op1->simplify();
  Formula simp_op2 = _op2->simplify();
  
  if(simp_op1->getType() == T_TRUE)
    return simp_op2;
  else if(simp_op2->getType() == T_TRUE)
    return make_shared<True>();
  else if(simp_op1->getType() == T_FALSE) 
    return make_shared<True>();
  else if(simp_op2->getType() == T_FALSE)
    return make_shared<Not>(simp_op1);
  else
    return make_shared<Imp>(simp_op1, simp_op2);
}

Formula Iff::simplify()
{
  /* Ekvivalencija se simplifikuje pomocu pravila:
     True <=> A === A, False <=> A === ~A i sl. */
  
  Formula simp_op1 = _op1->simplify();
  Formula simp_op2 = _op2->simplify();
  
  if(simp_op1->getType() == T_FALSE && 
     simp_op2->getType() == T_FALSE)
    return make_shared<True>();
  else if(simp_op1->getType() == T_TRUE)
    return simp_op2;
  else if(simp_op2->getType() == T_TRUE)
    return simp_op1;
  else if(simp_op1->getType() == T_FALSE) 
    return make_shared<Not>(simp_op2);
  else if(simp_op2->getType() == T_FALSE)
    return make_shared<Not>(simp_op1);
  else
    return make_shared<Iff>(simp_op1, simp_op2);
}
// ---------------------------------------------------------------------


// NNF FUNKCIJE --------------------------------------------------------

Formula AtomicFormula::nnf()
{
  return shared_from_this();
}

Formula Not::nnf()
{
  /* Eliminacija dvojne negacije */
  if(_op->getType() == T_NOT)
    {
      Not * not_op = (Not *) _op.get();
      return not_op->getOperand()->nnf();
    }
  /* De-Morganov zakon ~(A/\B) === ~A \/ ~B, pa zatim rekurzivna 
     primena nnf-a na ~A i ~B */
  else if(_op->getType() == T_AND)
    {
      And * and_op =  (And *) _op.get();
	
      return make_shared<Or>(make_shared<Not>(and_op->getOperand1())->nnf(),
			     make_shared<Not>(and_op->getOperand2())->nnf());
      
    }
  /* De-Morganov zakon ~(A\/B) === ~A /\ ~B, pa zatim rekurzivna 
     primena nnf-a na ~A i ~B */
  else if(_op->getType() == T_OR)
    {
      Or * or_op =  (Or *) _op.get();
      
      return make_shared<And>(make_shared<Not>(or_op->getOperand1())->nnf(),
			      make_shared<Not>(or_op->getOperand2())->nnf());
      
    }
  /* De-Morganov zakon ~(A==>B) === A /\ ~B, pa zatim rekurzivna
     primena nnf-a na A i ~B */
  else if(_op->getType() == T_IMP)
    {
      Imp * imp_op =  (Imp *) _op.get();
	
      return make_shared<And>(imp_op->getOperand1()->nnf(),
			      make_shared<Not>(imp_op->getOperand2())->nnf());
      
    }
  /* Primena pravila ~(A<=>B) === (A /\ ~B) \/ (B /\ ~A) */
  else if(_op->getType() == T_IFF)
    {
      Iff * iff_op =  (Iff *) _op.get();
      
      return make_shared<Or>(make_shared<And>(iff_op->getOperand1()->nnf(),
					      make_shared<Not>(iff_op->getOperand2())->nnf()),
			     make_shared<And>(iff_op->getOperand2()->nnf(),
					      make_shared<Not>(iff_op->getOperand1())->nnf()));      
    }
  else
    {
      return shared_from_this();
    } 
}

Formula And::nnf()
{
  return make_shared<And>(_op1->nnf(), _op2->nnf());
}

Formula Or::nnf()
{
  return make_shared<Or>(_op1->nnf(), _op2->nnf());
}

Formula Imp::nnf()
{
  /* Eliminacija implikacije, pa zatim rekurzivna primena nnf()-a */
  return make_shared<Or>(make_shared<Not>(_op1)->nnf(), _op2->nnf());
}

Formula Iff::nnf()
{
  /* Eliminacija ekvivalencije, pa zatim rekurzivna primena nnf()-a.
     Primetimo da se ovde velicina formule duplira */
  return make_shared<And>(make_shared<Or>(make_shared<Not>(_op1)->nnf(), _op2->nnf()),
			  make_shared<Or>(make_shared<Not>(_op2)->nnf(), _op1->nnf()));
}

// ---------------------------------------------------------------------


// POMOCNE FUNKCIJE ZA BARATANJE LISTAMA --------------------------------

/* Funkcija nadovezuje dve liste */
template <typename T>
T concatLists(const T & c1, const T & c2)
{
  T c = c1;
  
  for(auto it = c2.begin(), it_end = c2.end(); it != it_end; ++it)
    c.push_back(*it);

  //Krace:
  //c.resize(c1.size() + c2.size());
  //copy(c2.begin(), c2.end(), c.begin() + c1.size());

  // Ili, jos krace:
  //copy(c2.begin(), c2.end(), back_inserter(c));
  
  return c;
}


/* Funkcija nadovezuje svaku listu literala iz c1 sa svakom listom literala
   iz c2 i sve takve liste literala smesta u rezultujucu listu listi c */
LiteralListList makePairs(const LiteralListList & c1, 
			  const LiteralListList & c2)
	      
{
  LiteralListList c;

  for(auto & l1 : c1)
    for(auto & l2 : c2)
      c.push_back(concatLists(l1, l2));    
  return c;
}

/* Funkcija prikazuje listu listi */
ostream & operator << (ostream & ostr, const LiteralListList & l)
{
  ostr << "[ ";
  for(auto & ll :  l)
    {
      ostr << "[ ";
      for(auto & f : ll)
	{
	  ostr << f << " ";
	}
      ostr << "] ";
    }
  ostr << " ]";

  return ostr;
}


// ----------------------------------------------------------------------


// FUNKCIJE ZA ODREDJIVANJE CNF U OBLIKU LISTE LISTA LITERALA ----------

LiteralListList True::listCNF()
{
  /* Formuli True odgovara prazna lista klauza. Naime, po konvenciji,
     prazna lista klauza je zadovoljena u svakoj valuaciji (zato sto
     u skupu ne postoji klauza koja nije zadovoljena), pa je otuda
     logicki ekvivalentna formuli True. */
  return { };
}

LiteralListList False::listCNF()
{
  /* Formuli False odgovara lista klauza koja sadrzi samo jednu praznu
     klauzu. Po konvenciji, prazna klauza je netacna u svakoj valuaciji,
     (zato sto ne postoji literal koji je zadovoljen), pa je otuda 
     logicki ekvivalentna sa False */
  return {{}};
}

LiteralListList Atom::listCNF()
{
  /* Pozitivan literal (atom) se predstavlja listom klauza koja sadrzi
     samo jednu klauzu koja se sastoji iz tog jednog literala */
  return { { shared_from_this() } };
}

LiteralListList Not::listCNF()
{
  /* Negativan literal se predstavlja listom klauza koja sadrzi
     samo jednu klauzu koja se sastoji iz tog jednog literala */
  return { { shared_from_this() } };
}

LiteralListList And::listCNF()
{
  /* CNF lista se kod konjukcije dobija nadovezivanjem CNF listi
     podformula */
  LiteralListList cl1 = _op1->listCNF();
  LiteralListList cl2 = _op2->listCNF();

  
  return concatLists(cl1, cl2);
}

LiteralListList Or::listCNF()
{
  /* CNF lista disjunkcije se dobija tako sto se CNF liste podformula
     distributivno "pomnoze", tj. liste literala se nadovezu svaka sa
     svakom */
  
  LiteralListList cl1 = _op1->listCNF();
  LiteralListList cl2 = _op2->listCNF();
  
  return makePairs(cl1, cl2);
}

LiteralListList Imp::listCNF()
{
  throw "CNF not aplicable";
}

LiteralListList Iff::listCNF()
{
  throw "CNF not aplicable";
}

// -------------------------------------------------------------------

// FUNKCIJE ZA ODREDJIVANJE DNF U OBLIKU LISTE LISTA LITERALA ----------

LiteralListList True::listDNF()
{
  /* Formuli True odgovara lista koja sadrzi samo jednu praznu listu
     literala. Po konvenciji, prazna konjunkcija je tacna u svakoj valuaciji,
     (zato sto ne postoji literal koji je nezadovoljen), pa je otuda 
     logicki ekvivalentna sa True */
  return {{}};
}


LiteralListList False::listDNF()
{
  /* Formuli False odgovara prazna lista listi. Naime, po konvenciji,
     prazna lista listi je nezadovoljena u svakoj valuaciji (zato sto
     u skupu ne postoji lista literala koja je zadovoljena), pa je otuda
     logicki ekvivalentna formuli False. */
  return {};
}

LiteralListList Atom::listDNF()
{
  /* Pozitivan literal (atom) se predstavlja listom koja sadrzi
     samo jednu listu koja se sastoji iz tog jednog literala */
  return {{ shared_from_this() }};
}

LiteralListList Not::listDNF()
{
  /* Negativan literal se predstavlja listom koja sadrzi
     samo jednu listu koja se sastoji iz tog jednog literala */
  return {{ shared_from_this() }};
}

LiteralListList And::listDNF()
{
  /* DNF lista konjunkcije se dobija tako sto se DNF liste podformula
     distributivno "pomnoze", tj. liste literala se nadovezu svaka sa
     svakom */
  
  LiteralListList cl1 = _op1->listDNF();
  LiteralListList cl2 = _op2->listDNF();
  
  return makePairs(cl1, cl2);
}

LiteralListList Or::listDNF()
{
  /* DNF lista se kod disjunkcije dobija nadovezivanjem DNF listi
     podformula */
  LiteralListList cl1 = _op1->listDNF();
  LiteralListList cl2 = _op2->listDNF();
  
  return concatLists(cl1, cl2);
}


LiteralListList Imp::listDNF()
{
  throw "DNF not aplicable";
}

LiteralListList Iff::listDNF()
{
  throw "DNF not aplicable";
}

// -------------------------------------------------------------------


// OSTALE FUNKCIJE (konstruktori, get-eri...)

Atom::Atom(unsigned num)
  :_var_num(num)
{}

unsigned Atom::getVariableNumber() const 
{
  return _var_num;
}

UnaryConnective::UnaryConnective(const Formula & op)
  :_op(op)
{}

const Formula & UnaryConnective::getOperand() const
{
  return _op;
}

BinaryConnective::BinaryConnective(const Formula & op1, const Formula & op2)
  :_op1(op1),
   _op2(op2)
{}
  
const Formula & BinaryConnective::getOperand1() const
{
  return _op1;
}

const Formula & BinaryConnective::getOperand2() const
{
  return _op2;
}

// DEFINICIJE FUNKCIJA CLANICA KLASE Valuation ---------------------------

void Valuation::setInitial(const AtomSet & atoms)
{
  _map.clear();
  for(const auto & x : atoms)
    _map.insert(make_pair(x, false));
}

bool Valuation::nextValuation()
{
  map<unsigned, bool>::reverse_iterator it = _map.rbegin(),
    it_end = _map.rend();
  
  for(; it != it_end; ++it)
    {
      it->second = !it->second;
      if(it->second == true)
	return true;
    }
  return false;
}

bool Valuation::getValue(unsigned p) const
{
  map<unsigned, bool>::const_iterator it = _map.find(p);
  if(it == _map.end())
    throw "Unknown variable";
  else
    return it->second;
}

void Valuation::setValue(unsigned p, bool b)
{
  _map[p] = b;
}

void Valuation::printValuation(ostream & ostr) const 
{
  for(const auto & x : _map)
    ostr << x.second << " ";
}
// -----------------------------------------------------------------------


// OPERATORI ZA ISPIS ----------------------------------------------------

ostream & operator << (ostream & ostr, const Valuation & v)
{
  v.printValuation(ostr);
  return ostr;
}

ostream & operator << (ostream & ostr, const Formula & f)
{
  f->printFormula(ostr);
  return ostr;
}

// -----------------------------------------------------------------------

int main()
{
  Formula p0 = make_shared<Atom>(0);
  Formula p1 = make_shared<Atom>(1);
  Formula p2 = make_shared<Atom>(2);

  /*  Formula f1 = Formula(make_shared<Not>(p0);
  Formula f2 = Formula(make_shared<And>(f1, p1);
  Formula f3 = make_shared<Imp>(f2, f1);
  Formula f4 = make_shared<Or>(f3, p2);

  cout << f4 << endl;
  cout << f4->complexity() << endl;

  Valuation v;

  v[0] = true;
  v[1] = false;
  v[2] = true;

  cout << f4->eval(v) << endl;

  cout << f4->isTautology() << endl;

  cout << isEquivalent(f4, p1) << endl;
  */

  Formula f1 = make_shared<And>(p0, p1);
  Formula f2 = make_shared<Not>(f1);
  Formula f3 = make_shared<Imp>(f2, p2);
  Formula f4 = make_shared<Not>(p0);
  Formula f5 = make_shared<Not>(p1);
  Formula f6 = make_shared<Or>(f4, f5);
 
  Formula f7 = f3->substitute(f2, f6);

  cout << f3 << endl;
  cout << f7 << endl;
  cout << isEquivalent(f3, f7) << endl;

  Formula f8 = make_shared<And>(p0, make_shared<True>());
  Formula f9 = make_shared<Or>(f8, make_shared<True>());
  Formula f10 = make_shared<Imp>(f9, p1);
  Formula f11 = f10->simplify();

  cout << f10 << endl;
  cout << f11 << endl;
  cout << isEquivalent(f10, f11) << endl;

  Formula f12 = make_shared<Iff>(make_shared<False>(), make_shared<False>());

  cout << f12->simplify() << endl;

  Formula f13 = make_shared<Not>(f3);
  Formula f14= f13->simplify()->nnf();

  cout << f13 << endl;
  cout << f14 << endl;
  cout << isEquivalent(f13, f14) << endl;

  LiteralListList list = f14->listDNF();

  cout << list << endl;
  
  return 0;
}
