#include <iostream>
#include <memory>
#include <set>
#include <map>

#include <algorithm>
#include <iterator>

using namespace std;

class BaseFormula;

typedef shared_ptr<BaseFormula> Formula;

typedef set<unsigned> AtomSet;

class Valuation {
private:
  map<unsigned, bool> _map;
public:
  bool getValue(unsigned n) const
  {
    map<unsigned, bool>::const_iterator it = _map.find(n);
    if(it != _map.end())
      return it->second;
    else
      throw "Unknown variable";
  }

  void setValue(unsigned n, bool value)
  {
    _map[n] = value;
  }

  void initValuation(const AtomSet & s)
  {
    _map.clear();
    for(const auto & x : s)
      {
	_map.insert(make_pair(x, false));
      }
  }

  bool nextValuation()
  {
    for(map<unsigned, bool>::reverse_iterator it = _map.rbegin(),
	  it_end = _map.rend(); it != it_end; ++it)
      {
	it->second = !it->second;
	if(it->second)
	  return true;
      }
    return false;
  }

  void printValuation(ostream & ostr) const
  {
    for(const auto & p : _map)
      ostr << p.second << " ";
  }

};


ostream & operator << (ostream & ostr, const Valuation & v)
{
  v.printValuation(ostr);
  return ostr;
}


class BaseFormula : public enable_shared_from_this<BaseFormula> {

public:
  enum Type { T_TRUE, T_FALSE, T_ATOM, T_NOT, T_AND, T_OR, T_IMP, T_IFF };
  
  virtual Type getType() const = 0;
  virtual void printFormula(ostream & ostr) const = 0;
  virtual unsigned complexity() const = 0;
  virtual bool equalTo(const Formula & f) const = 0;
  virtual void getAtoms(AtomSet & s) const = 0;
  virtual Formula substitute(const Formula & a,
			     const Formula & b) = 0;
  virtual bool eval(const Valuation & v) const = 0;

  bool isSatisfiable(Valuation & v) const
  {
    AtomSet s;
    getAtoms(s);
    v.initValuation(s);

    do {
      if(eval(v))
	return true;
    } while(v.nextValuation());

    return false;
  }

  bool isTautology(Valuation & v) const
  {
    AtomSet s;
    getAtoms(s);
    v.initValuation(s);

    do {
      if(!eval(v))
	return false;
    } while(v.nextValuation());

    return true;
  }

  void printTruthTable(ostream & ostr)
  {
    Valuation v;
    AtomSet s;
    getAtoms(s);
    v.initValuation(s);
    
    do {
      ostr << v << " | " << eval(v) << endl;
    } while(v.nextValuation());
  }

  virtual ~BaseFormula() {}
};

bool isConsequence(const Formula &  a, const Formula &  b)
{
  Valuation v;
  AtomSet atoms;
  
  a->getAtoms(atoms);
  b->getAtoms(atoms);
  
  v.initValuation(atoms);
  
  do {
    if(a->eval(v) && !b->eval(v))
      return false;
    
  } while(v.nextValuation());
  
  return true;
} 

bool isEquivalent(const Formula &  a, const Formula &  b)
{
  Valuation v;
  AtomSet atoms;
  
  a->getAtoms(atoms);
  b->getAtoms(atoms);
  
  v.initValuation(atoms);
  
  do {
    if(a->eval(v) != b->eval(v))
      return false;
    
  } while(v.nextValuation());
  
  return true;
} 


ostream & operator << (ostream & ostr, const Formula & f)
{
  f->printFormula(ostr);
  return ostr;
}


class AtomicFormula : public BaseFormula {
public:
  virtual unsigned complexity() const
  {
    return 0;
  }

  virtual Formula substitute(const Formula & a,
			     const Formula & b)
  {
    if(equalTo(a))
      return b;
    else
      return shared_from_this();
  }
};


class LogicConstant : public AtomicFormula {
public:
  virtual bool equalTo(const Formula & f) const
  {
    return getType() == f->getType();
  }

  virtual void getAtoms(AtomSet & s) const
  {
    // No atoms
  }
};

class True  : public LogicConstant {
public:  
  virtual Type getType() const
  {
    return T_TRUE;
  }

  virtual void printFormula(ostream & ostr) const 
  {
    ostr << "TRUE";
  }

  virtual bool eval(const Valuation & v) const
  {
    return true;
  }
};

class False  : public LogicConstant {
public:
  virtual Type getType() const
  {
    return T_FALSE;
  }

  virtual void printFormula(ostream & ostr) const 
  {
    ostr << "FALSE";
  }

  virtual bool eval(const Valuation & v) const
  {
    return false;
  }
};

class Atom : public AtomicFormula {
private:
  unsigned _num;
public:
  Atom(unsigned n)
    :_num(n)
  {}

  unsigned getVariableNumber() const
  {
    return _num;
  }

  virtual Type getType() const
  {
    return T_ATOM;
  }

  virtual void printFormula(ostream & ostr) const 
  {
    ostr << "p" << _num;
  }

  virtual bool equalTo(const Formula & f) const
  {
    return f->getType() == T_ATOM &&
      ((Atom*)f.get())->getVariableNumber() == _num;
  }

  virtual void getAtoms(AtomSet & s) const
  {
    s.insert(_num);
  }

  virtual bool eval(const Valuation & v) const
  {
    return v.getValue(_num);
  }
};

class UnaryConnective : public BaseFormula {
protected:
  Formula _op;
public:
  UnaryConnective(const Formula & op)
    :_op(op)
  {}
  
  const Formula & getOperand() const
  {
    return _op;
  }

  virtual unsigned complexity() const
  {
    return _op->complexity() + 1;
  }

  virtual bool equalTo(const Formula & f) const
  {
    return f->getType() == getType() &&
      ((UnaryConnective*)f.get())->getOperand()->equalTo(_op);
  }

  virtual void getAtoms(AtomSet & s) const
  {
    _op->getAtoms(s);
  }

};

class Not : public UnaryConnective {
public:
  using UnaryConnective::UnaryConnective;
  
  virtual Type getType() const
  {
    return T_NOT;
  }

  virtual void printFormula(ostream & ostr) const 
  {
    ostr << "(~" << _op << ")";
  }

  
  virtual Formula substitute(const Formula & a,
			     const Formula & b)
  {
    if(equalTo(a))
      return b;
    else
      return make_shared<Not>(_op->substitute(a, b));
  }

  virtual bool eval(const Valuation & v) const
  {
    return !_op->eval(v);
  }
};

class BinaryConnective : public BaseFormula {
protected:
  Formula _op1;
  Formula _op2;
public:
  BinaryConnective(const Formula & op1, 
		   const Formula & op2)
    :_op1(op1),
     _op2(op2)
  {}
  
  const Formula & getOperand1() const
  {
    return _op1;
  }

  const Formula & getOperand2() const
  {
    return _op2;
  }

  virtual unsigned complexity() const
  {
    return _op1->complexity() + _op2->complexity() + 1;
  }

  virtual bool equalTo(const Formula & f) const
  {
    return f->getType() == getType() &&
      ((BinaryConnective*)f.get())->getOperand1()->equalTo(_op1) &&
      ((BinaryConnective*)f.get())->getOperand2()->equalTo(_op2);
  }

  virtual void getAtoms(AtomSet & s) const
  {
    _op1->getAtoms(s);
    _op2->getAtoms(s);
  }

};

class And : public BinaryConnective {
public:
  using BinaryConnective::BinaryConnective;
  
  virtual Type getType() const
  {
    return T_AND;
  }

  virtual void printFormula(ostream & ostr) const 
  {
    ostr << "(" << _op1 <<  " /\\ " << _op2 << ")";
  }

  virtual Formula substitute(const Formula & a,
			     const Formula & b)
  {
    if(equalTo(a))
      return b;
    else
      return make_shared<And>(_op1->substitute(a, b),
			      _op2->substitute(a, b));
  }

  virtual bool eval(const Valuation & v) const
  {
    return _op1->eval(v) && _op2->eval(v);
  }
};

class Or : public BinaryConnective {
public:
  using BinaryConnective::BinaryConnective;


  virtual Type getType() const
  {
    return T_OR;
  }

  virtual void printFormula(ostream & ostr) const 
  {
    ostr << "(" << _op1 << " \\/ " << _op2 << ")";
  }

  virtual Formula substitute(const Formula & a,
			     const Formula & b)
  {
    if(equalTo(a))
      return b;
    else
      return make_shared<Or>(_op1->substitute(a, b),
			     _op2->substitute(a, b));
  }

  virtual bool eval(const Valuation & v) const
  {
    return _op1->eval(v) || _op2->eval(v);
  }

};

class Imp : public BinaryConnective {
public:
  using BinaryConnective::BinaryConnective;

  virtual Type getType() const
  {
    return T_IMP;
  }

  virtual void printFormula(ostream & ostr) const 
  {
    ostr << "(" << _op1 <<  " => " << _op2 << ")";
  }

  virtual Formula substitute(const Formula & a,
			     const Formula & b)
  {
    if(equalTo(a))
      return b;
    else
      return make_shared<Imp>(_op1->substitute(a, b),
			      _op2->substitute(a, b));
  }

  virtual bool eval(const Valuation & v) const
  {
    return !_op1->eval(v) || _op2->eval(v);
  }
};

class Iff : public BinaryConnective {
public:
  using BinaryConnective::BinaryConnective;

  virtual Type getType() const
  {
    return T_IFF;
  }

  virtual void printFormula(ostream & ostr) const 
  {
    ostr << "(" << _op1 << " <=> " << _op2 << ")";
  }

  virtual Formula substitute(const Formula & a,
			     const Formula & b)
  {
    if(equalTo(a))
      return b;
    else
      return make_shared<Iff>(_op1->substitute(a, b),
			      _op2->substitute(a, b));
  }

  virtual bool eval(const Valuation & v) const
  {
    return _op1->eval(v) == _op2->eval(v);
  }
};


int main()
{
  Formula p1 = make_shared<Atom>(1);
  Formula p2 = make_shared<Atom>(2);
  Formula p3 = make_shared<Atom>(3);
  Formula p4 = make_shared<Atom>(4);
  
  Formula f1 = make_shared<And>(p1, p2);
  Formula f2 = make_shared<Imp>(p3, f1);
  Formula f3 = make_shared<Not>(f2);
  Formula f4 = make_shared<Iff>(f3, f1);
  Formula f5 = make_shared<Or>(f4, p4);
  

  cout << f5 << endl;
  cout << f5->complexity() << endl;
  cout << f5->equalTo(f4) << endl;
  cout << f5->equalTo(f5) << endl;

  AtomSet s;
  f5->getAtoms(s);
  for(const auto & x : s)
    cout << "p" << x << ", ";
  cout << endl;
  
  cout << f5->substitute(f1, f2) << endl;

  Valuation v;
  v.initValuation(s);
  v.setValue(1, true);
  v.setValue(3, true);

  cout << f5->eval(v) << endl;
  
  if(f5->isSatisfiable(v))
    {
      cout << "SAT: " << v << endl;
    }
  else 
    cout << "UNSAT" << endl;
  
  if(!f5->isTautology(v))
    {
      cout << "NOT TAUTOLOGY: " << v << endl;
    }
  else
    cout << "TAUTOLOGY"  << endl;


  f5->printTruthTable(cout);

  cout << f5 << endl;
  cout << isEquivalent(f5, f5) << endl;

  Formula f6 = make_shared<And>(p2, p1);
  Formula f7 = f5->substitute(f1, f6);
  cout << f7 << endl;
  cout << isEquivalent(f5, f7) << endl;
  
  return 0;
}
