`include "common.v"

/* Stanja konacnog automata memorije */
`define MEM_IDLE 2'b00  // memorija ceka komandu mastera
`define MEM_DONE 2'b01  // memorija je uspesno izvrsila komandu
`define MEM_READ_IN_PROGRESS 2'b10 // proces citanja u toku
`define MEM_WRITE_IN_PROGRESS 2'b11 // proces upisa u toku

// Model jednostavne memorije. Memorija ceka zahtev od mastera (procesora)
// preko signala bus_master_request i bus_command, nakon cega na pozitivnoj
// ivici casovnika odgovara na zahtev. Ukoliko memorija ne moze da
// odgovori na zahtev na prvoj sledecoj pozitivnoj ivici casovnika,
// ukljucuje bus_wait signal i drzi ga ukljucenog dokle god ne bude
// u stanju da izvrsi komandu u prvoj polovini sledeceg ciklusa. U
// ovom modelu se to kasnjenje vestacki simulira, tako sto se povremeno,
// pomocu funkcije $random prelazi u stanje u kome se ceka odredjeno
// vreme pre nego sto se predje u zavrsno stanje.
module memory(clk, data, addr, bus_master_request, bus_command, bus_wait);
   input clk, bus_master_request, bus_command;
   output reg bus_wait;
   input [31:0] addr;
   inout [31:0] data;

   // reg [31:0] 	_regs[0 : 30'h3fffffff]; 
   // // NE MOZE 4GB memorije da rezervise

   reg [31:0] 	_regs[0 : 30'hffff]; // memorija je niz 32-bitnih lokacija
   reg [1:0] 	state;               // stanje (2 bita)
   reg [31:0] 	read_data;  // registar koji se povezuje na data magistralu
   

   
   integer 	randnum; // Pomocna promenljiva za simulaciju kasnjenja
   
   
   // Bafer sa tri stanja koji pusta podatak iz registra read_data na
   // magistralu podataka. Podatak se pusta samo ako je u pitanju
   // zahtev za citanjem, a mi smo u stanju DONE, tj. podatak je
   // procitan, smesten u read_data i spreman za dostavu masteru.
   assign data = bus_master_request && 
		 state == `MEM_DONE && 
		 bus_command == `BUS_READ ? read_data : 32'hzzzzzzzz; 
      
   initial
     begin
	state <= `MEM_IDLE;
	bus_wait <= `FALSE;	
     end

   always @(posedge clk)
     case(state)
       `MEM_IDLE:
	 begin		    
	    if(bus_master_request)
	      begin
		 // Ako je u pitanju operacija citanja...
		 if(bus_command == `BUS_READ)
		   begin
		      // u 25% slucajeva simuliramo kasnjenje memorije
		      // 10 vremenskih jedinica. U tom slucaju se
		      // prelazi u stanje READ_IN_PROGRESS. U suprotnom,
		      // odmah dostavljamo trazeni podatak i prelazimo
		      // u stanje DONE.
		      randnum = $random % 4;
		      if(randnum == 1 || randnum == -1)
			begin
			   bus_wait <= `TRUE;
			   state <= `MEM_READ_IN_PROGRESS;		      
			end
		      else
			begin
			   read_data <= _regs[addr >> 2];
			   $display($time, ": read: mem[%h]", addr);
			   state <= `MEM_DONE;
			end
		      
		   end
		 // U slucaju operacije upisa u memoriju...		
		 else if(bus_command == `BUS_WRITE)
		   begin
		      // Slicno kao kod citanja, u 25% slucajeva simuliramo
		      // kasnjenje, tako sto prelazimo u stanje
		      // WRITE_IN_PROGRESS. U suprotnom, odmah vrsimo upis
		      // i odlazimo u stanje DONE.
		      randnum = $random % 4;
		      if(randnum == 1 || randnum == -1)
			begin
			   bus_wait <= `TRUE;
			   state <= `MEM_WRITE_IN_PROGRESS;		      
			end
		      else
			begin
			   _regs[addr >> 2] <= data;
			   $display($time, ": write: mem[%h]=%h (%d)", addr, data, data);	
			   state <= `MEM_DONE;	       	       		 	    
			end
		   end
	      end
	 end 
       `MEM_READ_IN_PROGRESS:
	 begin
	    // Cekamo pozitivnu ivicu casovnika nakon 10 vremenskih jedinica
	    #10 @(posedge clk) ; 
	    read_data <= _regs[addr >> 2];  // citanje
	    bus_wait <= `FALSE;	  // iskljucivanje bus_wait signala
	    $display($time, ": read: mem[%h]", addr);
	    state <= `MEM_DONE;	    
	 end
       `MEM_WRITE_IN_PROGRESS:
	 begin
	    // Cekamo pozitivnu ivicu casovnika nakon 10 vremenskih jedinica
	    #100 @(posedge clk) ;
	    bus_wait <= `FALSE; // iskljucujemo bus_wait signal
	    _regs[addr >> 2] <= data; // vrsimo upis
	    $display($time, ": write: mem[%h]=%h (%d)", addr, data, data);	
	    state <= `MEM_DONE;	    
	 end
       `MEM_DONE:
	 begin	
	    // Nakon zavrsetka transakcije vracamo se u IDLE stanje
	    if(!bus_master_request)
	      begin
		 state <= `MEM_IDLE;
	      end
	 end
     endcase	
   
   always @(state)
     case(state)
       `MEM_IDLE:
	 $display($time, ": mem_state: MEM_IDLE");
       `MEM_DONE:
	 $display($time, ": mem_state: MEM_DONE");
       `MEM_READ_IN_PROGRESS:
	 $display($time, ": mem_state: MEM_READ_IN_PROGRESS");
       `MEM_WRITE_IN_PROGRESS:
	 $display($time, ": mem_state: MEM_WRITE_IN_PROGRESS");

     endcase
   
   
endmodule // memory
