#include <iostream>
#include <vector>
#include <vector>
#include <cstdlib>

using namespace std;

typedef unsigned Variable;
typedef unsigned Literal;


// DIMACS:  1      2     3     4     5      6    
// VARS:    0      1     2     3     4      5      6
// LITS:  0  1   2  3   4 5   6 7   8 9    10 11  12 13

enum Polarity { P_POSITIVE, P_NEGATIVE };

Literal litFromVar(Variable v, Polarity p)
{
  return p == P_POSITIVE ? v << 1 : (v << 1) | 1;
}

Variable varFromLit(Literal l)
{
  return l >> 1;
}

bool isPositive(Literal l)
{
  return !(l & 1);
}

bool isNegative(Literal l)
{
  return l & 1;
}

Literal oppositeLiteral(Literal l)
{
  return l ^ 1;
}

int intFromLit(Literal l)
{
  return isPositive(l) ? (int)varFromLit(l) + 1 : -(int)(varFromLit(l) + 1);
}

Literal litFromInt(int i)
{
  return i > 0 ? litFromVar(i - 1, P_POSITIVE) : litFromVar(-i - 1, P_NEGATIVE);
}

typedef vector<Literal> Clause;
typedef vector<Clause> Formula;

void printClause(const Clause & c, ostream & ostr)
{
  for(Literal lit : c)
    ostr << intFromLit(lit) << " ";
  ostr << endl;
}

class Valuation {
private:
  vector<bool> _values;
public:
  Valuation(unsigned num_of_vars)
    :_values(num_of_vars, false)
  {}
  
  bool nextValuation()
  {
    for(vector<bool>::reverse_iterator i = _values.rbegin(); 
	i != _values.rend();
	i++)
      {
	*i = !(*i);
	if(*i)
	  return true;
      }
    return false;
  }

  bool variableValue(Variable v) const
  {
    return _values[v];
  }

  bool literalValue(Literal l) const
  {
    return isPositive(l) ? _values[varFromLit(l)] : !_values[varFromLit(l)];
  }

  bool isClauseFalse(const Clause & c) const
  {
    for(Clause::const_iterator i = c.begin(); i != c.end(); i++)
      {
	if(literalValue(*i) == true)
	  return false;
      }
    return true;
  }

  void print(ostream & ostr) const
  {
    for(unsigned i = 0; i < _values.size(); i++)
      {
	if(_values[i])
	  ostr << (int)(i + 1) << " ";
	else
	  ostr << - (int)(i + 1) << " ";
      }
    ostr << 0 << endl;
  }

};

int skipSpaces()
{
  int c;
  while((c = cin.get()) == ' ' || c == '\t' || c == '\n');
  return c;
}

void skipRestOfLine()
{
  while(cin.get() != '\n');
}

bool readDIMACS(Formula & f, unsigned & num_of_vars)
{
  unsigned num_of_clauses;
  int c;

  // Skip comment lines
  while((c = skipSpaces()) == 'c')
    skipRestOfLine();

  // read p line
  if(c != 'p')
    return false;
  else
    {
      string s;
      cin >> s;
      if(s != "cnf")
	return false;
      
      cin >> num_of_vars;
      cin >> num_of_clauses;
    }

  // Read clauses
  for(unsigned i = 0; i < num_of_clauses; i++)
    {
      Clause c;
      int n;
      cin >> n; 
      while(!cin.eof() && !cin.fail() && n != 0)
	{
	  c.push_back(litFromInt(n));
	  cin >> n;
	}
      
      if(cin.eof() || cin.fail())
	return false;

      f.push_back(c);
    }
  return true;
}

class Solver {
private:
  Formula _formula;
  Valuation _val;

public:
  Solver(const Formula & f, unsigned num_of_vars)
    :_formula(f),
     _val(num_of_vars)
  {}
  
 
  bool checkConflict()
  {
    for(unsigned i = 0; i < _formula.size(); i++)
      if(_val.isClauseFalse(_formula[i]))
	{
	  return true;
	}
    return false;
  }
  
 

  bool solve()
  {
    do {
      if(!checkConflict())
	return true;
    } while(_val.nextValuation());
  
    return false;
  }

  const Valuation & getValuation() const
  {
    return _val;
  }
 
};

int main()
{
  unsigned num_of_vars;
  Formula f;
  
  if(!readDIMACS(f, num_of_vars))
    {
      cerr << "Error reading input file" << endl;
      exit(1);
    }

  Solver solver(f, num_of_vars);

  if(!solver.solve())
    {
      cout << "UNSAT" << endl;
    }
  else
    {
      cout << "SAT" << endl;
      solver.getValuation().print(cout);
    }

  return 0;
}
